// name   : CAIAndWinPilotFileHandler.cs
// author : Harald Maier
// date   : 17.10.2003
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.IO;
using System.Collections;
using System.Windows.Forms;

using SoaringDotNET.Data;
using SoaringDotNET.GUI;

namespace SoaringDotNET.FileFormats
{
	/// <summary>
	/// 
	/// </summary>
	public class CAIAndWinPilotFileHandler : SoaringDotNET.FileFormats.FileHandlerBase
	{
		public CAIAndWinPilotFileHandler()
		{
			// 
			// TODO: Add constructor logic here
			//
		}
	
        public override void Load(StreamReader f, WaypointCatalog catalogue) {
            // TODO:  Add CAIAndWinPilotFileHandler.Load implementation
            string line;
            WayPoint wp;

            try {
                while((line = f.ReadLine()) != null) {
                    if (line[0] != '*') {
                        if ((wp = ParseLine(line)) != null) {
                            catalogue.Add(wp);
                        }
                    }
                }
            }
            catch (Exception e) {
                if (e.Message != "Abort") {
                    throw e;
                }
            }
        }
    
        public override void Load(StreamReader f, WaypointCatalog catalogue, ArrayList tasks) {
            // TODO:  Add CAIAndWinPilotFileHandler.Load implementation
            this.Load(f, catalogue);
        }
    
        public override void Save(StreamWriter f, WaypointCatalog catalogue) {
            // TODO:  Add CAIAndWinPilotFileHandler.Save implementation
            foreach (WayPoint w in catalogue) {
                f.WriteLine(BuildLine(w));
            }
        }
    
        public override void Save(StreamWriter f, WaypointCatalog catalogue, ArrayList tasks) {
            // TODO:  Add CAIAndWinPilotFileHandler.Save implementation
            this.Save(f, catalogue);
        }
    
        public override string BuildLine(WayPoint wp) {
            int i = 1, tmp;
            string type = "";
            string lat, lon;

            switch (wp.type) {
            case WayPointTypeId.Airport:
            case WayPointTypeId.Glidersite:
                type = "A";
                goto case WayPointTypeId.Outlanding;
            case WayPointTypeId.Outlanding:
                if (wp.landable) {
                    type += "L";
                }
                break;
            case WayPointTypeId.Landmark:
                type = "M";
                break;
            default:
                type = "T";
                break;
            }

            tmp = Math.Abs(wp.Latitude);
            lat = string.Format("{0:00}:{1:00}:{2:00}{3}", tmp / 360000, (tmp % 360000) / 6000, (tmp % 6000) / 100, wp.Latitude <= 0 ? "N" : "S");
            tmp = Math.Abs(wp.Longitude);
            lon = string.Format("{0:000}:{1:00}:{2:00}{3}", tmp / 360000, (tmp % 360000) / 6000, (tmp % 6000) / 100, wp.Longitude < 0 ? "W" : "E");

            // 1,48:23.083N,010:02.100E,1549F,AL,SCHWAI,Description,AreaInfo
            return string.Format("{0},{1},{2},{3},{4},{5},{6}", i++, lat, lon, wp.Elevation != -1 ? wp.Elevation.ToString() + "M" : "", type, wp.longName, wp.description);
        }
    
        public override WayPoint ParseLine(string line) {
            // TODO:  Add CAIAndWinPilotFileHandler.ParseLine implementation
            WayPoint wp = null;
            string [] tokens;
            string [] coordItems;
            int lat, lon;
            int elev;
            string t;
            char c;
            char [] airportOrLandable = {'A', 'L'};

            // 1,48:23.083N,010:02.100E,1549F,AL,SCHWAI,Description,AreaInfo
            tokens = line.Split(',');
            try {
                if (tokens.Length >= 6) {
                    // lat
                    coordItems = tokens[1].Split(':');
                    switch(coordItems.Length) {
                    case 2:
                        // 48:23.083N format
                        t = coordItems[1];
                        lat = (int.Parse(coordItems[0]) * 360000) + (int)(double.Parse(t.Substring(0, t.Length - 1), AppContents.ni) * 6000.0);
                        break;
                    case 3:
                        // 48:00:42.00N format
                        t = coordItems[2];
                        lat = (int.Parse(coordItems[0]) * 360000) + (int.Parse(coordItems[1]) * 6000) + (int)(double.Parse(t.Substring(0, t.Length - 1), AppContents.ni) * 100);
                        break;
                    default:
                        lat = 0;
                        t = "?";
                        break;
                    }

                    c = t[t.Length - 1];
                    if (c == 'N' || c == 'n') {
                        lat = -lat;
                    }

                    // lon
                    coordItems = tokens[2].Split(':');
                    switch(coordItems.Length) {
                    case 2:
                        // 48:23.083N format
                        t = coordItems[1];
                        lon = (int.Parse(coordItems[0]) * 360000) + (int)(double.Parse(t.Substring(0, t.Length - 1),  AppContents.ni) * 6000.0);
                        break;
                    case 3:
                        // 48:00:42.00N format
                        t = coordItems[2];
                        lon = (int.Parse(coordItems[0]) * 360000) + (int.Parse(coordItems[1]) * 6000) + (int)(double.Parse(t.Substring(0, t.Length - 1), AppContents.ni) * 100);
                        break;
                    default:
                        lon = 0;
                        t = "?";
                        break;
                    }
                        
                    c = t[t.Length - 1];
                    if (c == 'W' || c == 'w') {
                        lon = -lon;
                    }

                    // elevation
                    t = tokens[3];
                    c = t[t.Length - 1];

                    elev = int.Parse(t.Substring(0, t.Length - 1));
                    if (c != 'M' && c != 'm') {
                        if (c == 'F' || c == 'f') {
                            elev = (int)((double)elev * ConvertionFactors.FEET2METER);
                        }
                        else {
                            throw new Exception("Unknown elevation unit " + c + " in line\n" + line);
                        }
                    }

                    wp = new WayPoint(lat, lon, tokens[5].Trim());
                    wp.Elevation = elev;

                    t = tokens[4];

                    wp.landable = t.IndexOfAny(airportOrLandable) != -1;
                    if (wp.landable) {
                        wp.surface = t.IndexOf('A') != -1 ? SurfaceTypeId.Hard : SurfaceTypeId.Grass;
                        // we assume outlanding may be glidersite as well
                        wp.type = t.IndexOf('A') != -1 ? WayPointTypeId.Airport : WayPointTypeId.Outlanding;
                    }
                    else {
                        if (t.IndexOf('M') != -1) {
                            wp.type = WayPointTypeId.Landmark;
                        }
                    }


                    if (tokens.Length >= 7) {
                        wp.description = tokens[6].Trim();
                    }
                }
            }
            catch (Exception e) {
                if (MessageBox.Show("An exception occured while reading file (CAI/WinPilot)\n\nThe fault line is:\n\n" +
                    line + "\n\n" + e.StackTrace + "\n\n" + e.Message, "Error", MessageBoxButtons.OKCancel, MessageBoxIcon.Error) == DialogResult.Cancel) {
                    throw new Exception("Abort");
                }
                wp = null;
            }

            return wp;
        }
    }
}
