using System;
using System.Text;
using System.Text.RegularExpressions;
using System.Windows.Forms;

namespace SoaringDotNET.Recorders {
    public class Flarm : SoaringDotNET.Recorders.SerialConnectionRecorder {
        private enum FlarmFlow : byte { STARTFRAME = 0x73, ESC = 0x78, ESC_ESC = 0x55, ESC_STARTFRAME = 0x31, EOF = 0x1A }
        private enum FlarmCmd : byte {
            ACK = 0xA0,// pos acknowledge
            NACK = 0xB7,// neg. acknowledge
            PING = 0x01,// test device for response
            SETBAUD = 0x02,// set baudrate for transmition
            FLASHUPLOAD = 0x10,// Upload flash page (not used here)
            EXIT = 0x12,// reset device, exit binary mode
            SELECTRECORD = 0x20,// select log record
            GETRECORDINFO = 0x21,// Datenbank schreiben
            GETIGCDATA = 0x22,// get IGC log
        }

        private struct FlarmFrameStructure {
            public UInt16 length;
            public byte version;
            public UInt16 seqCount;
            public byte type;
            public UInt16 CRC;
        };

        private const int FlarmFrameStructSize = 8;
        private UInt16 seqCounter = 1;

        public Flarm() {
            options = RecorderOptions.DownloadFlights;

            name = "Flarm";
            ManufacurerCode = "XFL";
            ManufacurerLetter = "X";
            supportedLineSpeeds = new object[] {4800, 9600, 19200, 38400, 57600};
            preferedLineSpeed = 19200;
        }

        #region Public Functions
        public override bool Open(string port, int speed) {
            // TODO:  Add SoaringPilot.Open implementation
            //comPort.Cnfg.SetFlowControl();
            bool ok;
            int retries = 0;
            int i;

            comPort.Parity = System.IO.Ports.Parity.None;
            comPort.DataBits = 8;
            comPort.StopBits = System.IO.Ports.StopBits.One;
            comPort.Encoding = Encoding.ASCII;

            for (i = 0; i < supportedLineSpeeds.Length; i++) {
                try {
                    if (base.Open(port, (int)supportedLineSpeeds[i])) {
                        System.Threading.Thread.Sleep(200);
                        string inS = comPort.ReadExisting();
                        MessageBox.Show(inS);
                        comPort.Close();
                    }
                    else {
                        return false;
                    }
                }
                catch (Exception exe) {
                    comPort.Close();
                    MessageBox.Show(exe.Message);
                }
            }
            return false;
            ok = base.Open(port, preferedLineSpeed);
            if (ok) {
                while (retries < 5) {
                    comPort.DiscardInBuffer();
                    comPort.DiscardOutBuffer();
                    
                    comPort.Write("$PFLAX\n");
                    try {
                        if (Ping()) {
                            break;
                        }
                        else {
                            retries++;
                        }
                    }
                    catch (RecorderTimeoutException) {
                        retries++;
                    }
                }
                if (retries >= 5) {
                    throw new Exception("Device does not respond! Check all connections and device status and try again.");
                }
                else {
                    try {
                        if (speed != preferedLineSpeed) {
                            SwitchComSpeed(speed);
                        }
                    }
                    catch (RecorderTimeoutException ex) {
                        throw new Exception(string.Format("{0} Check all connections and device status and try again.", ex.Message));
                    }
                }
            }
            return ok;
        }

        public override void Close() {
            FlarmFrameStructure frame = new FlarmFrameStructure();

            if (comPort.IsOpen) {
                frame.length = 8;
                frame.version = 1;
                frame.seqCount = seqCounter++;
                frame.type = (byte)FlarmCmd.EXIT;
                frame.CRC = CalcCRC(frame, null);
                SendFrame(frame, null);

                base.Close();
            }
        }
        #endregion

        #region Protected Functions
        protected override void GetSerialNo() {
            serialNo = 0;
            bool ok = true;
            DirectoryEntry de;

            // try to select 1st record and read recordinfo
            // select serial number from record info
            try {
                de = GetRecordInfo(0x00);
                serialNo = ToSerialNo(de.fileName.Substring(4, 3));
            }
            catch (NoMoreRecordsException) {
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                ok = false;
            }
            finally {
                if (RecorderBase.SerialNoThreadNotify != null) {
                    RecorderBase.SerialNoThreadNotify(ok);
                }
            }
        }

        protected override void GetFlightDir() {
            byte idx = 0x00;
            DirectoryEntry de;
            bool ok = true;

            flightDirectory.Clear();
            try {
                while (true) {
                    de = GetRecordInfo(idx);
                    flightDirectory.Add(de);
                    if (idx == 0xff) {
                        break;
                    }
                    else {
                        idx++;
                    }
                }
            }
            catch (NoMoreRecordsException) {
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                ok = false;
            }
            finally {
                if (RecorderBase.FlightDirectoryThreadNotify != null) {
                    RecorderBase.FlightDirectoryThreadNotify(ok);
                }
            }
        }

        protected override void DownloadFlights() {
            bool ok = false;
            bool finish = false;
            IsInterrupted = false;
            DirectoryEntry de;
            FlarmFrameStructure frame = new FlarmFrameStructure();
            FlarmFrameStructure retFrame = new FlarmFrameStructure();
            byte[] buffer;
            string file = "";
            Regex split = new Regex("\r\n");
            string[] tokens;

            try {
                ok = true;
                foreach (int i in flightIndexes) {
                    de = (DirectoryEntry)flightDirectory[i];
                    de.igcFile.Reset();

                    file = "";
                    finish = false;

                    SelectRecord((byte)i);

                    frame.length = 8;
                    frame.version = 1;
                    frame.type = (byte)FlarmCmd.GETIGCDATA;

                    while (!finish) {
                        frame.seqCount = seqCounter++;
                        frame.CRC = CalcCRC(frame, null);
                        SendFrame(frame, null);                        
                        buffer = ReadFrame(ref retFrame);
                        if (retFrame.type == (byte)FlarmCmd.ACK) {
                            file += Encoding.ASCII.GetString(buffer, 3, buffer.Length - 3);
                            finish = (file[file.Length - 1] == (char)FlarmFlow.EOF);
                        }
                        else {
                            ok = false;
                            break;
                        }
                    }

                    if (finish) {
                        file = file.Remove(file.Length - 1);
                        tokens = split.Split(file.Trim());
                        foreach (string line in tokens) {
                            switch (line[0]) {
                            case 'B':
                            case 'E':
                            case 'F':
                            case 'L':
                                de.igcFile.BRecord.Add(line);
                                break;
                            case 'A':
                                de.igcFile.A = line;
                                break;
                            case 'H':
                                switch (line.Substring(2, 3)) {
                                case "DTE":
                                    de.igcFile.DTE = line;
                                    break;
                                case "FXA":
                                    de.igcFile.FXA = line;
                                    break;
                                case "PLT":
                                    de.igcFile.PLT = line;
                                    break;
                                case "GTY":
                                    de.igcFile.GTY = line;
                                    break;
                                case "GID":
                                    de.igcFile.GID = line;
                                    break;
                                case "DTM":
                                    de.igcFile.DTM = line;
                                    break;
                                case "FTY":
                                    de.igcFile.FTY = line;
                                    break;
                                case "RFW":
                                    de.igcFile.RFW = line;
                                    break;
                                case "RHW":
                                    de.igcFile.RHW = line;
                                    break;
                                case "GPS":
                                    de.igcFile.GPS = line;
                                    break;
                                case "PRS":
                                    de.igcFile.PRS = line;
                                    break;
                                case "CID":
                                    de.igcFile.CID = line;
                                    break;
                                case "CCL":
                                    de.igcFile.CCL = line;
                                    break;
                                case "SIT":
                                    de.igcFile.SIT = line;
                                    break;
                                case "TZN":
                                    de.igcFile.TZN = line;
                                    break;
                                case "OOI":
                                    de.igcFile.OOI = line;
                                    break;
                                default:
                                    de.igcFile.HRecord.Add(line);
                                    break;
                                }
                                break;
                            case 'I':
                                de.igcFile.I = line;
                                break;
                            case 'G':
                                de.igcFile.GRecord.Add(line);
                                break;
                            case 'C':
                                de.igcFile.CRecord.Add(line);
                                break;
                            default:
                                string tmp = "";
                                foreach (char c in line) {
                                    tmp += string.Format("{0} ({1})\n", c, (int)c);
                                }
                                MessageBox.Show(tmp, "unrecognized record");
                                break;
                            }
                        }
                    }
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                ok = false;
            }
            finally {
                if (RecorderBase.DownloadFlightsThreadNotify != null) {
                    RecorderBase.DownloadFlightsThreadNotify(ok);
                }
            }
        }

        protected override void UploadSUAs() {
            if (RecorderBase.UploadSUAsThreadNotify != null) {
                RecorderBase.UploadSUAsThreadNotify(true);
            }
        }
        #endregion

        #region Private Functions
        private DirectoryEntry GetRecordInfo(byte recordNo) {
            FlarmFrameStructure frame = new FlarmFrameStructure();
            DirectoryEntry de = new DirectoryEntry();
            byte[] buffer;
            string s = "";
            string[] tokens;

            SelectRecord(recordNo);
            frame.length = 8;
            frame.version = 1;
            frame.seqCount = seqCounter++;
            frame.type = (byte)FlarmCmd.GETRECORDINFO;
            frame.CRC = CalcCRC(frame, null);

            SendFrame(frame, null);
            buffer = ReadFrame(ref frame);
            if (frame.type == (byte)FlarmCmd.ACK) {
                s = Encoding.ASCII.GetString(buffer, 2, buffer.Length - 2);
                tokens = s.Split('|');
                de.fileName = tokens[0];
                de.pilot = tokens[4];
                de.firstFix = de.firstFix = new DateTime(int.Parse(tokens[1].Substring(0, 4)), int.Parse(tokens[1].Substring(5, 2)), int.Parse(tokens[1].Substring(8, 2)),
                    int.Parse(tokens[2].Substring(0, 2)), int.Parse(tokens[2].Substring(3, 2)), int.Parse(tokens[2].Substring(6, 2)));
                de.lastFix = de.firstFix.Add(new TimeSpan(int.Parse(tokens[3].Substring(0, 2)), int.Parse(tokens[3].Substring(3, 2)), int.Parse(tokens[3].Substring(6, 2))));
                de.competitionId = tokens[5];
            }
            return de;
        }

        private void SelectRecord(byte recordNo) {
            FlarmFrameStructure frame = new FlarmFrameStructure();
            byte[] b = new byte[1];

            // try to select 1st record and read recordinfo
            // select serial number from record info
            b[0] = recordNo;

            frame.length = 9;
            frame.version = 1;
            frame.seqCount = seqCounter++;
            frame.type = (byte)FlarmCmd.SELECTRECORD;
            frame.CRC = CalcCRC(frame, b);

            SendFrame(frame, b);
            ReadFrame(ref frame);
            if (frame.type != (byte)FlarmCmd.ACK) {
                throw new NoMoreRecordsException();
            }
        }

        private UInt16 UpdateCRC(UInt16 crc, byte data) {
            crc = (UInt16)(crc ^ (UInt16)(data << 8));
            for (int i = 0; i < 8; i++) {
                if ((crc & 0x8000) == 0x8000) {
                    crc = (UInt16)((crc << 1) ^ 0x1021);
                }
                else {
                    crc <<= 1;
                }
            }
            return crc;
        }

        private UInt16 CalcCRC(FlarmFrameStructure frame, byte[] buffer) {
            UInt16 crc = 0;
            int i;

            crc = UpdateCRC(crc, (byte)(frame.length));
            crc = UpdateCRC(crc, (byte)(frame.length >> 8));
            crc = UpdateCRC(crc, (byte)(frame.version));
            crc = UpdateCRC(crc, (byte)(frame.seqCount));
            crc = UpdateCRC(crc, (byte)(frame.seqCount >> 8));
            crc = UpdateCRC(crc, (byte)(frame.type));

            if (buffer != null) {
                for (i = 0; i < buffer.Length; i++) {
                    crc = UpdateCRC(crc, buffer[i]);
                }
            }
            return crc;
        }

        private FlarmFrameStructure DecodeFrame(byte[] buffer) {
            FlarmFrameStructure frame = new FlarmFrameStructure();
            byte [] b = new byte[2];
            Buffer.BlockCopy(buffer, 0, b, 0, 2);
            frame.length = LSBToInt(b);
            frame.version = buffer[2];
            Buffer.BlockCopy(buffer, 3, b, 0, 2);
            frame.seqCount = LSBToInt(b);
            frame.type = buffer[5];
            Buffer.BlockCopy(buffer, 6, b, 0, 2);
            frame.CRC = LSBToInt(b);

            return frame;
        }

        private byte[] EncodeFrame(FlarmFrameStructure frame) {
            byte[] b = new byte[FlarmFrameStructSize];

            b[1] = (byte)(frame.length >> 8);
            b[0] = (byte)(frame.length);
            b[2] = (byte)(frame.version);
            b[4] = (byte)(frame.seqCount >> 8);
            b[3] = (byte)(frame.seqCount);
            b[5] = (byte)(frame.type);
            b[7] = (byte)(frame.CRC >> 8);
            b[6] = (byte)(frame.CRC);
            
            return b;
        }

        private void SendFrame(FlarmFrameStructure frame, byte[] data) {
            byte[] startFrame = new byte[1];
            startFrame[0] = (byte)FlarmFlow.STARTFRAME;

            comPort.DiscardOutBuffer();
            comPort.DiscardInBuffer();

            comPort.Write(startFrame, 0, 1);
            comPort.Write(EncodeFrame(frame), 0, FlarmFrameStructSize);
            if (data != null) {
                comPort.Write(data, 0, data.Length);
            }
        }

        private byte[] ReadFrame(ref FlarmFrameStructure frame) {
            byte[] frameBytes;
            byte[] buffer;
            int ret;

            long s = System.DateTime.Now.Ticks;
            while (true) {
                try {
                    ret = comPort.ReadByte();
                    if (ret == (int)FlarmFlow.STARTFRAME) {
                        break;
                    }
                }
                catch (TimeoutException) {
                    if (System.DateTime.Now.Ticks - s > 50000000) {
                        throw new RecorderTimeoutException("no response from logger within 5 secs.");
                    }
                }
            }

            frameBytes = ReadStream(FlarmFrameStructSize);
            frame = DecodeFrame(frameBytes);

            buffer = ReadStream(frame.length - FlarmFrameStructSize);
            if (frame.CRC != CalcCRC(frame, buffer)) {
                throw new Exception("CRC error");
            }

            return buffer;
        }

        private byte[] IntToLSB(int number) {
            byte[] b = new byte[2];
            b[0] = (byte)number;
            b[1] = (byte)(number >> 8);
            return b;
        }

        private UInt16 LSBToInt(byte[] b) {
            UInt16 number;
            number = (UInt16)((b[1] << 8) | b[0]);
            return number;
        }

        private void SwitchComSpeed(int speed) {
            FlarmFrameStructure frame = new FlarmFrameStructure();
            byte[] b = new byte[1];
            int speedIdx;
            int retries = 0;

            switch (speed) {
            case 4800:
                speedIdx = 0;
                break;
            case 9600:
                speedIdx = 1;
                break;
            case 19200:
                speedIdx = 2;
                break;
            case 38400:
                speedIdx = 4;
                break;
            case 57600:
                speedIdx = 5;
                break;
            default:
                speedIdx = 2;
                break;
            }
            b[0] = (byte)speedIdx;

            frame.length = 9;
            frame.version = 1;
            frame.seqCount = seqCounter++;
            frame.type = (byte)FlarmCmd.SETBAUD;
            frame.CRC = CalcCRC(frame, b);

            SendFrame(frame, b);
            ReadFrame(ref frame);
            if (frame.type != (byte)FlarmCmd.ACK) {
                throw new RecorderTimeoutException(string.Format("could not switch to {0} baud", speed));
            }
            comPort.Close();
            comPort.BaudRate = speed;
            comPort.Open();

            while (retries++ < 10) {
                if (Ping()) {
                    break;
                }
            }
            if (retries >= 10) {
                throw new RecorderTimeoutException(string.Format("could not switch to {0} baud", speed));
            }
        }

        private bool Ping() {
            bool ok = true;
            FlarmFrameStructure frame = new FlarmFrameStructure();

            frame.length = 8;
            frame.version = 1;
            frame.seqCount = seqCounter++;
            frame.type = (byte)FlarmCmd.PING;
            frame.CRC = CalcCRC(frame, null);
            SendFrame(frame, null);

            try {
                ReadFrame(ref frame);
                if (frame.type != (byte)FlarmCmd.ACK) {
                    ok = false;
                }
            }
            catch {
                ok = false;
            }
            return ok;
        }

        private byte[] ReadStream(int cnt) {
            byte[] buffer = new byte[cnt];
            int i;
            int ret;
            bool escape = false;
            
            i = 0;
            long s = System.DateTime.Now.Ticks;
            while (i < cnt) {
                try {
                    ret = comPort.ReadByte();
                    switch (ret) {
                    case (int)FlarmFlow.ESC:
                        escape = true;
                        break;
                    case (int)FlarmFlow.ESC_ESC:
                        if (escape) {
                            ret = (int)FlarmFlow.ESC;
                            escape = false;
                        }
                        goto default;
                    case (int)FlarmFlow.ESC_STARTFRAME:
                        if (escape) {
                            ret = (int)FlarmFlow.STARTFRAME;
                            escape = false;
                        }
                        goto default;
                    default:
                        buffer[i++] = (byte)ret;
                        break;
                    }
                }
                catch (TimeoutException) {
                    if (System.DateTime.Now.Ticks - s > 50000000) {
                        throw new RecorderTimeoutException("no response from logger within 5 secs.");
                    }
                }
            }
        
            return buffer;
        }
        #endregion

        #region Attributes
        public override string Description {
            get {
                // TODO:  Add SoaringPilot.Description getter implementation
                return string.Format(@"Adapter for Flarm anti-collision system

This is a serial connection adapter. The preferred line speed is {0} baud.
Please wait until the device selftest is completed. Otherwise you may get a device not responding error.
If that happen, try to connect again.

", preferedLineSpeed) + base.OptionsString;
            }
        }
        #endregion
    }
}

internal class NoMoreRecordsException : Exception {
    public NoMoreRecordsException()
        : base() {
    }
}
