// name   : Flight.cs
// author : Harald Maier
// date   : 17.10.2003
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.IO;
using System.Text.RegularExpressions;
using System.Drawing;
using System.Windows.Forms;
using System.Collections;
using System.Threading;

using SoaringDotNET.Data;

namespace SoaringDotNET.GUI
{
	/// <summary>
	/// 
	/// </summary>
    public class Flight : SoaringDotNET.GUI.MapLineElement {
        private string name;
        private string fileName;
        private string pilot;
        private string date;
        private string glider;
        private string gliderId;
        private string compClass;
        private string compId;
        private int gliderIndex = 100;

        private int startIdx = -1;
        private int stopIdx = -1;
        private int takeoffIdx = -1;
        private int landingIdx = -1;
        
        public Task currentTask;
        public Task originalTask;
        public ArrayList optimizedTasks;

        public float [][] lengthMatrix = null;
        public int [] minElevationSinceStartOfTask = null;
        
        public Thread calcDistanceThread = null;

        public Flight (string f) :
            base(MapElementTypes.Flight) {
            FileInfo fi = new FileInfo(f);
            this.name = fi.Name;
            this.fileName = f;

            currentTask = new Task();
            originalTask = currentTask;

            optimizedTasks = new ArrayList();
            ReadIGCFile();
            GC.Collect();

            int i = 0;
            try {
                lengthMatrix = new float[Count][];
                for (i = 0; i < Count; i++) {
                    lengthMatrix[i] = new float[i + 1];
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.Message);
            }
            calcDistanceThread = new Thread(new ThreadStart(CalculateDistances));
            calcDistanceThread.IsBackground = true;

            calcDistanceThread.Start();
            optimizedTasks.Add(new Optimization.Declaration(this));
            optimizedTasks.Add(new Optimization.OLC2006Classic(this));
            optimizedTasks.Add(new Optimization.OLC2006FAI(this));
            optimizedTasks.Add(new Optimization.DMSt2004(this));
            optimizedTasks.Add(new Optimization.DMSt2004FAI(this));

            pen = new Pen(AppContents.flightColor.DrawColor);
            pen.Width = AppContents.flightColor.Width;
        }

        #region Public Functions
        public override void Draw(Graphics g, Pen pen, Brush brush, WgsArea view, int width, int height) {
            Pen oldPen = this.pen;
            this.pen = pen;
            this.Draw(g, view, width, height);
            this.pen = oldPen;
            this.pen.Color = AppContents.flightColor.DrawColor;
            currentTask.Draw(g, pen, brush, view, width, height);
        }

        public override void Draw(Graphics g, WgsArea view, int width, int height) {
            this.pen.Color = AppContents.flightColor.DrawColor;
            this.pen.Width = AppContents.flightColor.Width;
            base.Draw(g, view, width, height);
            currentTask.Draw(g, view, width, height);
        }

        public void CalcFlightArea() {
            // TODO:  Add Flight.CalcArea implementation
            base.CalcArea();
        }

        public void CalcTaskArea() {
            // TODO:  Add Flight.CalcArea implementation
            //area.TopLeft = currentTask.Area.TopLeft;
            //area.BottomRight = currentTask.Area.BottomRight;
            area.Area = currentTask.Area;
        }
    
        public void CalcTotalArea() {
            CalcArea();
        }

        #endregion

        #region Attributes
        public int RecordingTime {
            get {
                int start = this[0].fixTime;
                int end = this[Count - 1].fixTime;
                if (end < start) {
                    // next UTC day
                    return 86400 - start + end;
                }
                else {
                    return end - start;
                }
            }
        }

        public int Duration {
            get {
                int start = this[Takeoff].fixTime;
                int end = this[Landing].fixTime;
                if (end < start) {
                    // next UTC day
                    return 86400 - start + end;
                }
                else {
                    return end - start;
                }
            }
        }

        public int TaskDuration {
            get {
                if (currentTask.Count > 3 && currentTask[1].reachPoint.fixTime != -1 && currentTask[currentTask.Count - 2].reachPoint.fixTime != -1) {
                    int start = currentTask[1].reachPoint.fixTime;
                    int end = currentTask[currentTask.Count - 2].reachPoint.fixTime;
                    if (end < start) {
                        // next UTC day
                        return 86400 - start + end;
                    }
                    else {
                        return end - start;
                    }
                }
                else {
                    return Duration;
                }
            }
        }

        public new FlightPoint this[int pos] {
            get {
                return (FlightPoint)points[pos];
            }
        }

        public string Name {
            get {
                return name;
            }
        }

        public string IGCName {
            get {
                int pos = name.IndexOf(".");
                return (pos != -1 ? name.Substring(0, pos) : name);
            }
        }

        public string FileName {
            get {
                return fileName;
            }
        }

        public string Date {
            get {
                return date;
            }
        }

        public string Pilot {
            get {
                return pilot;
            }
        }

        public string Glider {
            get {
                return glider;
            }
            set {
                glider = value;
            }
        }

        public string GliderId {
            get {
                return gliderId;
            }
        }

        public int GliderIndex {
            get {
                return gliderIndex;
            }
            set {
                gliderIndex = value;
            }
        }

        public string CompetitionClass {
            get {
                return compClass;
            }
        }

        public string CompetitionId {
            get {
                return compId;
            }
        }

        public int Start {
            get {
                return startIdx;
            }
            set {
                if (value >= 0 && value < this.Count) {
                    if (value > stopIdx && stopIdx != -1) {
                        startIdx = stopIdx;
                        stopIdx = value;
                    }
                    else {
                        startIdx = value;
                    }
                }
                else {
                    startIdx = 0;
                }
                
                if (startIdx < takeoffIdx) {
                    takeoffIdx = startIdx;
                }
            }
        }

        public int Stop {
            get {
                return stopIdx;
            }
            set {
                if (value >= 0 && value < this.Count) {
                    if (value < startIdx && startIdx != -1) {
                        stopIdx = startIdx;
                        startIdx = value;
                    }
                    else {
                        stopIdx = value;
                    }
                }
                else {
                    stopIdx = this.Count - 1;
                }

                if (stopIdx > landingIdx) {
                    landingIdx = stopIdx;
                }
            }
        }

        public int Takeoff {
            get {
                return takeoffIdx;
            }
            set {
                if (value >= 0 && value < this.Count) {
                    if (value > startIdx) {
                        // takeoff not after start 
                        startIdx = value;
                    }
                    takeoffIdx = value;
                }
                else {
                    takeoffIdx = 0;
                }
            }
        }

        public int Landing {
            get {
                return landingIdx;
            }
            set {
                if (value >= 0 && value < this.Count) {
                    if (value < stopIdx) {
                        // landing not befor stop
                        stopIdx = value;
                    }
                    landingIdx = value;
                }
                else {
                    landingIdx = this.Count - 1;
                }
            }
        }
        #endregion

        #region Private Functions
        private void ReadIGCFile() {
            string line;
            Regex BRecord = new Regex("B([0-9]{2})([0-9]{2})([0-9]{2})([0-9]{2})([0-9]{5})([N|S])([0-9]{3})([0-9]{5})([E|W])([A|V])(-[0-9]{4}|[0-9]{5})(-[0-9]{4}|[0-9]{5})(.*)");
            Regex CRecord = new Regex("C([0-9]{2})([0-9]{5})([N|S])([0-9]{3})([0-9]{5})([E|W])(.*)");
            Match m;
            int colon, n, i, time, pAlt, gnssAlt, engineLevel;
            int lat, lon;
            int start, stop;
            int startEngineLevel;
            int stopEngineLevel;
            bool haveCRecord = false;
            bool proceedAnyway = false;
            bool takeoffAdded = false;
            bool landingAdded = false;
            GroupCollection coll;
            WayPoint wayP;
            FlightPoint fP;

            startEngineLevel = -1;
            stopEngineLevel = -1;
            engineLevel = 0;

            using (StreamReader inStream = new StreamReader(fileName)) {
                while ((line = inStream.ReadLine()) != null) {
                    line.Trim();
                    switch (line[0]) {
                    case 'B':
                        m = BRecord.Match(line);
                        if (m.Success) {
                            coll = m.Groups;
                            time = (int.Parse(coll[1].Value) * 3600) + (int.Parse(coll[2].Value) * 60) + int.Parse(coll[3].Value);
                            lat = (int.Parse(coll[4].Value) * 360000) + (int.Parse(coll[5].Value) * 6);
                            lon = (int.Parse(coll[7].Value) * 360000) + (int.Parse(coll[8].Value) * 6);
                            if (coll[6].Value == "N") {
                                lat = -lat;
                            }
                            if (coll[9].Value == "W") {
                                lon = -lon;
                            }
                            pAlt = int.Parse(coll[11].Value);
                            gnssAlt = int.Parse(coll[12].Value);

                            if (startEngineLevel != -1) {
                                engineLevel = int.Parse(line.Substring(startEngineLevel, stopEngineLevel - startEngineLevel + 1));
                            }

                            if (coll[10].Value == "A") {
                                fP = new FlightPoint(lat, lon, time, pAlt, gnssAlt, engineLevel);
                                points.Add(fP);
                            }
                        }
                        else {
                            throw new Exception("invalid B-Record found in file " + name);
                        }
                        break;
                    case 'C':
                        // read number of waypoints; n is 4 less than real number (+ takeoff, start, finish, landing)
                        // check if we had already a C-record block
                        // first definition way be wrong (1 less than real), found this on some LX20
                        // ask user to proceed and read one line each time
                        if (haveCRecord) {
                            if (!proceedAnyway) {
                                if (MessageBox.Show("WARNING: found further C-Record after C-Record block.\nThe C-Record declaration said " +
                                    currentTask.Count.ToString() + " waypoints!\nProceed anyway?", "Warning", MessageBoxButtons.YesNo, MessageBoxIcon.Exclamation) == DialogResult.Yes) {
                                    proceedAnyway = true;
                                }
                                else {
                                    points.Clear();
                                    currentTask.Clear();
                                    return;
                                }
                            }

                            line.Trim();
                            m = CRecord.Match(line);
                            if (m.Success) {
                                coll = m.Groups;
                                lat = (int.Parse(coll[1].Value) * 360000) + (int.Parse(coll[2].Value) * 6);
                                lon = (int.Parse(coll[4].Value) * 360000) + (int.Parse(coll[5].Value) * 6);
                                if (lat != 0 || lon != 0) {
                                    if (coll[3].Value == "N") {
                                        lat = -lat;
                                    }
                                    if (coll[6].Value == "W") {
                                        lon = -lon;
                                    }
                                    wayP = new WayPoint(lat, lon, coll[7].Value);
                                    currentTask.Add(wayP);
                                }
                            }
                            else {
                                throw new Exception("invalid C-Record found in file " + name);
                            }
                            n = 0;
                        }
                        else {
                            n = int.Parse(line.Substring(23, 2)) + 4;
                        }
                        i = 0;
                        while (i++ < n && (line = inStream.ReadLine()) != null) {
                            line.Trim();
                            m = CRecord.Match(line);
                            if (m.Success) {
                                coll = m.Groups;
                                lat = (int.Parse(coll[1].Value) * 360000) + (int.Parse(coll[2].Value) * 6);
                                lon = (int.Parse(coll[4].Value) * 360000) + (int.Parse(coll[5].Value) * 6);
                                if (lat != 0 || lon != 0) {
                                    if (coll[3].Value == "N") {
                                        lat = -lat;
                                    }
                                    if (coll[6].Value == "W") {
                                        lon = -lon;
                                    }
                                    wayP = new WayPoint(lat, lon, coll[7].Value);
                                    currentTask.Add(wayP);
                                }
                            }
                            else {
                                throw new Exception("invalid C-Record found in file " + name);
                            }
                        }
                        haveCRecord = true;
                        break;
                    case 'H':
                        colon = line.IndexOf(':') + 1;
                        switch (line.Substring(2, 3)) {
                        case "CCL":
                            // competition class
                            compClass = line.Substring(colon).Trim();
                            break;
                        case "CID":
                            // competition Id
                            compId = line.Substring(colon).Trim();
                            break;
                        case "DTE":
                            // Date
                            try {
                                date = string.Format("{0}.{1}.{2}", line.Substring(5, 2), line.Substring(7, 2), line.Substring(9, 2));
                            }
                            catch {
                                date = "00.00.00";
                            }
                            break;
                        case "PLT":
                            // pilot
                            pilot = line.Substring(colon).Trim();
                            break;
                        case "GTY":
                            // glider type
                            glider = line.Substring(colon).Trim();
                            break;
                        case "GID":
                            // glider ID
                            gliderId = line.Substring(colon).Trim();
                            break;
                        }
                        break;
                    case 'I':
                        n = int.Parse(line.Substring(1, 2));
                        for (i = 0; i < n; i++) {
                            start = int.Parse(line.Substring(3 + (i * 7), 2));
                            stop = int.Parse(line.Substring(5 + (i * 7), 2));
                            if (line.Substring(7 + (i * 7), 3) == "ENL") {
                                startEngineLevel = start - 1;
                                stopEngineLevel = stop - 1;
                            }
                        }
                        break;
                    case 'L':
                        if (line.ToUpper().IndexOf("TAKEOFF DETECTED") != -1) {
                            // use internal variable to provent index checking
                            startIdx = Count - 1;
                        }
                        break;
                    }
                }

                // some recorders have "incomplete" tasks, maybe takeoff and landing are missing
                // do some tests and extend Task
                if (currentTask.TaskType == TaskTypes.Unknown && currentTask.Count > 0) {
                    // first extend landing
                    currentTask.Add(currentTask[currentTask.Count - 1]);
                    if (currentTask.TaskType == TaskTypes.Unknown) {
                        // try takeoff and landing
                        currentTask.Insert(0, currentTask[0]);
                        if (currentTask.TaskType == TaskTypes.Unknown) {
                            // try only takeoff
                            currentTask.Delete(currentTask.Count - 1);
                            if (currentTask.TaskType == TaskTypes.Unknown) {
                                // give it up
                                currentTask.Delete(0);
                            }
                            else {
                                takeoffAdded = true;
                            }
                        }
                        else {
                            landingAdded = true;
                            takeoffAdded = true;
                        }
                    }
                    else {
                        landingAdded = true;
                    }
                }

                if (takeoffAdded || landingAdded) {
                    ;
                }

                FindTakeoffAndLanding();
                CalcArea();
            }
        }

        private void FindTakeoffAndLanding() {
            int i;
            FlightPoint p1, p2;

            if (Start == -1) {
                for (i = 0; i < Count - 1; i++) {
                    p1 = this[i];
                    p2 = this[i + 1];
                    if (p1.speedKMH(p2) > 20 && p1.varioMPS(p2) > 1) {
                        startIdx = i;
                        break;
                    }
                }
                if (Start == -1) {
                    startIdx = 0;
                }
            }

            if (Stop == -1) {
                for (i = Count - 1; i > 0; i--) {
                    p1 = this[i - 1];
                    p2 = this[i];
                    if (p1.speedKMH(p2) > 20) {
                        stopIdx = i;
                        break;
                    }
                }

                if (Stop == -1) {
                    stopIdx = Count;
                }
            }

            takeoffIdx = startIdx;
            landingIdx = stopIdx;
        }

        void CalculateDistances() {
            // calc length matrix for optimization points
            int i, j;
            float [] lVector;
            FlightPoint p;

            for (i = 0; i < Count; i++) {
                lVector = lengthMatrix[i];
                p = (FlightPoint)points[i];
                for (j = 0; j < i; j++) {
                    lVector[j] = (float)p.distanceKM((FlightPoint)points[j]);
                }
            }
        }
        #endregion

        #region Protected Functions
        protected override void CalcArea() {
            // TODO:  Add Flight.CalcArea implementation
            base.CalcArea();
            if (!currentTask.Area.IsEmpty) {
                area.Union(currentTask.Area);
            }
        }
    
        #endregion    
    }
}
