// name   : Map.cs
// author : Harald Maier
// date   : 03.08.2003
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;

using SoaringDotNET.GUI;
using SoaringDotNET.Data;

namespace SoaringDotNET.Controls
{
    /// <summary>
    /// Summary description for Map.
    /// </summary>
    public class Map : System.Windows.Forms.UserControl
    {
        private WgsArea viewArea;

        private Font font;
        private Pen dashPen;
        
        private Bitmap mapBmp = null;
        private TextureBrush mapBrush = null;
        private Pen mapPen = null;

        private Bitmap totalBmp = null;
        private TextureBrush totalBrush = null;
        private Pen totalPen = null;

        private Pen zoomPen = null;
        private int oldX = 0, oldY = 0;
        private int oldWidth = 0, oldHeight = 0;
        private int startX = 0, startY = 0;
        private Point popupWindowPos = new Point(0, 0);
        private int draggingTaskPointIdx = -1;
        private bool draggingArea;
        private bool draggingTask;
        private bool moving;

        private WaypointCatalog currentCatalog;
        private Flight currentFlight;
        private Task currentTask;
        private WayPoint dummyPoint;
        private AirspcaePopupInfo toolTip;
        private ContextMenuStrip contextMenuStripMap;
        private System.ComponentModel.IContainer components;
        private ToolStripMenuItem centerMapToolStripMenuItem;
        private ToolStripMenuItem centerMapToHomepointToolStripMenuItem;
        private ToolStripMenuItem centerMapTotaskToolStripMenuItem;
        private ToolStripMenuItem centerMapToflightToolStripMenuItem;
        private ToolStripSeparator toolStripSeparator1;
        private ToolStripMenuItem newTaskToolStripMenuItem;
        private ToolStripSeparator toolStripSeparator2;
        private ToolStripMenuItem newWaypointToolStripMenuItem;
        private ToolStripMenuItem editWaypointToolStripMenuItem;
        private ToolStripSeparator toolStripSeparator3;
        private ToolStripMenuItem airspaceinfoToolStripMenuItem;

        private enum TaskActions {MovePoint, AddPoint, RemovePoint};
        private TaskActions taskAction;

        public Map() {
            try {
                font = new Font("Arial", 8.0f);
            }
            catch {
                font = Font;
            }
            dashPen = new Pen(Color.Black);
            dashPen.DashStyle = DashStyle.Dash;
            zoomPen = new Pen(Color.Black);
            zoomPen.DashStyle = DashStyle.Dot;
            //
            // Required for Windows Form Designer support
            //
            InitializeComponent();

            //
            // TODO: Add any constructor code after InitializeComponent call
            //
            // this is the default viewing area; values are used as default in SoaringDotNet.ReadConfig();
            viewArea = new WgsArea(-49*360000, 8*360000, -47*360000, 10*360000);

            currentCatalog = null;
            currentFlight = null;
            currentTask = null;

            draggingArea = false;
            draggingTask = false;
            moving = false;

            dummyPoint = new WayPoint();
            taskAction = TaskActions.MovePoint;
            toolTip = new AirspcaePopupInfo(this);
        }

        #region Windows Form Designer generated code
        /// <summary>
        /// Required method for Designer support - do not modify
        /// the contents of this method with the code editor.
        /// </summary>
        private void InitializeComponent() {
            this.components = new System.ComponentModel.Container();
            this.contextMenuStripMap = new System.Windows.Forms.ContextMenuStrip(this.components);
            this.centerMapToolStripMenuItem = new System.Windows.Forms.ToolStripMenuItem();
            this.centerMapToHomepointToolStripMenuItem = new System.Windows.Forms.ToolStripMenuItem();
            this.centerMapTotaskToolStripMenuItem = new System.Windows.Forms.ToolStripMenuItem();
            this.centerMapToflightToolStripMenuItem = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator1 = new System.Windows.Forms.ToolStripSeparator();
            this.newTaskToolStripMenuItem = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator2 = new System.Windows.Forms.ToolStripSeparator();
            this.newWaypointToolStripMenuItem = new System.Windows.Forms.ToolStripMenuItem();
            this.editWaypointToolStripMenuItem = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator3 = new System.Windows.Forms.ToolStripSeparator();
            this.airspaceinfoToolStripMenuItem = new System.Windows.Forms.ToolStripMenuItem();
            this.contextMenuStripMap.SuspendLayout();
            this.SuspendLayout();
            // 
            // contextMenuStripMap
            // 
            this.contextMenuStripMap.Items.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this.centerMapToolStripMenuItem,
            this.centerMapToHomepointToolStripMenuItem,
            this.centerMapTotaskToolStripMenuItem,
            this.centerMapToflightToolStripMenuItem,
            this.toolStripSeparator1,
            this.newTaskToolStripMenuItem,
            this.toolStripSeparator2,
            this.newWaypointToolStripMenuItem,
            this.editWaypointToolStripMenuItem,
            this.toolStripSeparator3,
            this.airspaceinfoToolStripMenuItem});
            this.contextMenuStripMap.Name = "contextMenuStripMap";
            this.contextMenuStripMap.Size = new System.Drawing.Size(208, 220);
            this.contextMenuStripMap.Opening += new System.ComponentModel.CancelEventHandler(this.OnOpeningContextMenu);
            // 
            // centerMapToolStripMenuItem
            // 
            this.centerMapToolStripMenuItem.Name = "centerMapToolStripMenuItem";
            this.centerMapToolStripMenuItem.Size = new System.Drawing.Size(207, 22);
            this.centerMapToolStripMenuItem.Text = "&Center map";
            this.centerMapToolStripMenuItem.Click += new System.EventHandler(this.OnCenterMap);
            // 
            // centerMapToHomepointToolStripMenuItem
            // 
            this.centerMapToHomepointToolStripMenuItem.Image = global::SoaringDotNET.Properties.Resources.Home;
            this.centerMapToHomepointToolStripMenuItem.Name = "centerMapToHomepointToolStripMenuItem";
            this.centerMapToHomepointToolStripMenuItem.Size = new System.Drawing.Size(207, 22);
            this.centerMapToHomepointToolStripMenuItem.Text = "Center map to &homepoint";
            this.centerMapToHomepointToolStripMenuItem.Click += new System.EventHandler(this.OnCenterHome);
            // 
            // centerMapTotaskToolStripMenuItem
            // 
            this.centerMapTotaskToolStripMenuItem.Image = global::SoaringDotNET.Properties.Resources.Task;
            this.centerMapTotaskToolStripMenuItem.Name = "centerMapTotaskToolStripMenuItem";
            this.centerMapTotaskToolStripMenuItem.Size = new System.Drawing.Size(207, 22);
            this.centerMapTotaskToolStripMenuItem.Text = "Center map to &task";
            this.centerMapTotaskToolStripMenuItem.Click += new System.EventHandler(this.OnCenterTask);
            // 
            // centerMapToflightToolStripMenuItem
            // 
            this.centerMapToflightToolStripMenuItem.Image = global::SoaringDotNET.Properties.Resources.Flight;
            this.centerMapToflightToolStripMenuItem.Name = "centerMapToflightToolStripMenuItem";
            this.centerMapToflightToolStripMenuItem.Size = new System.Drawing.Size(207, 22);
            this.centerMapToflightToolStripMenuItem.Text = "Center map to &flight";
            this.centerMapToflightToolStripMenuItem.Click += new System.EventHandler(this.OnCenterFlight);
            // 
            // toolStripSeparator1
            // 
            this.toolStripSeparator1.Name = "toolStripSeparator1";
            this.toolStripSeparator1.Size = new System.Drawing.Size(204, 6);
            // 
            // newTaskToolStripMenuItem
            // 
            this.newTaskToolStripMenuItem.Image = global::SoaringDotNET.Properties.Resources.NewTask;
            this.newTaskToolStripMenuItem.Name = "newTaskToolStripMenuItem";
            this.newTaskToolStripMenuItem.Size = new System.Drawing.Size(207, 22);
            this.newTaskToolStripMenuItem.Text = "New t&ask";
            this.newTaskToolStripMenuItem.Click += new System.EventHandler(this.OnCreateTask);
            // 
            // toolStripSeparator2
            // 
            this.toolStripSeparator2.Name = "toolStripSeparator2";
            this.toolStripSeparator2.Size = new System.Drawing.Size(204, 6);
            // 
            // newWaypointToolStripMenuItem
            // 
            this.newWaypointToolStripMenuItem.Image = global::SoaringDotNET.Properties.Resources.NewWaypoint;
            this.newWaypointToolStripMenuItem.Name = "newWaypointToolStripMenuItem";
            this.newWaypointToolStripMenuItem.Size = new System.Drawing.Size(207, 22);
            this.newWaypointToolStripMenuItem.Text = "New &waypoint";
            this.newWaypointToolStripMenuItem.Click += new System.EventHandler(this.OnCreateWaypoint);
            // 
            // editWaypointToolStripMenuItem
            // 
            this.editWaypointToolStripMenuItem.Image = global::SoaringDotNET.Properties.Resources.Edit;
            this.editWaypointToolStripMenuItem.Name = "editWaypointToolStripMenuItem";
            this.editWaypointToolStripMenuItem.Size = new System.Drawing.Size(207, 22);
            this.editWaypointToolStripMenuItem.Text = "&Edit waypoint";
            this.editWaypointToolStripMenuItem.Click += new System.EventHandler(this.OnEditWaypoint);
            // 
            // toolStripSeparator3
            // 
            this.toolStripSeparator3.Name = "toolStripSeparator3";
            this.toolStripSeparator3.Size = new System.Drawing.Size(204, 6);
            // 
            // airspaceinfoToolStripMenuItem
            // 
            this.airspaceinfoToolStripMenuItem.Name = "airspaceinfoToolStripMenuItem";
            this.airspaceinfoToolStripMenuItem.Size = new System.Drawing.Size(207, 22);
            this.airspaceinfoToolStripMenuItem.Text = "Airspace &info";
            this.airspaceinfoToolStripMenuItem.Click += new System.EventHandler(this.OnAirspaceInfo);
            // 
            // Map
            // 
            this.BackColor = System.Drawing.Color.AliceBlue;
            this.ContextMenuStrip = this.contextMenuStripMap;
            this.Cursor = System.Windows.Forms.Cursors.Default;
            this.Name = "Map";
            this.Size = new System.Drawing.Size(292, 266);
            this.contextMenuStripMap.ResumeLayout(false);
            this.ResumeLayout(false);

        }
        #endregion

        #region Protected Functions
        protected override void OnPaint(PaintEventArgs e) {
            // TODO:  Add Map.OnPaint implementation
            Graphics g = e.Graphics;
            Graphics bmpMap, bmpTotal;
            long text;
            long lat, lon;
            long minLat, latRange;
            long minLon, lonRange;
            long step, startPos, diff;
			//int width = Width - 1;
			//int height = Height - 1;
            string txt;
            Pen drawingPen;

            // create a new map Bitmap
            // this is the "background" drawing
            if (oldWidth != Width || oldHeight != Height) {
                mapBmp = new Bitmap(Width, Height, CreateGraphics());
                totalBmp = new Bitmap(Width, Height, CreateGraphics());
                oldWidth = Width;
                oldHeight = Height;
            }

            bmpMap = Graphics.FromImage(mapBmp);
            bmpMap.Clear(BackColor);
            bmpTotal = Graphics.FromImage(totalBmp);

            SizeArea();

            minLat = viewArea.TopLat;
            latRange = viewArea.Height;

            minLon = viewArea.LeftLon;
            lonRange = viewArea.Width;

            // draw map raster
            step = 90000;
            while(latRange / step > 10) {
                step += 90000;
            }

            diff = (Math.Abs(minLat) % step);
            if (diff > 0) {
                if (minLat < 0) {
                    startPos = diff;
                }
                else {
                    startPos = step - diff;
                }
            }
            else {
                startPos = 0;
            }

            //text = minLat + diff;
            text = minLat + startPos;

            while (startPos <= latRange) {
                lat = startPos * Height / latRange;
                if (text % 360000 == 0) {
                    txt = string.Format("{0:00}{1}", Math.Abs(text / 360000), text <= 0 ? "N" : "S");
                    bmpMap.DrawString(txt, font, Brushes.Black, 0, lat - 12);
                    drawingPen = Pens.Black;
                }
                else if (text % 180000 == 0) {
                    txt = string.Format("{0:00}30{1}", Math.Abs(text / 360000), text <= 0 ? "N" : "S");
                    bmpMap.DrawString(txt, font, Brushes.Black, 0, lat - 12);
                    drawingPen = dashPen;
                }
                else {
                    drawingPen = dashPen;
                }

                startPos += step;
                text += step;

                bmpMap.DrawLine(drawingPen, 0, lat, Width, lat);
            }
            
            step = 90000;
            while(lonRange / step > 10) {
                step += 90000;
            }

            diff = (Math.Abs(minLon) % step);
            if (diff > 0) {
                if (minLon < 0) {
                    startPos = diff;
                }
                else {
                    startPos = step - diff;
                }
            }
            else {
                startPos = 0;
            }

            text = minLon + startPos;
            //text = minLon + (minLon >= 0 ? startPos : -startPos);            
            while (startPos <= lonRange) {
                lon = startPos * Width / lonRange;

                if (text <= -64800000) {
                    text += 129600000;
                }
                else if (text > 64800000) {
                    text -= 129600000;
                }

                if (text % 360000 == 0) {
                    txt = string.Format("{0:000}{1}", Math.Abs(text / 360000), text >= 0 ? "E" : "W");
                    bmpMap.DrawString(txt, font, Brushes.Black, lon, 0);
                    drawingPen = Pens.Black;
                }
                else if (text % 180000 == 0) {
                    txt = string.Format("{0:000}30{1}", Math.Abs(text / 360000), text >= 0 ? "E" : "W");
                    bmpMap.DrawString(txt, font, Brushes.Black, lon, 0);
                    drawingPen = dashPen;
                }
                else {
                    drawingPen = dashPen;
                }

                startPos += step;
                text += step;

                bmpMap.DrawLine(drawingPen, lon, 0, lon, Height);
            }

            // draw airspaces on map
            foreach (AirspaceElement space in AppContents.airspaces) {
                space.Draw(bmpMap, viewArea, Width, Height);
            }

            // draw waypoints on map
            if (currentCatalog != null) {
                currentCatalog.Draw(bmpMap, viewArea, Width, Height);
            }

            // draw flight on map, task "over" map
            if (currentFlight != null) {
                currentFlight.Draw(bmpMap, viewArea, Width, Height);
            }
            // show map
            
            g.DrawImage(mapBmp, 0, 0);
            bmpTotal.DrawImage(mapBmp, 0, 0);

            if (currentTask != null) {
                currentTask.Draw(g, viewArea, Width, Height);
                currentTask.Draw(bmpTotal, viewArea, Width, Height);
            }
            
            // create new brush an pen objects for layered drawing (for floating objects)
            mapBrush = new TextureBrush(mapBmp);
            mapPen = new Pen(mapBrush, 3);

            // create new brush an pen objects for zooming
            totalBrush = new TextureBrush(totalBmp);
            totalPen = new Pen(totalBrush, 3);

            System.GC.Collect();
        }
        
        protected override void OnMouseWheel(MouseEventArgs e) {
            // TODO:  Add Map.OnMouseWheel implementation
            if (e.Delta < 0) {
                this.OnZoomIn();
            }
            else {
                this.OnZoomOut();
            }
        }
    
        protected override void OnMouseMove(MouseEventArgs e) {
            // TODO:  Add Map.OnMouseMove implementation
            base.OnMouseMove (e);

            int lat = (int)((long)e.Y  * (long)viewArea.Height / (long)(Height)) + viewArea.TopLat;
            int lon = (int)((long)e.X * (long)viewArea.Width / (long)(Width)) + viewArea.LeftLon;
            if (lon <= -64800000) {
                lon += 129600000;
            }
            else if (lon > 64800000) {
                lon -= 129600000;
            }
            AppContents.application.UpdatePosInfo(lat, lon);

            if (e.X < 5) {
                if (e.Y < 5) {
                    Cursor = Cursors.PanNW;
                }
                else if (e.Y >= Height - 5) {
                    Cursor = Cursors.PanSW;
                }
                else {
                    Cursor = Cursors.PanWest;
                }
            }
            else if (e.X >= Width - 5) {
                if (e.Y < 5) {
                    Cursor = Cursors.PanNE;
                }
                else if (e.Y >= Height - 5) {
                    Cursor = Cursors.PanSE;
                }
                else {
                    Cursor = Cursors.PanEast;
                }
            }
            else if (e.Y < 5) {
                Cursor = Cursors.PanNorth;
            }
            else if (e.Y >= Height - 5) {
                Cursor = Cursors.PanSouth;
            }
            else {
                if (currentTask != null) {
                    if (currentTask.FindElementNearPos(e.X, e.Y, 3) != -1 ||
                        (currentCatalog != null && currentCatalog.FindElementNearPos(e.X, e.Y, 3) != -1)) {
                        Cursor = Cursors.Hand;
                    }
                    else {
                        Cursor = Cursors.Default;
                    }
                }
                else {
                    Cursor = Cursors.Default;
                }
            }

            if (draggingArea) {
                Graphics g = CreateGraphics();
                if (moving) {
                    // erase old carret
                    g.DrawRectangle(totalPen, oldX > startX ? startX : oldX, oldY > startY ? startY : oldY,
                        Math.Abs(oldX - startX), Math.Abs(oldY - startY));
                }
                else {
                    moving = true;
                }

                g.DrawRectangle(zoomPen, e.X > startX ? startX : e.X, e.Y > startY ? startY : e.Y,
                    Math.Abs(e.X - startX), Math.Abs(e.Y - startY));
                oldX = e.X;
                oldY = e.Y;
            }
            else if (draggingTask) {
                Graphics g = CreateGraphics();
                // erase old task
                int idx;
                currentTask.Draw(g, mapPen, mapBrush, viewArea, Width, Height);
                if ((idx = currentTask.FindElementNearPos(e.X, e.Y, 3)) != -1) {
                    // look for task points
                    currentTask[draggingTaskPointIdx] = currentTask[idx];
                }
                else if (currentCatalog != null && (idx = currentCatalog.FindElementNearPos(e.X, e.Y, 3)) != -1) {
                    // look for waypoints
                    currentTask[draggingTaskPointIdx] = (WayPoint)currentCatalog.VisibleWaypoints[idx];
                }
                else {
                    // insert temp point
                    dummyPoint.Latitude = lat;
                    dummyPoint.Longitude = lon;
                    currentTask[draggingTaskPointIdx] = dummyPoint;
                }
                currentTask.Draw(g, viewArea, Width, Height);
                moving = true;
            }
        }

        protected override void OnMouseDown(MouseEventArgs e) {
            base.OnMouseDown(e);
            startX = e.X;
            startY = e.Y;
            if (e.Button == MouseButtons.Left) {
                if (e.X < 5) {
                    if (e.Y < 5) {
                        OnMoveNW();
                    }
                    else if (e.Y >= Height - 5) {
                        OnMoveSW();
                    }
                    else {
                        OnMoveWest();
                    }
                }
                else if (e.X >= Width - 5) {
                    if (e.Y < 5) {
                        OnMoveNE();
                    }
                    else if (e.Y >= Height - 5) {
                        OnMoveSE();
                    }
                    else {
                        OnMoveEast();
                    }
                }
                else if (e.Y < 5) {
                    OnMoveNorth();
                }
                else if (e.Y >= Height - 5) {
                    OnMoveSouth();
                }
                else if (currentTask != null) {
                    draggingTaskPointIdx = currentTask.FindElementNearPos(e.X, e.Y, 3);
                    if (taskAction == TaskActions.AddPoint) {
                        Graphics g = CreateGraphics();
                        WayPoint w;

                        int lat = (e.Y  * viewArea.Height / (Height)) + viewArea.TopLat;
                        int lon = (e.X * viewArea.Width / (Width)) + viewArea.LeftLon;
                        int idx;
                        if (lon <= -64800000) {
                            lon += 129600000;
                        }
                        else if (lon > 64800000) {
                            lon -= 129600000;
                        }

                        // erase old task
                        currentTask.Draw(g, mapPen, mapBrush, viewArea, Width, Height);

                        if ((idx = currentTask.FindElementNearPos(e.X, e.Y, 3)) != -1) {
                            // look for task points
                            w = currentTask[idx];
                        }
                        else if (currentCatalog != null && (idx = currentCatalog.FindElementNearPos(e.X, e.Y, 3)) != -1) {
                            // look for waypoints
                            w = (WayPoint)currentCatalog.VisibleWaypoints[idx];
                        }
                        else {
                            // insert temp point
                            dummyPoint.Latitude = lat;
                            dummyPoint.Longitude = lon;
                            w = dummyPoint;
                        }

                        if (currentTask.Count == 0) {
                            // replace dummy point , create new one
                            currentTask.Insert(0, w != dummyPoint ? w : new WayPoint(w.Latitude, w.Longitude, "WP 1")/*, false*/);
                            draggingTaskPointIdx++;
                        }
                        currentTask.Insert(++draggingTaskPointIdx, w/*, false*/);
                        currentTask.Draw(g, viewArea, Width, Height);
                        AppContents.application.Update(currentTask);
                    }
                    draggingTask = draggingTaskPointIdx != -1;
                    draggingArea = !draggingTask;

                    moving = false;
                }
                else {
                    draggingArea = true;
                    moving = false;
                }
            }
        }

        protected override void OnMouseUp(MouseEventArgs e) {
            // TODO:  Add Map.OnMouseUp implementation
            base.OnMouseUp(e);
            if (draggingArea && moving) {
                // erase old carret
                Graphics g = CreateGraphics();

                g.DrawRectangle(totalPen, oldX > startX ? startX : oldX, oldY > startY ? startY : oldY,
                    Math.Abs(oldX - startX), Math.Abs(oldY - startY));

                if (Math.Abs(startX - oldX) > 5 && Math.Abs(startY - oldY) > 5) {
                    long startLat = viewArea.TopLat;
                    long startLon = viewArea.LeftLon;
                    long startWidth = viewArea.Width;
                    long startHeight = viewArea.Height;

                    viewArea.TopLat = (int)(((oldY > startY ? startY : oldY) * startHeight / (Height)) + startLat);
                    viewArea.LeftLon = (int)(((oldX > startX ? startX : oldX) * startWidth / (Width)) + startLon);
                    viewArea.BottomLat = (int)(((oldY > startY ? oldY : startY) * startHeight / (Height)) + startLat);
                    viewArea.RightLon = (int)(((oldX > startX ? oldX : startX) * startWidth / (Width)) + startLon);

                    RedrawMap();
                }
            }
            else if (draggingTask && draggingTaskPointIdx != -1) {
                if (taskAction == TaskActions.RemovePoint) {
                    Graphics g = CreateGraphics();
                    if (draggingTaskPointIdx != -1) {
                        currentTask.Draw(g, mapPen, mapBrush, viewArea, Width, Height);
                        currentTask.Delete(draggingTaskPointIdx);
                        currentTask.Draw(g, viewArea, Width, Height);
                        AppContents.application.Update(currentTask);
                    }
                }
                else if (currentTask[draggingTaskPointIdx] == dummyPoint) {
                    if (moving) {
                        // replace dummy point , create new one
                        currentTask[draggingTaskPointIdx] = new WayPoint(dummyPoint.Latitude, dummyPoint.Longitude, string.Format("WP {0}", draggingTaskPointIdx + 1));
                    }
                    else {
                        //remove dummy point
                        currentTask.Delete(draggingTaskPointIdx);
                    }
                }
                AppContents.application.Update(currentTask);
            }
            draggingTask = false;
            draggingArea = false;
            moving = false;
        }

            
        protected override void OnKeyDown(KeyEventArgs e) {
            // TODO:  Add Map.OnKeyDown implementation
            //base.OnKeyDown (e);
            switch (e.KeyCode) {
            case Keys.ControlKey:
                taskAction = TaskActions.RemovePoint;
                break;
            case Keys.ShiftKey:
                taskAction = TaskActions.AddPoint;
                break;
            case Keys.PageUp:
                OnZoomOut();
                break;
            case Keys.PageDown:
                OnZoomIn();
                break;
            }
        }
    
        protected override void OnKeyUp(KeyEventArgs e) {
            // TODO:  Add Map.OnKeyUp implementation
            //base.OnKeyUp (e);
            if (e.KeyCode == Keys.ControlKey || e.KeyCode == Keys.ShiftKey) {
                taskAction = TaskActions.MovePoint;
            }
        }

        protected override void OnMouseEnter(EventArgs e) {
            // TODO:  Add Map.OnMouseEnter implementation
            base.OnMouseEnter (e);
            Focus();
        }

        protected override void OnDoubleClick(EventArgs e) {
            // TODO:  Add Map.OnDoubleClick implementation
            int pos;
            if (currentCatalog != null && (pos = currentCatalog.FindElementNearPos(PointToClient(MousePosition), 3)) != -1) {
                AppContents.application.EditWaypoint((WayPoint)currentCatalog.VisibleWaypoints[pos]);
            }        

            base.OnDoubleClick (e);
        }
        #endregion

        #region Public Functions
        public void RedrawMap() {
            PaintEventArgs e = new PaintEventArgs(CreateGraphics(), new Rectangle());
            OnPaint(e);
        }

        public void OnZoomIn() {
            int widthExt = viewArea.Width / 16;
            int heightExt = viewArea.Height / 16;

            viewArea.LeftLon += widthExt;
            viewArea.RightLon -= widthExt;

            viewArea.TopLat += heightExt;
            viewArea.BottomLat -= heightExt;

            RedrawMap();
        }

        public void OnZoomOut() {
            int widthExt = viewArea.Width / 8;
            int heightExt = viewArea.Height / 8;
            if (viewArea.Width < 32400000 && viewArea.Height < 32400000 &&
               (viewArea.BottomLat + heightExt) < 32400000 && (viewArea.TopLat - heightExt) > -32400000) {
                    viewArea.LeftLon -= widthExt;
                    viewArea.RightLon += widthExt;
                    viewArea.TopLat -= heightExt;
                    viewArea.BottomLat += heightExt;

                RedrawMap();
            }
        }

        public void OnMoveWest() {
            int step = viewArea.Width / 5;
            viewArea.LeftLon -= step;
            viewArea.RightLon -= step;
            RedrawMap();
        }

        public void OnMoveEast() {
            int step = viewArea.Width / 5;
            viewArea.LeftLon += step;
            viewArea.RightLon += step;
            RedrawMap();
        }

        public void OnMoveNorth() {
            int step = viewArea.Height / 5;
            if (viewArea.TopLat - step > -32400000) {
                viewArea.TopLat -= step;
                viewArea.BottomLat -= step;
                RedrawMap();
            }
        }

        public void OnMoveSouth() {
            int step = viewArea.Height / 5;
            if (viewArea.BottomLat + step < 32400000) {
                viewArea.TopLat += step;
                viewArea.BottomLat += step;
                RedrawMap();
            }
        }

        public void OnMoveNW() {
            int step = viewArea.Height / 5;
            if (viewArea.TopLat - step > -32400000) {
                viewArea.TopLat -= step;
                viewArea.BottomLat -= step;

                step = viewArea.Width / 5;
                viewArea.LeftLon -= step;
                viewArea.RightLon -= step;

                RedrawMap();
            }
        }

        public void OnMoveNE() {
            int step = viewArea.Height / 5;
            if (viewArea.TopLat - step > -32400000) {
                viewArea.TopLat -= step;
                viewArea.BottomLat -= step;

                step = viewArea.Width / 5;
                viewArea.LeftLon += step;
                viewArea.RightLon += step;

                RedrawMap();
            }
        }

        public void OnMoveSW() {
            int step = viewArea.Height / 5;
            if (viewArea.BottomLat + step < 32400000) {
                viewArea.TopLat += step;
                viewArea.BottomLat += step;

                step = viewArea.Width / 5;
                viewArea.LeftLon -= step;
                viewArea.RightLon -= step;

                RedrawMap();
            }
        }

        public void OnMoveSE() {
            int step = viewArea.Height / 5;
            if (viewArea.BottomLat + step < 32400000) {
                viewArea.TopLat += step;
                viewArea.BottomLat += step;

                step = viewArea.Width / 5;
                viewArea.LeftLon += step;
                viewArea.RightLon += step;

                RedrawMap();
            }
        }

        public void CenterTo(WayPoint wp) {
            CenterTo(wp.Latitude, wp.Longitude);
        }

        public void CenterTo(int lat, int lon) {
            int w = viewArea.Width;
            int h = viewArea.Height;
            viewArea.LeftLon = lon - (w / 2);
            //viewArea.RightLon = lon + (w / 2);
            viewArea.RightLon = viewArea.LeftLon + w;
            if (lat - (h / 2) > -32400000 && lat + (h / 2) < 32400000) {
                viewArea.TopLat = lat - (h / 2);
                viewArea.BottomLat = viewArea.TopLat + h;
            }
            RedrawMap();
        }

        public void OnCenterHome() {
            CenterTo(AppContents.homePoint);
        }

        public void OnCenterTask() {
            if (currentFlight != null) {
                currentFlight.CalcTaskArea();
                if (!currentFlight.currentTask.Area.IsEmpty) {
                    //viewArea.TopLeft = currentFlight.currentTask.Area.TopLeft;
                    //viewArea.BottomRight= currentFlight.currentTask.Area.BottomRight;
                    viewArea.Area = currentFlight.currentTask.Area;
                    RedrawMap();
                }
                else {
                    OnCenterFlight();
                }
            }
            else if (currentTask != null) {
                currentTask.CalcArea();
                if (!currentTask.Area.IsEmpty) {
                    //viewArea.TopLeft = currentTask.Area.TopLeft;
                    //viewArea.BottomRight = currentTask.Area.BottomRight;
                    viewArea.Area = currentTask.Area;
                    RedrawMap();
                }
            }
        }

        public void OnCenterFlight() {
            if (currentFlight != null) {
                currentFlight.CalcFlightArea();
                if (!currentFlight.Area.IsEmpty) {
                    viewArea.TopLeft = currentFlight.Area.TopLeft;
                    viewArea.BottomRight= currentFlight.Area.BottomRight;
                    RedrawMap();
                }
            }
        }

        public void Switch(WaypointCatalog c) {
            if (currentCatalog != c) {
                currentCatalog = c;
                RedrawMap();
            }
        }

        public void Update(WaypointCatalog c) {
            if (c == currentCatalog) {
                RedrawMap();
            }
        }

        public void Switch(Flight f) {
            if (currentFlight != f) {
                currentFlight = f;
                currentTask = null;
                if (currentFlight != null) {
                    viewArea.Clear();
                    //viewArea.TopLeft = currentFlight.Area.TopLeft;
                    //viewArea.BottomRight = currentFlight.Area.BottomRight;
                    viewArea.Area = currentFlight.Area;
                }
                RedrawMap();
            }
        }

        public void Switch(Task k) {
            if (currentTask != k) {
                currentTask = k;
                currentFlight = null;
                if (currentTask != null && !currentTask.Area.IsEmpty) {
                    //viewArea.TopLeft = currentTask.Area.TopLeft;
                    //viewArea.BottomRight= currentTask.Area.BottomRight;
                    viewArea.Area = currentTask.Area;
                }
                RedrawMap();
            }
        }

        public void HideToolTip() {
            if (toolTip.Visible) {
                toolTip.Hide();
            }
        }

        public void SetArea(int topLat, int leftLon, int bottomLat, int rightLon) {
            viewArea.TopLat = topLat;
            viewArea.LeftLon = leftLon;
            viewArea.BottomLat = bottomLat;
            viewArea.RightLon = rightLon;
        }

        #endregion
        
        #region Private Functions
        private void SizeArea() {
            // calculate correct aspect ratio
            double mapFactor = Math.Cos(WgsPoint.InternalToRad(viewArea.Center.Latitude));
            double screenFactor = (double)Width / (double)Height;

            int reqWidth = (int)(viewArea.Height / mapFactor * screenFactor);

            if (viewArea.Width < reqWidth) {
                reqWidth = (reqWidth - viewArea.Width) / 2;
                // screen has sapce for more lon 
                viewArea.LeftLon -= reqWidth;
                viewArea.RightLon += reqWidth;
            }
            else if (viewArea.Width > reqWidth) {
                // we need more space for lat
                int reqHeight = (int)(viewArea.Width * mapFactor / screenFactor);
                reqHeight = (reqHeight - viewArea.Height) / 2;
                if (viewArea.TopLat - reqHeight > -32400000 && viewArea.BottomLat + reqHeight < 32400000) {
                    viewArea.TopLat -= reqHeight;
                    viewArea.BottomLat += reqHeight;
                }
            }
        }

        private void OnCreateWaypoint(object sender, System.EventArgs e) {
            int lat = (int)(((long)startY * (long)viewArea.Height / (long)Height)) + viewArea.TopLat;
            int lon = (int)(((long)startX * (long)viewArea.Width / (long)Width)) + viewArea.LeftLon;
            if (lon <= -64800000) {
                lon += 129600000;
            }
            else if (lon > 64800000) {
                lon -= 129600000;
            }
        
            AppContents.application.AddWaypoint(lat, lon);
        }

        private void OnCreateTask(object sender, System.EventArgs e) {
            AppContents.application.NewTask();        
        }

        private void OnCenterMap(object sender, System.EventArgs e) {
            long w = viewArea.Width;
            long h = viewArea.Height;
            long lat = ((long)startY * h / this.Height) + viewArea.TopLat;
            long lon = ((long)startX * w / this.Width) + viewArea.LeftLon;
            
            if (lon > 64800000) {
                lon -= 129600000;
            }
            else if (lon < -64800000) {
                lon += 129600000;
            }
            CenterTo((int)lat, (int)lon);
        }

        private void OnCenterHome(object sender, System.EventArgs e) {
            OnCenterHome();
        }


        private void OnCenterTask(object sender, System.EventArgs e) {
            OnCenterTask();
        }

        private void OnCenterFlight(object sender, System.EventArgs e) {
            OnCenterFlight();
        }

        private void OnOpeningContextMenu(object sender, System.ComponentModel.CancelEventArgs e) {
            newWaypointToolStripMenuItem.Enabled = currentCatalog != null;
            popupWindowPos = PointToClient(MousePosition);
            editWaypointToolStripMenuItem.Enabled = (currentCatalog != null && currentCatalog.FindElementNearPos(popupWindowPos, 3) != -1);
            centerMapToflightToolStripMenuItem.Enabled = currentFlight != null;
            centerMapTotaskToolStripMenuItem.Enabled = (currentFlight != null || currentTask != null);
        }

        private void OnEditWaypoint(object sender, System.EventArgs e) {
            int pos;
            if (currentCatalog != null && !popupWindowPos.IsEmpty && (pos = currentCatalog.FindElementNearPos(popupWindowPos, 3)) != -1) {
                AppContents.application.EditWaypoint((WayPoint)currentCatalog.VisibleWaypoints[pos]);
            }
        }

        private void OnAirspaceInfo(object sender, System.EventArgs e) {
            Point p = PointToScreen(popupWindowPos);
            if (AppContents.airspaces != null) {
                foreach (AirspaceElement space in AppContents.airspaces) {
                    if (space.region != null && space.region.IsVisible(popupWindowPos)) {
                        toolTip.Add(space.ToString());
                    }
                }

                if (popupWindowPos.X + toolTip.Width > Width) {
                    toolTip.Left = p.X - toolTip.Width;
                }
                else {
                    toolTip.Left = p.X;
                }

                if (popupWindowPos.Y + toolTip.Height > Height) {
                    toolTip.Top = p.Y - toolTip.Height;
                }
                else {
                    toolTip.Top = p.Y;
                }
                toolTip.Show();
            }
        }

        #endregion

        #region Attributes
        public WgsPoint Center {
            get {
                return viewArea.Center;
            }
        }

        public Point UpperLeft {
            get {
                return viewArea.TopLeft;
            }
        }

        public Point LowerRight {
            get {
                return viewArea.BottomRight;
            }
        }

        #endregion    
    }
}
