// name   : OLCDialog.cs
// author : Harald Maier
// date   : 14.05.2004
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.IO;
using System.Diagnostics;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using System.Collections;
using System.Collections.Specialized;

using SoaringDotNET.Data;
using SoaringDotNET.GUI;
using SoaringDotNET.Optimization;

namespace SoaringDotNET.Dialogs
{
	/// <summary>
	/// 
	/// </summary>
	public class OLCDialog : System.Windows.Forms.Form
	{
        private System.Windows.Forms.Button buttonSelectPilot2;
        private System.Windows.Forms.Button buttonSelectPilot1;
        private System.Windows.Forms.TextBox textBoxCompId;
        private System.Windows.Forms.Label label13;
        private System.Windows.Forms.TextBox textBoxGliderId;
        private System.Windows.Forms.ComboBox comboBoxCompClass;
        private System.Windows.Forms.Label label12;
        private System.Windows.Forms.Label label11;
        private System.Windows.Forms.ComboBox comboBoxGlider;
        private System.Windows.Forms.Label label10;
        private System.Windows.Forms.Label label9;
        private System.Windows.Forms.ColumnHeader columnTurnpoint;
        private System.Windows.Forms.ColumnHeader columnTime;
        private System.Windows.Forms.ColumnHeader columnLat;
        private System.Windows.Forms.ColumnHeader columnLon;
        private System.Windows.Forms.ColumnHeader columnDist;
        private System.Windows.Forms.ColumnHeader columnTaskDist;
        private System.Windows.Forms.ColumnHeader columnPoints;
        private System.Windows.Forms.Button buttonSend;
        private System.Windows.Forms.Button buttonSave;
        private System.Windows.Forms.Button buttonFinish;
        private System.Windows.Forms.ListView listViewTask;
        private System.Windows.Forms.Label label1;
        private System.Windows.Forms.Label label2;
        private System.Windows.Forms.DateTimePicker dateTimePickerPilot1;
        private System.Windows.Forms.CheckBox checkPureGlider;
        private System.Windows.Forms.DateTimePicker dateTimePickerPilot2;
        private System.Windows.Forms.Label label8;
        private System.Windows.Forms.TextBox textCoPilotPreName;
        private System.Windows.Forms.TextBox textPilotPreName;
        private System.Windows.Forms.TextBox textCoPilotSurName;
        private System.Windows.Forms.TextBox textPilotSurName;
        private System.Windows.Forms.Label label3;
        private System.Windows.Forms.Label label4;
        private System.Windows.Forms.Label label5;
        private System.Windows.Forms.ComboBox comboBoxCompetitionCountry;
        private System.Windows.Forms.ColumnHeader columnElevation;
        private System.Windows.Forms.Label label6;
        private System.Windows.Forms.ComboBox comboBoxDeparture;
        private GroupBox groupBox1;
        private RadioButton radioButtonShowDepartureAll;
        private RadioButton radioButtonShowDeparture50km;

        private Flight flight;

        private void InitializeComponent() {
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(OLCDialog));
            this.buttonSelectPilot2 = new System.Windows.Forms.Button();
            this.buttonSelectPilot1 = new System.Windows.Forms.Button();
            this.textBoxCompId = new System.Windows.Forms.TextBox();
            this.label13 = new System.Windows.Forms.Label();
            this.textBoxGliderId = new System.Windows.Forms.TextBox();
            this.comboBoxCompClass = new System.Windows.Forms.ComboBox();
            this.label12 = new System.Windows.Forms.Label();
            this.label11 = new System.Windows.Forms.Label();
            this.comboBoxGlider = new System.Windows.Forms.ComboBox();
            this.label10 = new System.Windows.Forms.Label();
            this.label9 = new System.Windows.Forms.Label();
            this.textCoPilotPreName = new System.Windows.Forms.TextBox();
            this.textPilotPreName = new System.Windows.Forms.TextBox();
            this.label8 = new System.Windows.Forms.Label();
            this.listViewTask = new System.Windows.Forms.ListView();
            this.columnTurnpoint = new System.Windows.Forms.ColumnHeader();
            this.columnTime = new System.Windows.Forms.ColumnHeader();
            this.columnLat = new System.Windows.Forms.ColumnHeader();
            this.columnLon = new System.Windows.Forms.ColumnHeader();
            this.columnElevation = new System.Windows.Forms.ColumnHeader();
            this.columnDist = new System.Windows.Forms.ColumnHeader();
            this.columnTaskDist = new System.Windows.Forms.ColumnHeader();
            this.columnPoints = new System.Windows.Forms.ColumnHeader();
            this.buttonSend = new System.Windows.Forms.Button();
            this.buttonSave = new System.Windows.Forms.Button();
            this.buttonFinish = new System.Windows.Forms.Button();
            this.label1 = new System.Windows.Forms.Label();
            this.label2 = new System.Windows.Forms.Label();
            this.dateTimePickerPilot1 = new System.Windows.Forms.DateTimePicker();
            this.dateTimePickerPilot2 = new System.Windows.Forms.DateTimePicker();
            this.checkPureGlider = new System.Windows.Forms.CheckBox();
            this.textCoPilotSurName = new System.Windows.Forms.TextBox();
            this.textPilotSurName = new System.Windows.Forms.TextBox();
            this.label3 = new System.Windows.Forms.Label();
            this.label4 = new System.Windows.Forms.Label();
            this.label5 = new System.Windows.Forms.Label();
            this.comboBoxCompetitionCountry = new System.Windows.Forms.ComboBox();
            this.label6 = new System.Windows.Forms.Label();
            this.comboBoxDeparture = new System.Windows.Forms.ComboBox();
            this.groupBox1 = new System.Windows.Forms.GroupBox();
            this.radioButtonShowDepartureAll = new System.Windows.Forms.RadioButton();
            this.radioButtonShowDeparture50km = new System.Windows.Forms.RadioButton();
            this.groupBox1.SuspendLayout();
            this.SuspendLayout();
            // 
            // buttonSelectPilot2
            // 
            this.buttonSelectPilot2.Image = ((System.Drawing.Image)(resources.GetObject("buttonSelectPilot2.Image")));
            this.buttonSelectPilot2.Location = new System.Drawing.Point(328, 247);
            this.buttonSelectPilot2.Name = "buttonSelectPilot2";
            this.buttonSelectPilot2.Size = new System.Drawing.Size(23, 23);
            this.buttonSelectPilot2.TabIndex = 35;
            this.buttonSelectPilot2.Click += new System.EventHandler(this.OnSelectPilot2);
            // 
            // buttonSelectPilot1
            // 
            this.buttonSelectPilot1.Image = ((System.Drawing.Image)(resources.GetObject("buttonSelectPilot1.Image")));
            this.buttonSelectPilot1.Location = new System.Drawing.Point(328, 215);
            this.buttonSelectPilot1.Name = "buttonSelectPilot1";
            this.buttonSelectPilot1.Size = new System.Drawing.Size(23, 23);
            this.buttonSelectPilot1.TabIndex = 34;
            this.buttonSelectPilot1.Click += new System.EventHandler(this.OnSelectPilot1);
            // 
            // textBoxCompId
            // 
            this.textBoxCompId.CharacterCasing = System.Windows.Forms.CharacterCasing.Upper;
            this.textBoxCompId.Location = new System.Drawing.Point(420, 312);
            this.textBoxCompId.Name = "textBoxCompId";
            this.textBoxCompId.Size = new System.Drawing.Size(32, 20);
            this.textBoxCompId.TabIndex = 33;
            // 
            // label13
            // 
            this.label13.AutoSize = true;
            this.label13.Location = new System.Drawing.Point(365, 314);
            this.label13.Name = "label13";
            this.label13.Size = new System.Drawing.Size(51, 13);
            this.label13.TabIndex = 32;
            this.label13.Text = "Comp. ID";
            // 
            // textBoxGliderId
            // 
            this.textBoxGliderId.CharacterCasing = System.Windows.Forms.CharacterCasing.Upper;
            this.textBoxGliderId.Location = new System.Drawing.Point(420, 280);
            this.textBoxGliderId.Name = "textBoxGliderId";
            this.textBoxGliderId.Size = new System.Drawing.Size(90, 20);
            this.textBoxGliderId.TabIndex = 31;
            // 
            // comboBoxCompClass
            // 
            this.comboBoxCompClass.Location = new System.Drawing.Point(112, 312);
            this.comboBoxCompClass.Name = "comboBoxCompClass";
            this.comboBoxCompClass.Size = new System.Drawing.Size(152, 21);
            this.comboBoxCompClass.Sorted = true;
            this.comboBoxCompClass.TabIndex = 30;
            // 
            // label12
            // 
            this.label12.AutoSize = true;
            this.label12.Location = new System.Drawing.Point(43, 314);
            this.label12.Name = "label12";
            this.label12.Size = new System.Drawing.Size(65, 13);
            this.label12.TabIndex = 29;
            this.label12.Text = "Comp. Class";
            // 
            // label11
            // 
            this.label11.AutoSize = true;
            this.label11.Location = new System.Drawing.Point(368, 282);
            this.label11.Name = "label11";
            this.label11.Size = new System.Drawing.Size(48, 13);
            this.label11.TabIndex = 28;
            this.label11.Text = "Glider ID";
            // 
            // comboBoxGlider
            // 
            this.comboBoxGlider.Location = new System.Drawing.Point(112, 280);
            this.comboBoxGlider.Name = "comboBoxGlider";
            this.comboBoxGlider.Size = new System.Drawing.Size(152, 21);
            this.comboBoxGlider.Sorted = true;
            this.comboBoxGlider.TabIndex = 27;
            // 
            // label10
            // 
            this.label10.AutoSize = true;
            this.label10.Location = new System.Drawing.Point(74, 282);
            this.label10.Name = "label10";
            this.label10.Size = new System.Drawing.Size(34, 13);
            this.label10.TabIndex = 26;
            this.label10.Text = "Glider";
            // 
            // label9
            // 
            this.label9.AutoSize = true;
            this.label9.Location = new System.Drawing.Point(66, 250);
            this.label9.Name = "label9";
            this.label9.Size = new System.Drawing.Size(42, 13);
            this.label9.TabIndex = 25;
            this.label9.Text = "Co-pilot";
            // 
            // textCoPilotPreName
            // 
            this.textCoPilotPreName.Location = new System.Drawing.Point(112, 248);
            this.textCoPilotPreName.Name = "textCoPilotPreName";
            this.textCoPilotPreName.Size = new System.Drawing.Size(104, 20);
            this.textCoPilotPreName.TabIndex = 24;
            // 
            // textPilotPreName
            // 
            this.textPilotPreName.Location = new System.Drawing.Point(112, 216);
            this.textPilotPreName.Name = "textPilotPreName";
            this.textPilotPreName.Size = new System.Drawing.Size(104, 20);
            this.textPilotPreName.TabIndex = 23;
            // 
            // label8
            // 
            this.label8.AutoSize = true;
            this.label8.Location = new System.Drawing.Point(81, 218);
            this.label8.Name = "label8";
            this.label8.Size = new System.Drawing.Size(27, 13);
            this.label8.TabIndex = 22;
            this.label8.Text = "Pilot";
            // 
            // listViewTask
            // 
            this.listViewTask.Columns.AddRange(new System.Windows.Forms.ColumnHeader[] {
            this.columnTurnpoint,
            this.columnTime,
            this.columnLat,
            this.columnLon,
            this.columnElevation,
            this.columnDist,
            this.columnTaskDist,
            this.columnPoints});
            this.listViewTask.Location = new System.Drawing.Point(16, 16);
            this.listViewTask.Name = "listViewTask";
            this.listViewTask.Size = new System.Drawing.Size(576, 168);
            this.listViewTask.TabIndex = 36;
            this.listViewTask.UseCompatibleStateImageBehavior = false;
            this.listViewTask.View = System.Windows.Forms.View.Details;
            // 
            // columnTurnpoint
            // 
            this.columnTurnpoint.Text = "Turnpoint";
            this.columnTurnpoint.Width = 80;
            // 
            // columnTime
            // 
            this.columnTime.Text = "Time";
            // 
            // columnLat
            // 
            this.columnLat.Text = "Latitude";
            this.columnLat.Width = 80;
            // 
            // columnLon
            // 
            this.columnLon.Text = "Longitude";
            this.columnLon.Width = 80;
            // 
            // columnElevation
            // 
            this.columnElevation.Text = "Elevation";
            this.columnElevation.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            // 
            // columnDist
            // 
            this.columnDist.Text = "Distance";
            this.columnDist.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            this.columnDist.Width = 66;
            // 
            // columnTaskDist
            // 
            this.columnTaskDist.Text = "Task dist.";
            this.columnTaskDist.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            this.columnTaskDist.Width = 66;
            // 
            // columnPoints
            // 
            this.columnPoints.Text = "Points";
            this.columnPoints.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            // 
            // buttonSend
            // 
            this.buttonSend.Location = new System.Drawing.Point(145, 458);
            this.buttonSend.Name = "buttonSend";
            this.buttonSend.Size = new System.Drawing.Size(80, 23);
            this.buttonSend.TabIndex = 37;
            this.buttonSend.Text = "Send to OLC";
            this.buttonSend.Click += new System.EventHandler(this.OnSend);
            // 
            // buttonSave
            // 
            this.buttonSave.Location = new System.Drawing.Point(245, 458);
            this.buttonSave.Name = "buttonSave";
            this.buttonSave.Size = new System.Drawing.Size(87, 23);
            this.buttonSave.TabIndex = 38;
            this.buttonSave.Text = "Save OLC file";
            this.buttonSave.Click += new System.EventHandler(this.OnSaveOLC);
            // 
            // buttonFinish
            // 
            this.buttonFinish.DialogResult = System.Windows.Forms.DialogResult.OK;
            this.buttonFinish.Location = new System.Drawing.Point(352, 458);
            this.buttonFinish.Name = "buttonFinish";
            this.buttonFinish.Size = new System.Drawing.Size(80, 23);
            this.buttonFinish.TabIndex = 39;
            this.buttonFinish.Text = "Finish";
            // 
            // label1
            // 
            this.label1.AutoSize = true;
            this.label1.Location = new System.Drawing.Point(371, 218);
            this.label1.Name = "label1";
            this.label1.Size = new System.Drawing.Size(45, 13);
            this.label1.TabIndex = 40;
            this.label1.Text = "Birthday";
            // 
            // label2
            // 
            this.label2.AutoSize = true;
            this.label2.Location = new System.Drawing.Point(371, 250);
            this.label2.Name = "label2";
            this.label2.Size = new System.Drawing.Size(45, 13);
            this.label2.TabIndex = 41;
            this.label2.Text = "Birthday";
            // 
            // dateTimePickerPilot1
            // 
            this.dateTimePickerPilot1.Format = System.Windows.Forms.DateTimePickerFormat.Short;
            this.dateTimePickerPilot1.Location = new System.Drawing.Point(420, 216);
            this.dateTimePickerPilot1.Name = "dateTimePickerPilot1";
            this.dateTimePickerPilot1.Size = new System.Drawing.Size(90, 20);
            this.dateTimePickerPilot1.TabIndex = 42;
            // 
            // dateTimePickerPilot2
            // 
            this.dateTimePickerPilot2.Format = System.Windows.Forms.DateTimePickerFormat.Short;
            this.dateTimePickerPilot2.Location = new System.Drawing.Point(420, 248);
            this.dateTimePickerPilot2.Name = "dateTimePickerPilot2";
            this.dateTimePickerPilot2.Size = new System.Drawing.Size(90, 20);
            this.dateTimePickerPilot2.TabIndex = 43;
            // 
            // checkPureGlider
            // 
            this.checkPureGlider.Location = new System.Drawing.Point(112, 344);
            this.checkPureGlider.Name = "checkPureGlider";
            this.checkPureGlider.Size = new System.Drawing.Size(104, 24);
            this.checkPureGlider.TabIndex = 44;
            this.checkPureGlider.Text = "pure glider";
            // 
            // textCoPilotSurName
            // 
            this.textCoPilotSurName.Location = new System.Drawing.Point(224, 248);
            this.textCoPilotSurName.Name = "textCoPilotSurName";
            this.textCoPilotSurName.Size = new System.Drawing.Size(96, 20);
            this.textCoPilotSurName.TabIndex = 46;
            // 
            // textPilotSurName
            // 
            this.textPilotSurName.Location = new System.Drawing.Point(224, 216);
            this.textPilotSurName.Name = "textPilotSurName";
            this.textPilotSurName.Size = new System.Drawing.Size(96, 20);
            this.textPilotSurName.TabIndex = 45;
            // 
            // label3
            // 
            this.label3.AutoSize = true;
            this.label3.Location = new System.Drawing.Point(110, 200);
            this.label3.Name = "label3";
            this.label3.Size = new System.Drawing.Size(49, 13);
            this.label3.TabIndex = 47;
            this.label3.Text = "Prename";
            // 
            // label4
            // 
            this.label4.AutoSize = true;
            this.label4.Location = new System.Drawing.Point(222, 200);
            this.label4.Name = "label4";
            this.label4.Size = new System.Drawing.Size(49, 13);
            this.label4.TabIndex = 48;
            this.label4.Text = "Surname";
            // 
            // label5
            // 
            this.label5.AutoSize = true;
            this.label5.Location = new System.Drawing.Point(46, 378);
            this.label5.Name = "label5";
            this.label5.Size = new System.Drawing.Size(62, 13);
            this.label5.TabIndex = 49;
            this.label5.Text = "Competition";
            // 
            // comboBoxCompetitionCountry
            // 
            this.comboBoxCompetitionCountry.Location = new System.Drawing.Point(112, 376);
            this.comboBoxCompetitionCountry.Name = "comboBoxCompetitionCountry";
            this.comboBoxCompetitionCountry.Size = new System.Drawing.Size(173, 21);
            this.comboBoxCompetitionCountry.Sorted = true;
            this.comboBoxCompetitionCountry.TabIndex = 50;
            this.comboBoxCompetitionCountry.SelectedIndexChanged += new System.EventHandler(this.OnSelectedCompetitionChanged);
            // 
            // label6
            // 
            this.label6.AutoSize = true;
            this.label6.Location = new System.Drawing.Point(25, 414);
            this.label6.Name = "label6";
            this.label6.Size = new System.Drawing.Size(83, 13);
            this.label6.TabIndex = 51;
            this.label6.Text = "Departure name";
            // 
            // comboBoxDeparture
            // 
            this.comboBoxDeparture.Location = new System.Drawing.Point(112, 411);
            this.comboBoxDeparture.Name = "comboBoxDeparture";
            this.comboBoxDeparture.Size = new System.Drawing.Size(173, 21);
            this.comboBoxDeparture.TabIndex = 52;
            // 
            // groupBox1
            // 
            this.groupBox1.Controls.Add(this.radioButtonShowDepartureAll);
            this.groupBox1.Controls.Add(this.radioButtonShowDeparture50km);
            this.groupBox1.Location = new System.Drawing.Point(307, 376);
            this.groupBox1.Name = "groupBox1";
            this.groupBox1.Size = new System.Drawing.Size(145, 55);
            this.groupBox1.TabIndex = 53;
            this.groupBox1.TabStop = false;
            this.groupBox1.Text = "Depature Names";
            // 
            // radioButtonShowDepartureAll
            // 
            this.radioButtonShowDepartureAll.AutoSize = true;
            this.radioButtonShowDepartureAll.Location = new System.Drawing.Point(6, 36);
            this.radioButtonShowDepartureAll.Name = "radioButtonShowDepartureAll";
            this.radioButtonShowDepartureAll.Size = new System.Drawing.Size(63, 17);
            this.radioButtonShowDepartureAll.TabIndex = 1;
            this.radioButtonShowDepartureAll.Text = "show all";
            this.radioButtonShowDepartureAll.UseVisualStyleBackColor = true;
            this.radioButtonShowDepartureAll.CheckedChanged += new System.EventHandler(this.OnDepartureOptionsChanged);
            // 
            // radioButtonShowDeparture50km
            // 
            this.radioButtonShowDeparture50km.AutoSize = true;
            this.radioButtonShowDeparture50km.Checked = true;
            this.radioButtonShowDeparture50km.Location = new System.Drawing.Point(6, 18);
            this.radioButtonShowDeparture50km.Name = "radioButtonShowDeparture50km";
            this.radioButtonShowDeparture50km.Size = new System.Drawing.Size(126, 17);
            this.radioButtonShowDeparture50km.TabIndex = 0;
            this.radioButtonShowDeparture50km.TabStop = true;
            this.radioButtonShowDeparture50km.Text = "50 km around takeoff";
            this.radioButtonShowDeparture50km.UseVisualStyleBackColor = true;
            this.radioButtonShowDeparture50km.CheckedChanged += new System.EventHandler(this.OnDepartureOptionsChanged);
            // 
            // OLCDialog
            // 
            this.AcceptButton = this.buttonFinish;
            this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
            this.ClientSize = new System.Drawing.Size(609, 496);
            this.Controls.Add(this.groupBox1);
            this.Controls.Add(this.comboBoxDeparture);
            this.Controls.Add(this.label6);
            this.Controls.Add(this.comboBoxCompetitionCountry);
            this.Controls.Add(this.label5);
            this.Controls.Add(this.label4);
            this.Controls.Add(this.label3);
            this.Controls.Add(this.textCoPilotSurName);
            this.Controls.Add(this.textPilotSurName);
            this.Controls.Add(this.label2);
            this.Controls.Add(this.label1);
            this.Controls.Add(this.textBoxCompId);
            this.Controls.Add(this.label13);
            this.Controls.Add(this.textBoxGliderId);
            this.Controls.Add(this.label12);
            this.Controls.Add(this.label11);
            this.Controls.Add(this.label10);
            this.Controls.Add(this.label9);
            this.Controls.Add(this.textCoPilotPreName);
            this.Controls.Add(this.textPilotPreName);
            this.Controls.Add(this.label8);
            this.Controls.Add(this.checkPureGlider);
            this.Controls.Add(this.dateTimePickerPilot2);
            this.Controls.Add(this.dateTimePickerPilot1);
            this.Controls.Add(this.buttonFinish);
            this.Controls.Add(this.buttonSave);
            this.Controls.Add(this.buttonSend);
            this.Controls.Add(this.listViewTask);
            this.Controls.Add(this.buttonSelectPilot2);
            this.Controls.Add(this.buttonSelectPilot1);
            this.Controls.Add(this.comboBoxCompClass);
            this.Controls.Add(this.comboBoxGlider);
            this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.FixedToolWindow;
            this.MaximizeBox = false;
            this.MinimizeBox = false;
            this.Name = "OLCDialog";
            this.ShowInTaskbar = false;
            this.StartPosition = System.Windows.Forms.FormStartPosition.CenterParent;
            this.Text = "Send to OLC";
            this.groupBox1.ResumeLayout(false);
            this.groupBox1.PerformLayout();
            this.ResumeLayout(false);
            this.PerformLayout();

        }
    
		public OLCDialog(Flight f)
		{
			// 
			// TODO: Add constructor logic here
			//
            InitializeComponent();

            Task t;
            int i;
            WayPoint wp;
            ListViewItem item;
            double sumPoints = 0, length, points = 0;
            string lStringAll, lStringTask, pString = "";
            OptimizationBase optimizer = null;


            this.flight = f;

            foreach (DictionaryEntry entry in AppContents.gliderIndexes) {
                comboBoxGlider.Items.Add(entry.Value);
            }

            comboBoxCompetitionCountry.Items.AddRange(AppContents.compCountries.ToArray());
            comboBoxCompClass.Items.AddRange(AppContents.compClasses);
            comboBoxDeparture.Items.Clear();

            radioButtonShowDeparture50km.Checked = true;
            OnDepartureOptionsChanged(null, null);

            //find nearest takeoff
            double minDist = double.MaxValue;
            double dist;
            SoaringDotNET.Data.OLCAirfield selField = null;
            if (flight.currentTask != null && flight.currentTask.Count > 0) {
                wp = flight.currentTask[0];
                foreach (OLCAirfield olcField in comboBoxDeparture.Items) {
                    dist = olcField.distance(wp);
                    if (dist < minDist) {
                        selField = olcField;
                        minDist = dist;
                    }
                }
            }
            if (selField != null) {
                comboBoxDeparture.SelectedItem = selField;
            }

            t = flight.currentTask;
            if (t != null) {
                // find optimizer
                foreach (OptimizationBase opt in flight.optimizedTasks) {
                    if (opt.OptimizedTask == t) {
                        optimizer = opt;
                        break;
                    }
                }

                for (i = 0; i < t.Count; i++) {
                    wp = t[i];

                    item = new ListViewItem(wp.longName);
                    item.ImageIndex = 0;
                    // ignore takeoff and takeoff -> start
                    length = (i > 0 ? wp.distanceKM(t[i - 1]) : 0);
                    lStringAll = (i > 0 ? string.Format(AppContents.ni, "{0:F1} km", length) : "");
                    if (i > 1 && i < t.Count - 1) {
                        lStringTask = lStringAll;
                    }
                    else {
                        lStringTask = "";
                    }
                    if (t.TaskType != TaskTypes.TriangleStartOnLeg) {
                        // no points from takoff -> start
                        // no points from finish -> landing
                        if (i > 1 && i < t.Count - 1 && optimizer != null) {
                            points = length * 100.0 / flight.GliderIndex * optimizer.GetLegWeight(i - 2);
                            pString = string.Format(AppContents.ni, "{0:F1}", points);
                        }
                        else {
                            points = 0.0;
                            pString = "";
                        }
                    }
                    else {
                        // no points from takoff -> start, start -> TP 1
                        // no points from TP 3 -> finish, finish -> landing
                        if (i > 1 && i < t.Count - 2 && optimizer != null) {
                            if (i == 2) {
                                // first TP, get length to last TP
                                length = wp.distanceKM(t[4]);
                                lStringTask = string.Format(AppContents.ni, "{0:F1} km", length);
                            }
                            points = length * 100.0 / flight.GliderIndex * optimizer.GetLegWeight(i - 2);
                            pString = string.Format(AppContents.ni, "{0:F1}", points);
                        }
                        else {
                            points = 0.0;
                            pString = "";
                            lStringTask = "";
                        }
                    }
                    item.SubItems.AddRange(new string [] {wp.ToStringTime(), wp.ToStringLat(), wp.ToStringLong(), string.Format("{0} m", wp.reachPoint.Elevation), lStringAll, lStringTask, pString});
                    listViewTask.Items.Add(item);
                    sumPoints += points;
                }

                item = new ListViewItem("Total");
                item.ImageIndex = 0;
                item.SubItems.AddRange(new string [] {"", "", "", "", string.Format(AppContents.ni, "{0:F1} km", t.TotalDistanceKM), string.Format(AppContents.ni, "{0:F1} km", t.TaskDistanceKM), string.Format(AppContents.ni, "{0:F1}", sumPoints)});
                listViewTask.Items.Add(item);

                Regex split = new Regex("[ \t]+");
                string []tokens = split.Split(f.Pilot);
                Pilot p = AppContents.FindPilot(tokens.Length > 0 ? tokens[0] : "", tokens.Length > 1 ? tokens[1] : "");
                textPilotPreName.Text = p.FirstName;
                textPilotSurName.Text = p.LastName;
                dateTimePickerPilot1.Value = p.BirthDay;
                p = AppContents.FindPilot(tokens.Length > 2 ? tokens[2] : "", tokens.Length > 3 ? tokens[3] : "");
                textCoPilotPreName.Text = p.FirstName;
                textCoPilotSurName.Text = p.LastName;
                dateTimePickerPilot2.Value = p.BirthDay;

                textBoxCompId.Text = f.CompetitionId;
                textBoxGliderId.Text = f.GliderId;

                string s = f.Glider.ToUpper().Replace(" ", "");
                if (AppContents.gliderIndexes.ContainsKey(s)) {
                    comboBoxGlider.SelectedItem = AppContents.gliderIndexes[s];
                }

                if ((i = comboBoxCompClass.FindString(f.CompetitionClass)) != -1) {
                    comboBoxCompClass.SelectedIndex = i;
                }
            }
		}

        private void OnSelectPilot1(object sender, System.EventArgs e) {
            SelectPilotDialog d = new SelectPilotDialog();
            if (d.ShowDialog(this) == DialogResult.OK) {
                textPilotPreName.Text = d.selectedPilot.FirstName;
                textPilotSurName.Text = d.selectedPilot.LastName;
                dateTimePickerPilot1.Value = d.selectedPilot.BirthDay;
            }
        }

        private void OnSelectPilot2(object sender, System.EventArgs e) {
            SelectPilotDialog d = new SelectPilotDialog();
            if (d.ShowDialog(this) == DialogResult.OK) {
                textCoPilotPreName.Text = d.selectedPilot.FirstName;
                textCoPilotSurName.Text = d.selectedPilot.LastName;
                dateTimePickerPilot2.Value = d.selectedPilot.BirthDay;
            }
        }

        private void OnSend(object sender, System.EventArgs e) {
            byte[] olcResponse;
            Regex refExp = new Regex("response[\t ]*=[\t ]*([A-Za-z0-9]+)");
            Regex userUnknown = new Regex("please[\t ]*identify[\t ]*yourself", RegexOptions.IgnoreCase);
            Regex missingFile = new Regex("no[\t ]*file[\t ]*uploaded", RegexOptions.IgnoreCase);
            string response;
            Match m = null;
            GroupCollection coll;

            if (comboBoxCompetitionCountry.SelectedItem != null) {
                if (comboBoxDeparture.SelectedItem != null) {
                    //string olc = BuildOlcString();
                    //Process p = new Process();
                    //p.StartInfo.FileName = "IEXPLORE.EXE";
                    //p.StartInfo.Arguments = ((CompetitionCountry)comboBoxCompetitionCountry.SelectedItem).URL + "?" + olc;
                    //p.Start();
                    System.Net.WebClient olcClient = new System.Net.WebClient();
                    olcClient.Headers.Add("user-agent", "Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.2;)");
                    olcClient.Headers.Add("Cache-Control", "no-cache");
                    System.Net.WebProxy wp = System.Net.WebProxy.GetDefaultProxy();
                    wp.UseDefaultCredentials = true;

                    olcClient.Proxy = wp;
                    try {
                        Cursor = Cursors.WaitCursor;
                        olcResponse = olcClient.UploadValues(((CompetitionCountry)comboBoxCompetitionCountry.SelectedItem).URL, "POST", GetOlcParameter());
                        response = System.Text.Encoding.ASCII.GetString(olcResponse);
                        m = refExp.Match(response);
                        if (m.Success) {
                            coll = m.Groups;
                            string refNum = coll[1].Value;
                            MessageBox.Show(this, "Upload of " + flight.Name + " to OLC successfull!", "Success", MessageBoxButtons.OK,
                                MessageBoxIcon.Information);
                        }
                        else {
                            string reason = "Unknow error.";
                            if (userUnknown.IsMatch(response)) {
                                reason = string.Format("Unknown competitor.\n\nFirst name: {0}\nLast Name: {1}\nBirthday: {2}",
                                    textPilotPreName.Text, textPilotSurName.Text, dateTimePickerPilot1.Value.ToString("dd.MM.yy"));
                            }
                            else if (missingFile.IsMatch(response)) {
                                reason = string.Format("Missing file.",
                                    textPilotPreName.Text, textPilotSurName.Text, dateTimePickerPilot1.Value.ToString("dd.MM.yy"));
                            }
                            MessageBox.Show(this, string.Format("Upload of {0} to OLC failed!\n\nReason: {1}\n\nPlease correct the error or save your flight to an OLC file and\ntry to upload ist to the server.", flight.Name, reason), "Error", MessageBoxButtons.OK,
                                MessageBoxIcon.Stop);
                            System.Console.WriteLine(response);
                        }
                    }
                    catch (Exception ex) {
                        MessageBox.Show(this, ex.Message);
                    }
                    finally {
                        Cursor = Cursors.Default;
                    }
                }
                else {
                    MessageBox.Show(this, "Please select your departure location!", "Error", MessageBoxButtons.OK, MessageBoxIcon.Stop);
                }
            }
            else {
                MessageBox.Show(this, "Please select your competition country!", "Error", MessageBoxButtons.OK, MessageBoxIcon.Stop);
            }
        }

        private string BuildOlcString() {
            string txt = "";
            string tmp;
            int i;
            long latDeg, lonDeg;
            double latMin, lonMin;
            double elev;
            WayPoint wp;
            Task olcTask = null;
            //Task olcFAITask = null;
            Task dmstTask = null;
            Task task = flight.currentTask;
            FlightPoint fp;

            if (task.OptimizedFor != OptimizeTypes.OLC) {
                foreach (OptimizationBase opt in flight.optimizedTasks) {
                    Task t = opt.OptimizedTask;
                    if (t.OptimizedFor == OptimizeTypes.OLC) {
                        if (t.IsFAI) {
                            //                        olcFAITask = t;
                        }
                        else {
                            olcTask = t;
                        }
                    }
                }
            }
            else {
                olcTask = task;
            }

            if (task.OptimizedFor == OptimizeTypes.DMST || task.OptimizedFor == OptimizeTypes.DMSTFreeFlight) {
                dmstTask = task;
            }

//            if (olcTask != null) {
//                t = olcTask;
//            }
//            else if (olcFAITask != null) {
//                t = olcFAITask;
//            }
            if (olcTask == null) {
                throw new Exception("internal error: no OLC optimized task found!");
            }

            // competitor
            txt += string.Format("OLCvnolc={0}&na={1}&geb={2}&", textPilotPreName.Text, textPilotSurName.Text, dateTimePickerPilot1.Value.ToString("dd.MM.yy"));
            // co pilot
            if (textCoPilotPreName.Text != "") {
                txt += string.Format("bvn={0}&bna={1}&bgeb={2}&", textCoPilotPreName.Text, textCoPilotSurName.Text, dateTimePickerPilot2.Value.ToString("dd.MM.yy"));
            }
            // igcfile and comp. class
            txt += string.Format("igcfn={0}&Klasse={1}&", flight.Name, comboBoxCompClass.SelectedItem == null ? 0 : ((CompetitionClass)comboBoxCompClass.SelectedItem).olcId);

            // glider type, call sign, pure glider
            txt += string.Format("igty={0}&gid={1}&flugzeug={2}&", comboBoxGlider.SelectedItem != null ? ((Glider)comboBoxGlider.SelectedItem).Abbreviation : "",
                textBoxGliderId.Text, checkPureGlider.Checked == true ? 1 : 2);

            // departure location
            txt += string.Format("sta={0}&", comboBoxDeparture.SelectedItem != null ? ((OLCAirfield)comboBoxDeparture.SelectedItem).ID : "");

            // takeoff date/time
            tmp = string.Format("{0}", JulianDate(flight.Date));
            wp = olcTask[0];
            fp = wp.reachPoint;
            txt += string.Format("ft={0}&t0={1:00}:{2:00}:{3:00}&", tmp, fp.fixTime / 3600, (fp.fixTime % 3600) / 60, fp.fixTime % 60);

            i = flight.Start;
            fp = flight[i];
            elev = fp.Elevation;
            while (i < flight.Count) {
                if (flight[i] == olcTask[1].reachPoint) {
                    break;
                }
                if (flight[i].Elevation < elev) {
                    fp = flight[i];
                    elev = fp.Elevation;
                }
                i++;
            }

            txt += string.Format("s0={0:00}:{1:00}:{2:00}&", fp.fixTime / 3600, (fp.fixTime % 3600) / 60, fp.fixTime % 60);

            // finish of task
            wp = olcTask[olcTask.Count - 2];
            fp = wp.reachPoint;
            txt += string.Format("s6={0:00}:{1:00}:{2:00}&", fp.fixTime / 3600, (fp.fixTime % 3600) / 60, fp.fixTime % 60);

            // fill OLC fields
            // olcType is w for normal OLC, fai for OLC FAI triangles
            string olcType = "w";
//            for (j = 0; j < 2; j++) {
//                if (j == 0) {
//                    t = olcTask;
//                    olcType = "w";
//                }
//                else if (j == 1) {
//                    t = olcFAITask;
//                    olcType = "fai";
//                }

//                if (t == null) {
//                    continue;
//                }

                for (i = 1; i < olcTask.Count - 1; i++) {
                    wp = olcTask[i];
                    fp = wp.reachPoint;
                    latDeg = Math.Abs(wp.Latitude) / 360000;
                    latMin = (Math.Abs(wp.Latitude) % 360000) / 6000.0;
                    lonDeg = Math.Abs(wp.Longitude) / 360000;
                    lonMin = (Math.Abs(wp.Longitude) % 360000) / 6000.0;

                    txt += string.Format("{0}{1}bh={2}&{0}{1}bg={3:00}&{0}{1}bm={4:00}&{0}{1}bmd={5:000}&{0}{1}lh={6}&{0}{1}lg={7:000}&{0}{1}lm={8:00}&{0}{1}lmd={9:000}&", 
                        olcType,
                        i - 1,
                        wp.Latitude < 0 ? "N" : "S",
                        latDeg,
                        Math.Floor(latMin),
                        (int)Math.Round((latMin - Math.Floor(latMin)) * 1000.0, 0),
                        wp.Longitude < 0 ? "W" : "E",
                        lonDeg,
                        Math.Floor(lonMin),
                        (int)Math.Round((lonMin - Math.Floor(lonMin)) * 1000.0, 0));
                }
//            }

            if (dmstTask != null) {
                // fill in DMST fields
                wp = dmstTask[1];
                fp = wp.reachPoint;
                txt += string.Format("des0={0:00}:{1:00}:{2:00}&", fp.fixTime / 3600, (fp.fixTime % 3600) / 60, fp.fixTime % 60);
                // finish
                wp = dmstTask[dmstTask.Count - 2];
                fp = wp.reachPoint;
                txt += string.Format("des4={0:00}:{1:00}:{2:00}&", fp.fixTime / 3600, (fp.fixTime % 3600) / 60, fp.fixTime % 60);

                for (i = 1; i < dmstTask.Count - 1; i++) {
                    wp = dmstTask[i];
                    fp = wp.reachPoint;
                    latDeg = Math.Abs(wp.Latitude) / 360000;
                    latMin = (Math.Abs(wp.Latitude) % 360000) / 6000.0;
                    lonDeg = Math.Abs(wp.Longitude) / 360000;
                    lonMin = (Math.Abs(wp.Longitude) % 360000) / 6000.0;

                    txt += string.Format("dew{0}debh={1}&dew{0}bg={2:00}&dew{0}bm={3:00}&dew{0}bmd={4:000}&dew{0}lh={5}&dew{0}lg={6:000}&dew{0}lm={7:00}&dew{0}lmd={8:000}&", 
                        i - 1,
                        wp.Latitude < 0 ? "N" : "S",
                        latDeg,
                        Math.Floor(latMin),
                        (int)Math.Round((latMin - Math.Floor(latMin)) * 1000.0, 0),
                        wp.Longitude < 0 ? "W" : "E",
                        lonDeg,
                        Math.Floor(lonMin),
                        (int)Math.Round((lonMin - Math.Floor(lonMin)) * 1000.0, 0));
                }
            }

            txt += string.Format("Software={0}_{1}", Application.ProductName, Application.ProductVersion);

            return txt;
        }

        private NameValueCollection GetOlcParameter() {
            NameValueCollection parameters = new NameValueCollection();
            int i;
            long latDeg, lonDeg;
            double latMin, lonMin;
            double elev;
            WayPoint wp;
            Task olcTask = null;
            Task dmstTask = null;
            Task task = flight.currentTask;
            FlightPoint fp;
            string olcFile = "";
            string tmp;

            if (task.OptimizedFor != OptimizeTypes.OLC || task.IsFAI) {
                foreach (OptimizationBase opt in flight.optimizedTasks) {
                    Task t = opt.OptimizedTask;
                    if (t.OptimizedFor == OptimizeTypes.OLC && !t.IsFAI) {
                        olcTask = t;
                    }
                }
            }
            else {
                olcTask = task;
            }

            if (task.OptimizedFor == OptimizeTypes.DMST || task.OptimizedFor == OptimizeTypes.DMSTFreeFlight) {
                dmstTask = task;
            }

            if (olcTask == null) {
                throw new Exception("internal error: no OLC optimized task found!");
            }

            // competitor
            parameters.Add("OLCvnolc", textPilotPreName.Text);
            parameters.Add("na", textPilotSurName.Text);
            parameters.Add("geb", dateTimePickerPilot1.Value.ToString("dd.MM.yy"));

            // co pilot
            if (textCoPilotPreName.Text != "") {
                parameters.Add("bvn", textCoPilotPreName.Text);
                parameters.Add("bna", textCoPilotSurName.Text);
                parameters.Add("bgeb", dateTimePickerPilot2.Value.ToString("dd.MM.yy"));
            }

            // igcfile and comp. class
            parameters.Add("igcfn", flight.Name);
            parameters.Add("Klasse", (comboBoxCompClass.SelectedItem == null ? 0 : ((CompetitionClass)comboBoxCompClass.SelectedItem).olcId).ToString());
            
            // glider type, call sign, pure glider
            parameters.Add("igty", comboBoxGlider.SelectedItem != null ? ((Glider)comboBoxGlider.SelectedItem).Abbreviation : "");
            parameters.Add("gid", textBoxGliderId.Text);
            parameters.Add("flugzeug", checkPureGlider.Checked == true ? "1" : "0");
            
            // departure location
            parameters.Add("sta", comboBoxDeparture.SelectedItem != null ? ((OLCAirfield)comboBoxDeparture.SelectedItem).ID : "");
            
            // takeoff date/time
            wp = olcTask[0];
            fp = wp.reachPoint;
            parameters.Add("ft", JulianDate(flight.Date).ToString());
            parameters.Add("t0", string.Format("{0:00}:{1:00}:{2:00}&", fp.fixTime / 3600, (fp.fixTime % 3600) / 60, fp.fixTime % 60));
            
            i = flight.Start;
            fp = flight[i];
            elev = fp.Elevation;
            while (i < flight.Count) {
                if (flight[i] == olcTask[1].reachPoint) {
                    break;
                }
                if (flight[i].Elevation < elev) {
                    fp = flight[i];
                    elev = fp.Elevation;
                }
                i++;
            }

            parameters.Add("s0", string.Format("{0:00}:{1:00}:{2:00}&", fp.fixTime / 3600, (fp.fixTime % 3600) / 60, fp.fixTime % 60));
            
            // finish of task
            wp = olcTask[olcTask.Count - 2];
            fp = wp.reachPoint;
            parameters.Add("s6", string.Format("{0:00}:{1:00}:{2:00}&", fp.fixTime / 3600, (fp.fixTime % 3600) / 60, fp.fixTime % 60));
            
            // fill OLC fields
            // olcType is w for normal OLC, fai for OLC FAI triangles
            string olcType = "w";

            for (i = 1; i < olcTask.Count - 1; i++) {
                wp = olcTask[i];
                fp = wp.reachPoint;
                latDeg = Math.Abs(wp.Latitude) / 360000;
                latMin = (Math.Abs(wp.Latitude) % 360000) / 6000.0;
                lonDeg = Math.Abs(wp.Longitude) / 360000;
                lonMin = (Math.Abs(wp.Longitude) % 360000) / 6000.0;

                parameters.Add(string.Format("{0}{1}bh", olcType, i - 1), wp.Latitude < 0 ? "N" : "S");
                parameters.Add(string.Format("{0}{1}bg", olcType, i - 1), string.Format("{0:00}", latDeg));
                parameters.Add(string.Format("{0}{1}bm", olcType, i - 1), string.Format("{0:00}", Math.Floor(latMin)));
                parameters.Add(string.Format("{0}{1}bmd", olcType, i - 1), string.Format("{0:000}", Math.Round((latMin - Math.Floor(latMin)) * 1000.0, 0)));
                parameters.Add(string.Format("{0}{1}lh", olcType, i - 1), wp.Longitude < 0 ? "W" : "E");
                parameters.Add(string.Format("{0}{1}lg", olcType, i - 1), string.Format("{0:00}", lonDeg));
                parameters.Add(string.Format("{0}{1}lm", olcType, i - 1), string.Format("{0:00}", Math.Floor(lonMin)));
                parameters.Add(string.Format("{0}{1}lmd", olcType, i - 1), string.Format("{0:000}", Math.Round((lonMin - Math.Floor(lonMin)) * 1000.0, 0)));
            }

            if (dmstTask != null) {
                // fill in DMST fields
                wp = dmstTask[1];
                fp = wp.reachPoint;
                parameters.Add("des0", string.Format("{0:00}:{1:00}:{2:00}&", fp.fixTime / 3600, (fp.fixTime % 3600) / 60, fp.fixTime % 60));

                // finish
                wp = dmstTask[dmstTask.Count - 2];
                fp = wp.reachPoint;
                parameters.Add("des4", string.Format("{0:00}:{1:00}:{2:00}&", fp.fixTime / 3600, (fp.fixTime % 3600) / 60, fp.fixTime % 60));

                for (i = 1; i < dmstTask.Count - 1; i++) {
                    wp = dmstTask[i];
                    fp = wp.reachPoint;
                    latDeg = Math.Abs(wp.Latitude) / 360000;
                    latMin = (Math.Abs(wp.Latitude) % 360000) / 6000.0;
                    lonDeg = Math.Abs(wp.Longitude) / 360000;
                    lonMin = (Math.Abs(wp.Longitude) % 360000) / 6000.0;

                    parameters.Add(string.Format("dew{0}bh", i - 1), wp.Latitude < 0 ? "N" : "S");
                    parameters.Add(string.Format("dew{0}bg", i - 1), string.Format("{0:00}", latDeg));
                    parameters.Add(string.Format("dew{0}bm", i - 1), string.Format("{0:00}", Math.Floor(latMin)));
                    parameters.Add(string.Format("dew{0}bmd", i - 1), string.Format("{0:000}", Math.Round((latMin - Math.Floor(latMin)) * 1000.0, 0)));
                    parameters.Add(string.Format("dew{0}lh", i - 1), wp.Longitude < 0 ? "W" : "E");
                    parameters.Add(string.Format("dew{0}lg", i - 1), string.Format("{0:00}", lonDeg));
                    parameters.Add(string.Format("dew{0}lm", i - 1), string.Format("{0:00}", Math.Floor(lonMin)));
                    parameters.Add(string.Format("dew{0}lmd", i - 1), string.Format("{0:000}", Math.Round((lonMin - Math.Floor(lonMin)) * 1000.0, 0)));

                }
            }

            parameters.Add("Software", string.Format("{0}_{1}", Application.ProductName, Application.ProductVersion));
            
            using (StreamReader inStream = new StreamReader(flight.FileName)) {
                while ((tmp = inStream.ReadLine()) != null) {
                    olcFile += tmp + Environment.NewLine;
                }
            }
            parameters.Add("IGCigcIGC", olcFile);
            return parameters;
        }

        private int JulianDate(string date) {
            DateTime d = DateTime.ParseExact(date, new string [] {"dd.MM.yy", "dd.MM.yyyy"}, null, System.Globalization.DateTimeStyles.NoCurrentDateDefault);
            DateTime d2 = new DateTime(1601, 1, 1);
            TimeSpan t = d - d2;
            return t.Days;
        }

        private void OnSelectedCompetitionChanged(object sender, System.EventArgs e) {
        }

        private void OnSaveOLC(object sender, System.EventArgs e) {
            SaveFileDialog d = new SaveFileDialog();
            d.FileName = Path.GetFileNameWithoutExtension(flight.FileName) + ".OLC";
            d.Filter = "OLC files (*.olc)|*.olc|All files (*.*)|*.*";
            d.AddExtension = true;
            d.CheckPathExists = true;
            d.DefaultExt = "OLC";
            d.OverwritePrompt = true;
            d.Title = "Save OLC file";
            string line;

            if (d.ShowDialog(this) == DialogResult.OK) {
                using (StreamWriter outStream = new StreamWriter(d.FileName)) {
                    outStream.Write(BuildOlcString() + "&IGCigcIGC=");
                    using (StreamReader inStream = new StreamReader(flight.FileName)) {
                        while ((line = inStream.ReadLine()) != null) {
                            outStream.WriteLine(line);
                        }
                    }
                }
            }
        }

        private void OnDepartureOptionsChanged(object sender, EventArgs e) {
            comboBoxDeparture.Items.Clear();
            comboBoxDeparture.Sorted = false;
            Cursor = Cursors.WaitCursor;
            if (radioButtonShowDeparture50km.Checked) {
                if (flight.currentTask != null && flight.currentTask.Count > 0) {
                    WayPoint wp = flight.currentTask[0];
                    foreach (SoaringDotNET.Data.OLCAirfield f in AppContents.olcAirfields) {
                        if (f.distanceKM(wp) <= 50.0) {
                            comboBoxDeparture.Items.Add(f);
                        }
                    }
                }
            }
            else if (radioButtonShowDepartureAll.Checked) {
                comboBoxDeparture.Items.AddRange(AppContents.olcAirfields.ToArray());
            }
            comboBoxDeparture.Sorted = true;
            Cursor = Cursors.Default;
        }
	}
}
