// name   : RecorderBase.cs
// author : Harald Maier
// date   : 25.11.2003
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.Collections;
using System.Threading;

using SoaringDotNET.GUI;

namespace SoaringDotNET.Recorders
{
	/// <summary>
	/// 
	/// </summary>
	public enum RecorderOptions {DownloadFlights = 1, DownloadTasks = 2, DownloadWaypoints = 4, UploadDeclaration = 8,
        UploadTask = 16, UploadWaypoints = 32, UploadSUA = 64, DownloadPilots = 128, UploadPilots = 256};

    public enum RecorderConnectionType {Unknown, Serial, FileSystem};

    public delegate void Notify(bool success);

    public struct Capacity {
        public int maxNrWaypoints;
        public int maxNrTasks;
        public int maxNrWaypointsPerTask;
        public int maxNrPilots;
    }

    public enum CapacityNumbers {NotAvailable = 0, Unlimited = -1};

    public abstract class RecorderBase
	{
        protected static readonly string c36 = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
        protected RecorderOptions options = RecorderOptions.DownloadFlights;
        protected bool breakTransfer = false;
        protected bool redirectToFile = false;
        protected Mutex mutex = new Mutex();
        protected uint serialNo = 0;
        protected string name = "";
        protected string ManufacurerCode = "";
        protected string ManufacurerLetter = "";
        protected RecorderConnectionType connectionType = RecorderConnectionType.Unknown;
        protected bool haveDatabase = false;

        public WaypointCatalog currentCatalogue;// = null;
        public ArrayList tasks;// = new ArrayList();
        public ArrayList suas;// = new ArrayList();
        public ArrayList flightDirectory;// = new ArrayList();
        public ArrayList flightIndexes;// = new ArrayList();
        public FlightDeclaration flightDeclaration;// = new FlightDeclaration();
        public Task declarationTask;// = new Task();
        public ArrayList pilots;
        public bool secureDownload;// = true;

        public static Notify SerialNoThreadNotify = null;
        public static Notify FlightDirectoryThreadNotify = null;
        public static Notify DownloadWaypointsThreadNotify = null;
        public static Notify UploadWaypointsThreadNotify = null;
        public static Notify DownloadTasksThreadNotify = null;
        public static Notify UploadTasksThreadNotify = null;
        public static Notify UploadDeclarationThreadNotify = null;
        public static Notify DownloadFlightsThreadNotify = null;
        public static Notify UploadSUAsThreadNotify = null;
        public static Notify DownloadPilotsThreadNotify = null;
        public static Notify UploadPilotsThreadNotify = null;

        public Capacity capacity;// = new Capacity();

		public RecorderBase()
		{
			// 
			// TODO: Add constructor logic here
			//
            currentCatalogue = null;
            mutex = new Mutex();
            tasks = new ArrayList();
            suas = new ArrayList();
            flightDirectory = new ArrayList();
            flightIndexes = new ArrayList();
            flightDeclaration = new FlightDeclaration();
            declarationTask = new Task();
            pilots = new ArrayList();

            name = "Recorder Base";
            secureDownload = true;
            connectionType = RecorderConnectionType.Unknown;

            capacity = new Capacity();
            capacity.maxNrWaypoints = (int)CapacityNumbers.NotAvailable;
            capacity.maxNrTasks = (int)CapacityNumbers.NotAvailable;
            capacity.maxNrWaypointsPerTask = (int)CapacityNumbers.NotAvailable;
            capacity.maxNrPilots = (int)CapacityNumbers.NotAvailable;
        }

        abstract public bool Open(string port, int speed);
        abstract public bool Open();
        abstract protected void GetFlightDir();
        abstract protected void DownloadWaypoints();
        abstract protected void UploadWaypoints();
        abstract protected void DownloadTasks();
        abstract protected void UploadTasks();
        abstract protected void DownloadPilots();
        abstract protected void UploadPilots();
        abstract protected void UploadDeclaration();
        abstract protected void DownloadFlights();
        abstract protected void UploadSUAs();

        #region Public Functions
        public virtual void Close() {
            haveDatabase = false;
        }

        protected virtual void GetSerialNo() {
            // TODO:  Add GetSerialNoThread implementation
            serialNo = 0;
            if (RecorderBase.SerialNoThreadNotify != null) {
                RecorderBase.SerialNoThreadNotify(true);
            }
        }
    

        public Thread SerialNoThread() {
            return new Thread(new ThreadStart(GetSerialNo));
        }

        public Thread FlightDirectoryThread() {
            return new Thread(new ThreadStart(GetFlightDir));
        }

        public Thread DownloadWaypointsThread() {
            return new Thread(new ThreadStart(DownloadWaypoints));
        }

        public Thread UploadWaypointsThread() {
            return new Thread(new ThreadStart(UploadWaypoints));
        }

        public Thread DownloadTasksThread() {
            return new Thread(new ThreadStart(DownloadTasks));
        }

        public Thread UploadTasksThread() {
            return new Thread(new ThreadStart(UploadTasks));
        }

        public Thread DownloadPilotsThread() {
            return new Thread(new ThreadStart(DownloadPilots));
        }

        public Thread UploadPilotsThread() {
            return new Thread(new ThreadStart(UploadPilots));
        }

        public Thread UploadDeclarationThread() {
            return new Thread(new ThreadStart(UploadDeclaration));
        }

        public Thread DownloadFlightsThread() {
            return new Thread(new ThreadStart(DownloadFlights));
        }

        public Thread UploadSUASThread() {
            return new Thread(new ThreadStart(UploadSUAs));
        }

        public void BreakTransfer() {
            lock(mutex) {
                breakTransfer = true;
            }
        }

        public override string ToString() {
            // TODO:  Add Volkslogger.ToString implementation
            return name;
        }

        #endregion

        #region Protected Functions
        protected uint ToSerialNo(string txt) {
            uint n;
            n = (uint)(c36.IndexOf(txt[0]) * 36 * 36) + (uint)(c36.IndexOf(txt[1]) * 36) + (uint)c36.IndexOf(txt[2]);
            return n;
        }
        #endregion

        #region Attributes
        public RecorderConnectionType ConnectionType {
            get {
                return connectionType;
            }
        }

        public RecorderOptions Options {
            get {
                return options;
            }
        }

        public bool IsInterrupted {
            get {
                bool b;
                lock(mutex) {
                    b = breakTransfer;
                }
                return b;
            }
            set {
                lock(mutex) {
                    breakTransfer = value;
                }
            }
        }

        public string SerialNo {
            get {
                if (serialNo >= 46656) {
                    serialNo = 46655;
                }
                return string.Format("{0}{1}{2}", c36[(int)serialNo / 36 / 36], c36[((int)serialNo / 36) % 36], c36[(int)serialNo % 36]);
            }
        }

        public virtual string Description {
            get {
                return "";
            }
        }

        public string OptionsString {
            get {
                return string.Format(@"
Options:
    Download flights    : {0}
    Waypoints           : {1}/{2}
    Tasks               : {3}/{4}
    Pilots              : {5}/{6}
    Declaration form    : {7}
    SUAs                : {8}

Capacity:
    Waypoints       : {9}
    Tasks           : {10}
    Waypoints/Task  : {11}
    Pilots          : {12}", 
                    (options & RecorderOptions.DownloadFlights) == RecorderOptions.DownloadFlights ? "yes" : "n.a.",
                    (options & RecorderOptions.UploadWaypoints) == RecorderOptions.UploadWaypoints ? "upload" : "n.a.",
                    (options & RecorderOptions.DownloadWaypoints) == RecorderOptions.DownloadWaypoints ? "download" : "n.a.",
                    (options & RecorderOptions.UploadTask) == RecorderOptions.UploadTask ? "upload" : "n.a.",
                    (options & RecorderOptions.DownloadTasks) == RecorderOptions.DownloadTasks ? "download" : "n.a.",
                    (options & RecorderOptions.UploadPilots) == RecorderOptions.UploadPilots ? "upload" : "n.a.",
                    (options & RecorderOptions.DownloadPilots) == RecorderOptions.DownloadPilots ? "download" : "n.a.",
                    (options & RecorderOptions.UploadDeclaration) == RecorderOptions.UploadDeclaration ? "yes" : "n.a",
                    (options & RecorderOptions.UploadSUA) == RecorderOptions.UploadSUA ? "yes" : "n.a.",
                    capacity.maxNrWaypoints == (int) CapacityNumbers.Unlimited ? "unlimited" : capacity.maxNrWaypoints.ToString(), 
                    capacity.maxNrTasks == (int) CapacityNumbers.Unlimited ? "unlimited" : capacity.maxNrTasks.ToString(), 
                    capacity.maxNrWaypointsPerTask == (int) CapacityNumbers.Unlimited ? "unlimited" : capacity.maxNrWaypointsPerTask.ToString(),
                    capacity.maxNrPilots == (int) CapacityNumbers.Unlimited ? "unlimited" : capacity.maxNrPilots.ToString());
            }
        }
        #endregion
    
    }

    public class RecorderTimeoutException : Exception {
        public RecorderTimeoutException(string message)
            :
            base(message) {
        }
    }
}
