// name   : SUAFileHandlerTimNewport.cs
// author : Harald Maier
// date   : 22.02.2006
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.IO;
using System.Collections;
using System.Text.RegularExpressions;
using System.Windows.Forms;

using SoaringDotNET.GUI;
using SoaringDotNET.Data;

namespace SoaringDotNET.FileFormats
{
	/// <summary>
	/// 
	/// </summary>
	public class SUAFileHandlerOpenAir : SoaringDotNET.FileFormats.SUAFileHandlerBase
	{
		public SUAFileHandlerOpenAir()
		{
			// 
			// TODO: Add constructor logic here
			//
		}

        public override ArrayList WriteFile(ArrayList suas) {
            ArrayList file = new ArrayList(suas.Count * 6);
            bool firstSeg;
            string tmp = "";
            Airspace space = null;
            AirspaceSegmentTypes direction;

            file.Add(string.Format("* Generated by {0} Version {1}", Application.ProductName, Application.ProductVersion));
            file.Add("* NOTE: This file, like any other generated by this program, comes with absolutly no warranty.");
            file.Add("* It's in the users responsibility to check for correctness and accuracy. This data is unofficial");
            file.Add(string.Format("* and is used by your own risk!{0}", Environment.NewLine));

            foreach (AirspaceElement sua in suas) {
                direction = AirspaceSegmentTypes.ArcClock;

                space = sua.airspace;

                switch (space.airspaceType) {
                case AirspaceTypes.ControlArea:
                case AirspaceTypes.TerminalControlArea:
                    if (space.airspaceClass != AirspaceClasses.Unknown) {
                        tmp = space.airspaceClass.ToString();
                    }
                    break;
                case AirspaceTypes.ControlZone:
                    tmp = "CTR";
                    break;
                case AirspaceTypes.Danger:
                    tmp = "Q";
                    break;
                case AirspaceTypes.Prohibited:
                    tmp = "P";
                    break;
                case AirspaceTypes.Restricted:
                    tmp = "R";
                    break;
                case AirspaceTypes.Soaring:
                    tmp = "W";
                    break;
                case AirspaceTypes.Alert:
                case AirspaceTypes.GliderProhibited:
                case AirspaceTypes.Military:
                case AirspaceTypes.MilitaryAerodromeTraffic:
                case AirspaceTypes.TMZ:
                    tmp = "GP";
                    break;
                default:
                    // ignore all others
                    continue;
                }
                file.Add(string.Format("AC {0}", tmp));
                file.Add(string.Format("AN {0}", space.name));
                file.Add(string.Format("AH {0}", space.upperBound));
                file.Add(string.Format("AL {0}", space.lowerBound));

                firstSeg = true;
                foreach (AirspaceSegment seg in sua) {
                    switch (seg.type) {
                    case AirspaceSegmentTypes.Line:
                        if (firstSeg) {
                            file.Add(string.Format("DP {0} {1}", WgsPoint.ToColonString(seg.latFrom, true), WgsPoint.ToColonString(seg.lonFrom, false)));
                            firstSeg = false;
                        }
                        file.Add(string.Format("DP {0} {1}", WgsPoint.ToColonString(seg.latTo, true), WgsPoint.ToColonString(seg.lonTo, false)));
                        break;
                    case AirspaceSegmentTypes.Circle:
                        file.Add(string.Format(AppContents.ni, "V X={0} {1}", WgsPoint.ToColonString(seg.latCenter, true), WgsPoint.ToColonString(seg.lonCenter, false)));
                        file.Add(string.Format(AppContents.ni, "DC {0}", seg.radius));
                        break;
                    case AirspaceSegmentTypes.ArcClock:
                    case AirspaceSegmentTypes.ArcCounterClock:
                        if (direction != seg.type) {
                            direction = seg.type;
                            file.Add(string.Format("V D={0}", direction == AirspaceSegmentTypes.ArcClock ? "+" : "-"));
                        }
                        file.Add(string.Format(AppContents.ni, "V X={0} {1}", WgsPoint.ToColonString(seg.latCenter, true), WgsPoint.ToColonString(seg.lonCenter, false)));
                        file.Add(string.Format("DB {0} {1}, {2} {3}", WgsPoint.ToColonString(seg.latFrom, true), WgsPoint.ToColonString(seg.lonFrom, false), 
                            WgsPoint.ToColonString(seg.latTo, true), WgsPoint.ToColonString(seg.lonTo, false)));
                        break;
                    }
                }
                file.Add("*");
            }
            file.Add("END");

            return file;
        }


        public override ArrayList ReadFile(ArrayList file, bool useMichaelMeier, AirspaceClasses defaultClass, bool overwriteClass, SUATypes defaultType, bool overwriteType) {
            ArrayList suas = new ArrayList();
            Match m = null;
            GroupCollection coll;
            string t;
            int lat, lon, latCenter, lonCenter;
            int lineNo = 0;
            AirspaceSegmentTypes direction;
            double distance;
            WgsPoint wp1 = new WgsPoint();
            WgsPoint wp2 = new WgsPoint();

            Regex tokens = new Regex("(?:(?<first>^AC)[ \t,]+(?<second>.+))|" +
                "(?:(?<first>^AN)[ \t,]+(?<second>.+))|" +
                "(?:(?<first>^AL)[ \t,]+(?<second>.+))|" +
                "(?:(?<first>^AH)[ \t,]+(?<second>.+))|" +
                "(?:(?<first>^V)[ \t]+(?<var>D)[ \t]*=[ \t]*(?<second>\\+|-))|" +
                "(?:(?<first>^V)[ \t]+(?<var>X)[ \t]*=[ \t]*(?<lat>[0-9]{1,2}:[0-9]{1,2}(?::[0-9]{1,2}(?:\\.[0-9]+)?|\\.[0-9]+)[ \t]*[N|S])[ \t]+(?<lon>[0-9]{1,3}:[0-9]{1,2}(?::[0-9]{1,2}(?:\\.[0-9]+)?|\\.[0-9]+)[ \t]*[W|E]))|" +
                "(?:(?<first>^DP)[ \t]+(?<lat>[0-9]{1,2}:[0-9]{1,2}(?::[0-9]{1,2}(?:\\.[0-9]+)?|\\.[0-9]+)[ \t]*[N|S])[ \t]+(?<lon>[0-9]{1,3}:[0-9]{1,2}(?::[0-9]{1,2}(?:\\.[0-9]+)?|\\.[0-9]+)[ \t]*[W|E]))|" +
                "(?:(?<first>^DA)[ \t]+(?<radius>[0-9\\.]+)[, \t]+(?<from>[0-9]+)[, \t]+(?<to>[0-9]+))|" +
                "(?:(?<first>^DB)[ \t]+(?<latFrom>[0-9]{1,2}:[0-9]{1,2}(?::[0-9]{1,2}(?:\\.[0-9]+)?|\\.[0-9]+)[ \t]*[N|S])[ \t]+(?<lonFrom>[0-9]{1,3}:[0-9]{1,2}(?::[0-9]{1,2}(?:\\.[0-9]+)?|\\.[0-9]+)[ \t]*[W|E])[, \t]+(?<latTo>[0-9]{1,2}:[0-9]{1,2}(?::[0-9]{1,2}(?:\\.[0-9]+)?|\\.[0-9]+)[ \t]*[N|S])[ \t]+(?<lonTo>[0-9]{1,3}:[0-9]{1,2}(?::[0-9]{1,2}(?:\\.[0-9]+)?|\\.[0-9]+)[ \t]*[W|E]))|" +
                "(?:(?<first>^DC)[ \t]+(?<radius>[0-9\\.]+))", RegexOptions.IgnoreCase);

            Airspace space = new Airspace();
            direction = AirspaceSegmentTypes.ArcClock;
            latCenter = lonCenter = 0;
            AirspaceSegment segment = null;

            try {
                foreach (string line in file) {
                    lineNo++;
                    m = tokens.Match(line);
                    if (m.Success) {
                        coll = m.Groups;
                        t = coll["first"].Value.ToUpper();
                        switch (t) {
                        case "AC":
                            if (space.name.Length > 0) {
                                CheckImportOptions(space, useMichaelMeier, defaultClass, overwriteClass, defaultType, overwriteType);
                                suas.Add(space);
                            }
                            space = new Airspace();
                            direction = AirspaceSegmentTypes.ArcClock;
                            latCenter = lonCenter = 0;
                            segment = null;

                            switch (coll["second"].Value.ToUpper()) {
                            case "A":
                                space.airspaceType = AirspaceTypes.ControlArea;
                                space.airspaceClass = AirspaceClasses.A;
                                break;
                            case "B":
                                space.airspaceType = AirspaceTypes.ControlArea;
                                space.airspaceClass = AirspaceClasses.B;
                                break;
                            case "C":
                                space.airspaceType = AirspaceTypes.ControlArea;
                                space.airspaceClass = AirspaceClasses.C;
                                break;
                            case "D":
                                space.airspaceType = AirspaceTypes.ControlArea;
                                space.airspaceClass = AirspaceClasses.D;
                                break;
                            case "E":
                                space.airspaceType = AirspaceTypes.ControlArea;
                                space.airspaceClass = AirspaceClasses.E;
                                break;
                            case "F":
                                space.airspaceType = AirspaceTypes.ControlArea;
                                space.airspaceClass = AirspaceClasses.F;
                                break;
                            case "G":
                                space.airspaceType = AirspaceTypes.ControlArea;
                                space.airspaceClass = AirspaceClasses.G;
                                break;
                            case "R":
                                space.airspaceType = AirspaceTypes.Restricted;
                                break;
                            case "Q":
                                space.airspaceType = AirspaceTypes.Danger;
                                break;
                            case "P":
                                space.airspaceType = AirspaceTypes.Prohibited;
                                break;
                            case "GP":
                                space.airspaceType = AirspaceTypes.GliderProhibited;
                                break;
                            case "CTR":
                                space.airspaceType = AirspaceTypes.ControlZone;
                                break;
                            case "W":
                            case "GSEC":
                                space.airspaceType = AirspaceTypes.Soaring;
                                break;
                            case "TMZ":
                                space.airspaceType = AirspaceTypes.TMZ;
                                break;
                            default:
                                space.airspaceType = AirspaceTypes.Other;
                                break;
                            }
                            break;
                        case "AN":
                            space.name = coll["second"].Value;
                            break;
                        case "AL":
                            space.lowerBound = ParseAltitude(coll["second"].Value);
                            break;
                        case "AH":
                            space.upperBound = ParseAltitude(coll["second"].Value);
                            break;
                        case "V":
                            if (coll["var"].Value == "D") {
                                direction = coll["second"].Value == "+" ? AirspaceSegmentTypes.ArcClock : AirspaceSegmentTypes.ArcCounterClock;
                            }
                            else {
                                latCenter = WgsPoint.FromColonString(coll["lat"].Value, true);
                                lonCenter = WgsPoint.FromColonString(coll["lon"].Value, false);
                            }
                            break;
                        case "DP":
                            lat = WgsPoint.FromColonString(coll["lat"].Value, true);
                            lon = WgsPoint.FromColonString(coll["lon"].Value, false);
                            if (segment == null) {
                                segment = new AirspaceSegment();
                                // from
                                segment.latFrom = lat;
                                segment.lonFrom = lon;
                            }
                            else {
                                segment.type = AirspaceSegmentTypes.Line;
                                // to
                                segment.latTo = lat;
                                segment.lonTo = lon;

                                space.points.Add(segment);
                                segment = new AirspaceSegment();
                                segment.type = AirspaceSegmentTypes.Unknown;
                                // from = to
                                segment.latFrom = lat;
                                segment.lonFrom = lon;
                            }
                            break;
                        case "DA":
                            if (latCenter == 0 && lonCenter == 0) {
                                throw new Exception("No prev. CENTER definition for clockwise or anti-clockwise arc!");
                            }
                            if (segment == null) {
                                segment = new AirspaceSegment();
                            }
                            segment.type = direction;

                            // center, radius, to
                            segment.latCenter = latCenter;
                            segment.lonCenter = lonCenter;
                            segment.radius = float.Parse(coll["radius"].Value, AppContents.ni);

                            distance = ConvertionFactors.NAUTICMILES2KILOMETERS * segment.radius;
                            wp1.Latitude = latCenter;
                            wp1.Longitude = lonCenter;

                            wp2 = wp1.PosOfDistAndBearing(double.Parse(coll["from"].Value) / 180.0 * Math.PI, distance);
                            segment.latFrom = wp2.Latitude;
                            segment.lonFrom = wp2.Longitude;

                            wp2 = wp1.PosOfDistAndBearing(double.Parse(coll["to"].Value) / 180.0 * Math.PI, distance);
                            lat = wp2.Latitude;
                            lon = wp2.Longitude;
                            segment.latTo = wp2.Latitude;
                            segment.lonTo = wp2.Longitude;

                            space.points.Add(segment);
                            segment = new AirspaceSegment();
                            segment.type = AirspaceSegmentTypes.Unknown;
                            // from = to
                            segment.latFrom = lat;
                            segment.lonFrom = lon;
                            break;
                        case "DB":
                            if (latCenter == 0 && lonCenter == 0) {
                                throw new Exception("No prev. CENTER definition for clockwise or anti-clockwise arc!");
                            }

                            if (segment == null) {
                                segment = new AirspaceSegment();
                            }
                            segment.type = direction;

                            // from, to
                            segment.latCenter = latCenter;
                            segment.lonCenter = lonCenter;

                            segment.latFrom = WgsPoint.FromColonString(coll["latFrom"].Value, true);
                            segment.lonFrom = WgsPoint.FromColonString(coll["lonFrom"].Value, false);
                            
                            wp1.Latitude = latCenter;
                            wp1.Longitude = lonCenter;
                            wp2.Latitude = segment.latFrom;
                            wp2.Longitude = segment.lonFrom;

                            segment.radius = (float)wp1.distanceNM(wp2);

                            lat = WgsPoint.FromColonString(coll["latTo"].Value, true);
                            lon = WgsPoint.FromColonString(coll["lonTo"].Value, false);
                            segment.latTo = lat;
                            segment.lonTo = lon;

                            space.points.Add(segment);
                            segment = new AirspaceSegment();
                            segment.type = AirspaceSegmentTypes.Unknown;

                            // from = to
                            segment.latFrom = lat;
                            segment.lonFrom = lon;
                            break;
                        case "DC":
                            if (space.points.Count > 0) {
                                throw new Exception("Found other segments with circle entry");
                            }
                            else if (latCenter == 0 && lonCenter == 0 && space.name.Length > 0) {
                                throw new Exception("No prev. CENTER definition for circle!");
                            }
                            
                            if (segment == null) {
                                segment = new AirspaceSegment();
                            }

                            segment.type = AirspaceSegmentTypes.Circle;

                            // center, radius
                            segment.latCenter = latCenter;
                            segment.lonCenter = lonCenter;
                            segment.radius = float.Parse(coll["radius"].Value, AppContents.ni);
                            space.points.Add(segment);
                            segment = new AirspaceSegment();
                            break;
                        }
                    }
                }

                if (space.name.Length > 0) {
                    CheckImportOptions(space, useMichaelMeier, defaultClass, overwriteClass, defaultType, overwriteType);
                    suas.Add(space);
                }
            }
            catch (Exception e) {
                MessageBox.Show("An exception occured while reading file (Open Air)\n\nin line " + lineNo.ToString() + 
                    "\nThe fault line is:\n\n" + m.ToString() + "\n\n" + e.StackTrace + "\n\n" + e.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Stop);
                suas.Clear();
            }

            return suas;
        }
    }
}
