using System;
using System.Collections;
using System.Windows.Forms;
using System.Text;
using System.IO;

using SoaringDotNET.Data;
using SoaringDotNET.GUI;
using SoaringDotNET.FileFormats;

namespace SoaringDotNET.Recorders {
    public class SoaringPilotFile : SoaringDotNET.Recorders.FileSystemRecorder {
        public SoaringPilotFile() {
            options = RecorderOptions.DownloadTasks |
                RecorderOptions.DownloadWaypoints |
                RecorderOptions.UploadTask |
                RecorderOptions.UploadWaypoints |
                RecorderOptions.UploadSUA;

            capacity.maxNrTasks = (int)CapacityNumbers.Unlimited;
            capacity.maxNrWaypoints = (int)CapacityNumbers.Unlimited;
            capacity.maxNrWaypointsPerTask = (int)CapacityNumbers.Unlimited;

            name = "Soaring Pilot (file based)";
            ManufacurerCode = "XSP";
            ManufacurerLetter = "X";
        }

        #region Protected Functions
        protected ArrayList ReadFile(string fileName) {
            // TODO:  Add SoaringPilotFile.ReadFile implementation
            string line;
            ArrayList file = new ArrayList();
            OpenFileDialog fd = new OpenFileDialog();
            fd.CheckFileExists = true;
            fd.Filter = "All files (*.*)|*.*|Waypoints (*.dat)|*.dat|Tasks (*.spt|*.spt|SUA (*.sua)|*.sua";
            fd.FilterIndex = 0;
            fd.FileName = fileName;

            if (fd.ShowDialog() == DialogResult.OK) {
                try {
                    using (StreamReader inStream = new StreamReader(fd.FileName)) {
                        while ((line = inStream.ReadLine()) != null) {
                            file.Add(line);
                        }
                    }
                }
                catch (Exception e) {
                    MessageBox.Show(e.ToString(), "Read Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
            return file;
        }

        protected void WriteFile(ArrayList file, string fileName) {
            // TODO:  Add SoaringPilotFile.WriteFile implementation
            SaveFileDialog fd = new SaveFileDialog();
            fd.CheckPathExists = true;
            fd.OverwritePrompt = false;
            fd.ValidateNames = true;
            fd.FileName = fileName;
            fd.Filter = "All files (*.*)|*.*|Waypoints (*.dat)|*.dat|Tasks (*.spt|*.spt|SUA (*.sua)|*.sua";
            fd.FilterIndex = 0;
            StreamWriter outStream = null;
            if (fd.ShowDialog() == DialogResult.OK) {
                try {
                    outStream = new StreamWriter(fd.FileName, false);
                    foreach (string line in file) {
                        outStream.WriteLine(line);
                    }
                }
                catch (Exception e) {
                    MessageBox.Show(e.ToString(), "Write Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
                finally {
                    if (outStream != null) {
                        outStream.Close();
                    }
                }
            }
        }

        protected override void DownloadTasks() {
            bool success = true;
            bool takeoff = false, landing = false;
            int nrWayointsInTask = 0, nrWaypointsRead = 0;
            int lat = 0, lon = 0;
            int elev = 0;
            SoaringPilotFlags type;

            string[] tokens;
            string[] coordItems;
            Task task = null;
            WayPoint wp;
            string t = "";
            char c;

            IsInterrupted = false;
            try {
                ArrayList file = ReadFile("tasks.spt");
                foreach (string line in file) {
                    tokens = line.Split(',');
                    if (tokens.Length > 0) {
                        switch (tokens[0]) {
                        case "TS":
                            nrWayointsInTask = int.Parse(tokens[2]);
                            nrWaypointsRead = 0;
                            if (tokens.Length > 3) {
                                takeoff = (tokens[3].IndexOf('T') != -1);
                                landing = (tokens[3].IndexOf('L') != -1);
                            }
                            else {
                                takeoff = landing = false;
                            }
                            task = new Task(tokens[1]);
                            break;
                        case "TW":
                            // lat
                            coordItems = tokens[1].Split(':');
                            switch (coordItems.Length) {
                            case 2:
                                // 48:23.083N format
                                t = coordItems[1];
                                lat = (int.Parse(coordItems[0]) * 360000) + (int)(double.Parse(t.Substring(0, t.Length - 1), AppContents.ni) * 6000.0);
                                break;
                            case 3:
                                // 48:00:42.00N format
                                t = coordItems[2];
                                lat = (int.Parse(coordItems[0]) * 360000) + (int.Parse(coordItems[1]) * 6000) + (int)(double.Parse(t.Substring(0, t.Length - 1), AppContents.ni) * 100);
                                break;
                            }

                            c = t[t.Length - 1];
                            if (c == 'N' || c == 'n') {
                                lat = -lat;
                            }

                            // lon
                            coordItems = tokens[2].Split(':');
                            switch (coordItems.Length) {
                            case 2:
                                // 48:23.083N format
                                t = coordItems[1];
                                lon = (int.Parse(coordItems[0]) * 360000) + (int)(double.Parse(t.Substring(0, t.Length - 1), AppContents.ni) * 6000.0);
                                break;
                            case 3:
                                // 48:00:42.00N format
                                t = coordItems[2];
                                lon = (int.Parse(coordItems[0]) * 360000) + (int.Parse(coordItems[1]) * 6000) + (int)(double.Parse(t.Substring(0, t.Length - 1), AppContents.ni) * 100);
                                break;
                            }

                            c = t[t.Length - 1];
                            if (c == 'W' || c == 'w') {
                                lon = -lon;
                            }

                            // type
                            try {
                                type = (SoaringPilotFlags)int.Parse(tokens[3]);
                            }
                            catch {
                                type = SoaringPilotFlags.Turnpoint;
                            }

                            // elevation
                            try {
                                t = tokens[4];
                                c = t[t.Length - 1];
                                elev = int.Parse(t.Substring(0, t.Length - 1));
                            }
                            catch {
                                elev = 0;
                                c = 'm';
                            }

                            if (c != 'M' && c != 'm') {
                                if (c == 'F' || c == 'f') {
                                    elev = (int)((double)elev * ConvertionFactors.FEET2METER);
                                }
                                else {
                                    throw new Exception("Unknown elevation unit " + c + " in line\n" + line);
                                }
                            }

                            if (tokens.Length > 6) {
                                wp = new WayPoint(lat, lon, tokens[6].Trim());
                                wp.shortName = tokens[5].Trim();
                            }
                            else {
                                wp = new WayPoint(lat, lon, tokens[5].Trim());
                            }
                            wp.Elevation = elev;
                            task.Add(wp);
                            nrWaypointsRead++;

                            if ((type & SoaringPilotFlags.Area) == SoaringPilotFlags.Area && tokens.Length > 7) {
                                // read area info
                                // 005998005329101202
                                // |     |     |  |
                                t = tokens[7];
                                SectorDefinition sector = task.GetSector(task.Count - 1);
                                sector.sectorType = SectorTypes.Area;
                                try {
                                    // round to 100m
                                    sector.radius1 = (int)Math.Round(double.Parse(t.Substring(0, 6)) * ConvertionFactors.NAUTICMILES2METERS / 1000 / 100) * 100;
                                    sector.radius2 = (int)Math.Round(double.Parse(t.Substring(6, 6)) * ConvertionFactors.NAUTICMILES2METERS / 1000 / 100) * 100;
                                    sector.directionFrom = int.Parse(t.Substring(12, 3));
                                    sector.directionTo = int.Parse(t.Substring(15, 3));
                                }
                                catch {
                                    task.SetSector(task.Count - 1, new SectorDefinition(AppContents.GetDefaultSector(TurnpointTypes.Turnpoint)));
                                }
                            }
                            break;
                        case "TE":
                            if (nrWayointsInTask != nrWaypointsRead) {
                                throw new Exception("Invalid task definition!\n" + nrWayointsInTask.ToString() + " waypoints defined, " +
                                    nrWaypointsRead.ToString() + " found!");
                            }
                            else {
                                // append takeoff and landing, if not in task
                                if (!takeoff) {
                                    task.Insert(1, task[0]);
                                }

                                if (!landing) {
                                    task.Add(task[task.Count - 1]);
                                }
                                tasks.Add(task);
                            }
                            break;
                        }
                    }
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                success = false;
            }
            finally {
                if (RecorderBase.DownloadTasksThreadNotify != null) {
                    RecorderBase.DownloadTasksThreadNotify(success);
                }
            }
        }

        protected override void DownloadWaypoints() {
            currentCatalogue = new WaypointCatalog("SoaringPilot" + SerialNo);
            FileHandlerBase fh = null;
            bool success = false;
            string[] tokens;
            WayPoint wp;

            IsInterrupted = false;
            try {
                ArrayList file = ReadFile("waypoints.cup");
                success = file.Count > 1;
                if (success) {
                    //twiddle out the file structure, cai or seeyou
                    foreach (string line in file) {
                        tokens = line.Split(',');
                        if (tokens.Length >= 11) {
                            fh = new SeeYouFileHandler();
                            break;
                        }
                        else if (tokens.Length >= 6) {
                            fh = new CAIAndWinPilotFileHandler();
                            break;
                        }
                    }
                    if (fh == null) {
                        throw new Exception("Unknown file structure");
                    }

                    foreach (string line in file) {
                        if ((wp = fh.ParseLine(line)) != null) {
                            currentCatalogue.Add(wp);
                        }
                    }
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.Message, "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                success = false;
            }
            finally {
                if (RecorderBase.DownloadWaypointsThreadNotify != null) {
                    RecorderBase.DownloadWaypointsThreadNotify(success);
                }
            }
        }

        protected override void UploadSUAs() {
            bool success = false;
            ArrayList file;
            IsInterrupted = false;
            SUAFileHandlerTimNewport fileHandler = new SUAFileHandlerTimNewport();
            try {
                file = fileHandler.WriteFile(suas);
                WriteFile(file, "suadata.sua");
                success = true;
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                success = false;
            }
            finally {
                if (RecorderBase.UploadSUAsThreadNotify != null) {
                    RecorderBase.UploadSUAsThreadNotify(success);
                }
            }
        }

        protected override void UploadTasks() {
            // TODO:  Add SoaringPilot.UploadTasks implementation
            //    ** -------------------------------------------------------------
            //    **      SOARINGPILOT Version 1.9.7 Tasks
            //    **      Date: 13 Feb 2004
            //    ** -------------------------------------------------------------
            //    TS,TASK002,6,TL
            //    TW,48:00:42.00N,010:06:07.00E,2,1903F,Tannhm,Tannhm
            //    TW,48:00:42.00N,010:06:07.00E,2,1903F,Tannhm,Tannhm
            //    TW,48:06:43.00N,009:45:49.00E,2,1903F,Bibrch,Bibrch
            //    TW,48:14:10.00N,010:08:19.00E,2,1680F,Illrts,Illrts,Area declaration
            //    TW,48:00:42.00N,010:06:07.00E,2,1903F,Tannhm,Tannhm
            //    TW,48:00:42.00N,010:06:07.00E,2,1903F,Tannhm,Tannhm
            //    TE
            bool success = false;
            string lat, lon, area;
            int tmp, i;
            SoaringPilotFlags type;
            WayPoint w;
            SectorDefinition sector;
            ArrayList file = new ArrayList();
            IsInterrupted = false;
            try {
                foreach (Task t in tasks) {
                    file.Add(string.Format("TS,{0},{1},{2}", t.Name, t.Count, t.Count > 3 ? "TL" : ""));
                    for (i = 0; i < t.Count; i++) {
                        w = t[i];
                        sector = t.GetSector(i);
                        tmp = Math.Abs(w.Latitude);
                        lat = string.Format("{0:00}:{1:00}:{2:00}{3}", tmp / 360000, (tmp % 360000) / 6000, (tmp % 6000) / 100, w.Latitude <= 0 ? "N" : "S");
                        tmp = Math.Abs(w.Longitude);
                        lon = string.Format("{0:000}:{1:00}:{2:00}{3}", tmp / 360000, (tmp % 360000) / 6000, (tmp % 6000) / 100, w.Longitude < 0 ? "W" : "E");

                        if (sector.sectorType == SectorTypes.Area) {
                            // write area info
                            // 005998005329101202
                            // |     |     |  |
                            type = SoaringPilotFlags.Area;
                            area = string.Format(",{0:000000}{1:000000}{2:000}{3:000}", (int)(sector.radius1 / ConvertionFactors.NAUTICMILES2METERS * 1000),
                                (int)(sector.radius2 / ConvertionFactors.NAUTICMILES2METERS * 1000), sector.directionFrom, sector.directionTo);
                        }
                        else {
                            area = "";
                            switch (w.type) {
                            case WayPointTypeId.Airport:
                            case WayPointTypeId.Glidersite:
                                type = SoaringPilotFlags.Airport;
                                break;
                            case WayPointTypeId.Landmark:
                                type = SoaringPilotFlags.Landmark;
                                break;
                            case WayPointTypeId.Outlanding:
                                type = SoaringPilotFlags.Outlanding;
                                break;
                            default:
                                type = SoaringPilotFlags.Turnpoint;
                                break;
                            }

                            if (w.landable) {
                                type |= SoaringPilotFlags.Landable;
                            }
                        }
                        file.Add(string.Format("TW,{0},{1},{2},{3},{4},{5}{6}", lat, lon, (int)type, w.Elevation != -1 ? w.Elevation.ToString() + "M" : "", w.shortName, w.longName, area));
                    }
                    file.Add("TE");
                }
                WriteFile(file, "tasks.spt");
                success = true;
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                success = false;
            }
            finally {
                if (RecorderBase.UploadTasksThreadNotify != null) {
                    RecorderBase.UploadTasksThreadNotify(success);
                }
            }
        }

        protected override void UploadWaypoints() {
            SeeYouFileHandler fh = new SeeYouFileHandler();
            bool success = true;

            IsInterrupted = false;
            try {
                if (currentCatalogue != null) {
                    ArrayList file = new ArrayList(currentCatalogue.Count);

                    foreach (WayPoint w in currentCatalogue.VisibleWaypoints) {
                        file.Add(fh.BuildLine(w));
                    }

                    WriteFile(file, "waypoints.cup");
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                success = false;
            }
            finally {
                if (RecorderBase.UploadWaypointsThreadNotify != null) {
                    RecorderBase.UploadWaypointsThreadNotify(success);
                }
            }
        }

        #endregion

        #region Attributes
        public override string Description {
            get {
                return string.Format(@"Adapter for SoaringPilot PDA Software

This is a file based adapter.

You can download the waypoints either in Cambridge or in SeeYou format. The latter may contain more details.
Upload is in general in SeeYou format.

Download procedure  : None, just select the files from the file system.
Upload procedure    : None, just select the files from the file system.
") + base.OptionsString;
            }
        }
        #endregion
    }
}
