// name   : WaypointCatalog.cs
// author : Harald Maier
// date   : 17.10.2003
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.Collections;
using System.Drawing;
using System.Drawing.Printing;
using System.IO;
using System.Windows.Forms;

using SoaringDotNET.Data;
using SoaringDotNET.Dialogs;
using SoaringDotNET.FileFormats;

namespace SoaringDotNET.GUI {
    public enum CatalogOperation {Add, Replace, Abort, Update};

    /// <summary>
	/// 
	/// </summary>
	public class WaypointCatalog : SoaringDotNET.GUI.MapElement
	{
        private ArrayList waypoints;
        private ArrayList visibleWaypoints;
        private Hashtable waypointsByName;
        private Hashtable waypointsByShortName;
        private bool dirty;
        private string fileName;
        private string catalogName;
        private DateTime lastModification;
        private WayPointFilter filter;
        private Font font;
        private enum OverwriteExisting {Check, Yes, No, Add};
        private OverwriteExisting checkOverwrite;
        private int pagenum;
        private WayPointSorter sorter;

        public WaypointCatalog() : this(null) {}
        public WaypointCatalog(string name) :
            base(MapElementTypes.WaypointCatalog)
		{
			// 
			// TODO: Add constructor logic here
			//

            waypointsByName = new Hashtable();
            waypointsByShortName = new Hashtable();
            waypoints = new ArrayList();
            visibleWaypoints = new ArrayList();

            dirty = false;
            checkOverwrite = OverwriteExisting.Check;
            lastModification = DateTime.MinValue;
            font = new Font("Arial", 7.0f);
            if (name != null) {
                catalogName = name;
            }

            filter = new WayPointFilter();
            sorter = new WayPointSorter();
        }


        #region Public Functions
        public void Load(string file, ArrayList tasks) {
            FileInfo info = new FileInfo(file);
            this.fileName = file;
            this.catalogName = info.Name;
            this.lastModification = info.LastWriteTime;

            checkOverwrite = OverwriteExisting.Check;
            using (StreamReader inStream = new StreamReader(file, true)) {
                switch (info.Extension.ToLower()) {
                case ".dat":
                    ImportCAIAndWinPilot(inStream);
                    break;
                case ".dbt":
                    ImportVolkslogger(inStream);
                    break;
                case ".cup":
                    ImportSeeYou(inStream, tasks);
                    break;
                }
            }

            // force update
            Filter = filter;
            Dirty = false;
        }

        public void Merge(string file) {
            FileInfo info = new FileInfo(file);

            checkOverwrite = OverwriteExisting.Check;
            using (StreamReader inStream = new StreamReader(file)) {
                switch (info.Extension.ToLower()) {
                case ".dat":
                    ImportCAIAndWinPilot(inStream);
                    break;
                case ".dbt":
                    ImportVolkslogger(inStream);
                    break;
                case ".cup":
                    ImportSeeYou(inStream, null);
                    break;
                }

                // force update
                Dirty = true;
            }
        }


        public void Save(ArrayList tasks, string defaultWaypointPath) {
            if (dirty) {
                if (fileName == null) {
                    SaveAs(tasks, defaultWaypointPath);
                }
                else {
                    SaveAs(fileName, tasks);
                }
            }
        }

        public void SaveAs(ArrayList tasks, string defaultWaypointPath) {
            SaveFileDialog fd = new SaveFileDialog();
            fd.CheckPathExists = true;
            fd.FileName = catalogName;
            fd.OverwritePrompt = true;
            fd.ValidateNames = true;
            fd.InitialDirectory = defaultWaypointPath;
            fd.Filter = "All known Catalogs|*.dbt;*.cup;*.dat|Volkslogger (*.dbt)|*.dbt|SeeYou (*.cup)|*.cup|WinPilot/CAI (*.dat)|*.dat";
            FileInfo fi = new FileInfo(catalogName);

            switch (fi.Extension.ToLower()) {
            case ".dat":
                fd.FilterIndex = 4;
                break;
            case ".dbt":
                fd.FilterIndex = 2;
                break;
            case ".cup":
                fd.FilterIndex = 3;
                break;
            default:
                fd.FilterIndex = 1;
                break;
            }
            if (fd.ShowDialog() == DialogResult.OK) {
                SaveAs(fd.FileName, tasks);
            }
        }

        public void SaveAs(string fileName, ArrayList tasks) {
            this.fileName = fileName;
            FileInfo info = new FileInfo(fileName);
            using (StreamWriter outStream = new StreamWriter(fileName)) {
                switch (info.Extension.ToLower()) {
                case ".dat":
                    ExportCAIAndWinPilot(outStream);
                    break;
                case ".dbt":
                    ExportVolkslogger(outStream);
                    break;
                case ".cup":
                    ExportSeeYou(outStream);
                    break;
                default:
                    MessageBox.Show("Not a valid waypoint filetype '" + info.Extension.ToLower() + "'", "Error", MessageBoxButtons.OK, MessageBoxIcon.Stop);
                    break;
                }
            }

            this.catalogName = info.Name;
            dirty = false;
            this.lastModification = info.LastWriteTime;
        }

        public CatalogOperation Add(WayPoint wp) {
            CatalogOperation res = CatalogOperation.Add;
            if (LongNameInCatalog(wp.longName)) {
                if (checkOverwrite == OverwriteExisting.Check) {
                    ConfirmMessageBox m = new ConfirmMessageBox(wp.longName + " is already in catalog " + catalogName + 
                        "!\nWhat would you like to do?");
                    m.ShowDialog();
                    
                    switch (m.Result) {
                    case ConfirmResults.Replace:
                        // replace old one
                        waypoints.Remove(waypointsByName[wp.longName]);
                        waypointsByShortName.Remove(((WayPoint)waypointsByName[wp.longName]).shortName);
                        visibleWaypoints.Remove(waypointsByName[wp.longName]);
                        waypoints.Add(wp);
                        if (!filter.IsSet || filter.IsVisible(wp)) {
                            visibleWaypoints.Add(wp);
                        }
                        waypointsByName[wp.longName] = wp;
                        GenerateShortName(wp);
                        res = CatalogOperation.Replace;
                        Dirty = true;
                        break;
                    case ConfirmResults.ReplaceAll:
                        // accept all further
                        checkOverwrite = OverwriteExisting.Yes;
                        goto case ConfirmResults.Replace;
                    case ConfirmResults.Skip:
                        res = CatalogOperation.Abort;
                        break;
                    case ConfirmResults.SkipAll:
                        checkOverwrite = OverwriteExisting.No;
                        goto case ConfirmResults.Skip;
                    case ConfirmResults.Add:
                        res = CatalogOperation.Add;
                        GenerateLongName(wp);
                        waypoints.Add(wp);
                        waypointsByName[wp.longName] = wp;
                        if (!filter.IsSet || filter.IsVisible(wp)) {
                            visibleWaypoints.Add(wp);
                        }
                        GenerateShortName(wp);
                        Dirty = true;
                        break;
                    case ConfirmResults.AddAll:
                        checkOverwrite = OverwriteExisting.Add;
                        goto case ConfirmResults.Add;
                    }
                }
                else if (checkOverwrite == OverwriteExisting.Yes) {
                    // replace old one
                    waypoints.Remove(waypointsByName[wp.longName]);
                    waypointsByShortName.Remove(((WayPoint)waypointsByName[wp.longName]).shortName);
                    visibleWaypoints.Remove(waypointsByName[wp.longName]);
                    waypoints.Add(wp);
                    if (!filter.IsSet || filter.IsVisible(wp)) {
                        visibleWaypoints.Add(wp);
                    }
                    waypointsByName[wp.longName] = wp;
                    GenerateShortName(wp);
                    res = CatalogOperation.Replace;
                }
                else if (checkOverwrite == OverwriteExisting.Add) {
                    GenerateLongName(wp);
                    waypoints.Add(wp);
                    waypointsByName[wp.longName] = wp;
                    if (!filter.IsSet || filter.IsVisible(wp)) {
                        visibleWaypoints.Add(wp);
                    }
                    GenerateShortName(wp);
                    Dirty = true;
                    res = CatalogOperation.Add;
                }
            }
            else {
                // add new one, store index in hashtable
                waypoints.Add(wp);
                waypointsByName[wp.longName] = wp;
                if (!filter.IsSet || filter.IsVisible(wp)) {
                    visibleWaypoints.Add(wp);
                }
                GenerateShortName(wp);
                Dirty = true;
            }
            return res;
        }

        public CatalogOperation Update(WayPoint oldWp, WayPoint newWp) {
            CatalogOperation res = CatalogOperation.Update;
            // check for duplicate if name changes
            if (this[newWp.longName] != null && oldWp.longName != newWp.longName) {
                ConfirmMessageBox m = new ConfirmMessageBox(newWp.longName + " is already in catalog " + catalogName + 
                    "!\nWhat would you like to do?");
                m.ShowDialog();
                        
                switch (m.Result) {
                case ConfirmResults.Replace:
                case ConfirmResults.ReplaceAll:
                    // remove old waypoint and old name ref
                    waypoints.Remove(oldWp);
                    waypoints.Remove(this[newWp.longName]);
                    visibleWaypoints.Remove(this[newWp.longName]);
                    waypoints.Add(newWp);
                    if (!filter.IsSet || filter.IsVisible(newWp)) {
                        visibleWaypoints.Add(newWp);
                    }
                    waypointsByName.Remove(oldWp.longName);
                    waypointsByShortName.Remove(oldWp.shortName);
                    waypointsByName[newWp.longName] = newWp;
                    GenerateShortName(newWp);
                    res = CatalogOperation.Replace;
                    Dirty = true;
                    break;
                case ConfirmResults.Skip:
                case ConfirmResults.SkipAll:
                    res = CatalogOperation.Abort;
                    break;
                case ConfirmResults.Add:
                case ConfirmResults.AddAll:
                    GenerateLongName(newWp);
                    waypoints.Add(newWp);
                    if (!filter.IsSet || filter.IsVisible(newWp)) {
                        visibleWaypoints.Add(newWp);
                    }
                    waypointsByName.Add(newWp.longName, newWp);
                    GenerateShortName(newWp);
                    Dirty = true;
                    res = CatalogOperation.Add;
                    break;
                }
            }
            else {
                waypoints.Remove(oldWp);
                visibleWaypoints.Remove(oldWp);
                waypoints.Add(newWp);
                if (!filter.IsSet || filter.IsVisible(newWp)) {
                    visibleWaypoints.Add(newWp);
                }
                waypointsByName.Remove(oldWp.longName);
                waypointsByName.Add(newWp.longName, newWp);
                waypointsByShortName.Remove(oldWp.shortName);
                GenerateShortName(newWp);
                Dirty = true;
            }
            return res;
        }

        public void Remove(WayPoint wp) {
            waypoints.Remove(wp);
            waypointsByName.Remove(wp.longName);
            waypointsByShortName.Remove(wp.shortName);
            visibleWaypoints.Remove(wp);
            Dirty = true;
        }

        /// <summary>
        /// Remove cataloge from disk
        /// </summary>
        public void Destroy() {
            try {
                if (fileName != null) {
                    File.Delete(fileName);
                }
            }
            catch(Exception ex) {
                MessageBox.Show("Error occured while deleting catalog\n\n" + fileName + "\n\n" +
                    ex.Message + "\n\n" + ex.StackTrace, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        public void Clear() {
            waypoints.Clear();
            waypointsByName.Clear();
            waypointsByShortName.Clear();
            visibleWaypoints.Clear();
            dirty = true;
        }

        public IEnumerator GetEnumerator() {
            return waypoints.GetEnumerator();
        }

        public object [] ToArray() {
            return waypoints.ToArray();
        }

        public override void Draw(System.Drawing.Graphics g, WgsArea view, int width, int height) {
            // TODO:  Add WaypointCatalog.Draw implementation
            long lat, lon;
            long latRange, lonRange;
            long minLat, minLon;
            long diff;
            int i;

            minLat = view.TopLat;
            latRange = view.Height;
            minLon = view.LeftLon;
            lonRange = view.Width;

            mapPos.Clear();
            i = 0;
            Brush landableBrush = new SolidBrush(AppContents.landableColor.DrawColor);
            Brush outlandingBrush = new SolidBrush(AppContents.outlandingColor.DrawColor);
            Brush notLandableBrush = new SolidBrush(AppContents.notLandableColor.DrawColor);

            foreach (WayPoint p in visibleWaypoints) {
                if (view.PointInArea(p)) {
                    lat = (p.Latitude - minLat) * height / latRange;
                    diff = p.Longitude - minLon;
                    if (diff <= -64800000) {
                        diff += 129600000;
                    }
                    else if (diff > 64800000) {
                        diff -= 129600000;
                    }
                    lon = diff * width / lonRange;
                    if (p.landable) {
                        if (p.type != WayPointTypeId.Outlanding) {
                            g.FillEllipse(landableBrush, lon - 3, lat - 3, 7, 7);
                        }
                        else {
                            g.FillEllipse(outlandingBrush, lon - 3, lat - 3, 7, 7);
                        }
                    }
                    else {
                        g.FillEllipse(notLandableBrush, lon - 3, lat - 3, 7, 7);
                    }
                    g.DrawString(p.longName, font, Brushes.Black, lon + 5, lat - 10);
                    mapPos[i] = new Point((int)lon, (int)lat);
                }
                i++;
            }
        }

        public bool LongNameInCatalog(string name) {
            return waypointsByName.Contains(name);
        }

        public override string ToString() {
            // TODO:  Add WaypointCatalog.ToString implementation
            return catalogName;
        }

        #endregion

        #region Private Functions
        private void ExportVolkslogger(StreamWriter f) {
            VolksloggerFileHandler h = new VolksloggerFileHandler();
            h.Save(f, this);
        }
        
        private void ImportVolkslogger(StreamReader f) {
            VolksloggerFileHandler h = new VolksloggerFileHandler();
            h.Load(f, this);
        }

        private void ExportSeeYou(StreamWriter f) {
            SeeYouFileHandler h = new SeeYouFileHandler();
            switch (MessageBox.Show("Save with tasks.", "SeeYou cup file", MessageBoxButtons.YesNoCancel, MessageBoxIcon.Question, MessageBoxDefaultButton.Button1)) {
            case DialogResult.Yes:
                h.Save(f, this, AppContents.tasks);
                break;
            case DialogResult.No:
                h.Save(f, this);
                break;
            }
        }

        private void ImportSeeYou(StreamReader f, ArrayList tasks) {
            SeeYouFileHandler h = new SeeYouFileHandler();
            h.Load(f, this, tasks);            
        }

        private void ExportCAIAndWinPilot(StreamWriter f) {
            CAIAndWinPilotFileHandler h = new CAIAndWinPilotFileHandler();
            h.Save(f, this);
        }

        private void ImportCAIAndWinPilot(StreamReader f) {
            CAIAndWinPilotFileHandler h = new CAIAndWinPilotFileHandler();
            h.Load(f, this);
        }

        private void GenerateLongName(WayPoint wp) {
            string t = wp.longName;
            int i = 1;
            while (waypointsByName.ContainsKey(t)) {
                t = string.Format("{0}{1}", wp.longName, i);
                i++;
            }
            wp.longName = t;
        }

        private void GenerateShortName(WayPoint wp) {
            string t = wp.longName.Replace(" ", "");
            int i = 0;
            t = t.Substring(0, Math.Min(6, t.Length)).ToUpper();
            while (waypointsByShortName.ContainsKey(t)) {
                t = t.Substring(0, Math.Min(t.Length - i.ToString().Length, t.Length)) + i.ToString();
                i++;
            }
            wp.shortName = t;
            waypointsByShortName[t] = null;
        }

        #endregion

        #region Attributes
        public bool Dirty {
            get {
                return dirty;
            }
            set {
                dirty = value;
            }
        }

        public WayPoint this[int pos] {
            get {
                return (WayPoint)waypoints[pos];
            }
        }

        public WayPoint this[string name] {
            get {
                return (WayPoint)waypointsByName[name];
            }
        }

        public int Count {
            get {
                return waypoints.Count;
            }
        }

        public int VisibleWaypointsCount {
            get {
                return visibleWaypoints.Count;
            }
        }

        public ArrayList VisibleWaypoints {
            get {
                return visibleWaypoints;
            }
        }

        public string Name {
            get {
                return catalogName;
            }
            set {
                catalogName = value;
                fileName = null;
            }
        }

        public string FileName {
            get {
                return fileName;
            }
        }

        public string LastModification {
            get {
                return lastModification == DateTime.MinValue ? "never" : lastModification.ToShortDateString() + " " + lastModification.ToLongTimeString();
            }
        }

        public WayPointFilter Filter {
            get {
                return filter;
            }
            set {
                filter = value;
                visibleWaypoints = new ArrayList(Count);
                if (filter.IsSet) {
                    foreach (WayPoint wp in waypoints) {
                        if (filter.IsVisible(wp)) {
                            visibleWaypoints.Add(wp);
                        }
                    }
                }
                else {
                    visibleWaypoints.AddRange(waypoints);
                }
            }
        }
        #endregion

        public void Print() {
            PrintDialog pd = new PrintDialog();
            PrintDocument doc = new PrintDocument();
            doc.DocumentName = Name;
            doc.DefaultPageSettings.Landscape = true;
            doc.PrintPage += new PrintPageEventHandler(PrintHandler);
            pd.Document = doc;
            
            if (pd.ShowDialog() == DialogResult.OK) {
                visibleWaypoints.Sort(sorter);
                pagenum = 1;
                doc.Print();
            }
        }

        private void PrintHandler(object sender, PrintPageEventArgs ev) {
            Graphics g = ev.Graphics;
            float x, y;
            string s;
            int linesPerPage;
            int start, stop;
            SizeF stringSize = new SizeF(0, 0);
            RectangleF rect = new RectangleF(0, 0, 0, 0);
            Font font;
            WayPoint wp;
            StringFormat rightAlign = new StringFormat();
            rightAlign.Alignment = StringAlignment.Far;

            x = 25;
            y = 25;
            if (pagenum == 1) {
                font = new Font("Times New Roman", 12);
                s = FileName;
                stringSize = g.MeasureString(s, font);
                g.DrawString(s, font, Brushes.Black, x, y);
                s = DateTime.Today.ToShortDateString();
                stringSize = g.MeasureString(s, font);
                g.DrawString(s, font, Brushes.Black, ev.PageBounds.Width - 50 - stringSize.Width, y);
                y += stringSize.Height;
                x = 25;
                font = new Font("Times New Roman", 8);
                s = string.Format("{0} items", VisibleWaypointsCount);
                if (filter.IsSet) {
                    s += string.Format(" of {0}", Count);
                }
                stringSize = g.MeasureString(s, font);
                g.DrawString(s, font, Brushes.Red, x, y);
                y += stringSize.Height;

                g.DrawLine(new Pen(Brushes.DarkGray, 3), x, y, ev.PageBounds.Width - 50, y);

                y += 5;
            }
            
            font = new Font("Times New Roman", 9);
            stringSize.Height = font.GetHeight(g);

            rect.Y = y;
            rect.Height = stringSize.Height;
            
            linesPerPage = (int)Math.Floor((ev.PageBounds.Height - y - 50) / stringSize.Height);
            stop = Math.Min(pagenum * linesPerPage, VisibleWaypointsCount);

            rect.X = x;
            rect.Width = 150;
            g.DrawString("Name", font, Brushes.Black, rect);
            rect.X += rect.Width;
            rect.Width = 75;
            g.DrawString("Latitude", font, Brushes.Black, rect);
            rect.X += rect.Width;
            rect.Width = 80;
            g.DrawString("Longitude", font, Brushes.Black, rect);
            rect.X += rect.Width;
            rect.Width = 65;
            g.DrawString("Type", font, Brushes.Black, rect);
            rect.X += rect.Width;
            rect.Width = 60;
            g.DrawString("Elevation", font, Brushes.Black, rect, rightAlign);
            rect.X += rect.Width + 10;
            rect.Width = 60;
            g.DrawString("Frequency", font, Brushes.Black, rect, rightAlign);
            rect.X += rect.Width + 10;
            rect.Width = 50;
            g.DrawString("Runway", font, Brushes.Black, rect);
            rect.X += rect.Width;
            rect.Width = 60;
            g.DrawString("Length", font, Brushes.Black, rect, rightAlign);
            rect.X += rect.Width + 10;
            rect.Width = 60;
            g.DrawString("Surface", font, Brushes.Black, rect);
            rect.X += rect.Width;
            rect.Width = 80;
            g.DrawString("Country", font, Brushes.Black, rect);
            rect.X += rect.Width;
            rect.Width = 100;
            g.DrawString("Description", font, Brushes.Black, rect);
            rect.X += rect.Width;
            rect.Width = ev.PageBounds.Width - 50 - rect.X;
            g.DrawString("Comment", font, Brushes.Black, rect);
            rect.Y += stringSize.Height;

            for (start = (pagenum - 1) * linesPerPage; start < stop; start++) {
                wp = (WayPoint)VisibleWaypoints[start];
                rect.X = x;
                rect.Width = 150;
                g.DrawString(wp.longName, font, Brushes.Black, rect);
                rect.X += rect.Width;
                rect.Width = 75;
                g.DrawString(wp.ToStringLat(), font, Brushes.Black, rect);
                rect.X += rect.Width;
                rect.Width = 80;
                g.DrawString(wp.ToStringLong(), font, Brushes.Black, rect);
                rect.X += rect.Width;
                rect.Width = 65;
                g.DrawString(wp.type.ToString(), font, Brushes.Black, rect);
                rect.X += rect.Width;
                rect.Width = 60;
                g.DrawString(wp.Elevation != -1 ? string.Format("{0} m", wp.Elevation) : "", font, Brushes.Black, rect, rightAlign);
                rect.X += rect.Width + 10;
                rect.Width = 60;
                g.DrawString(wp.frequency, font, Brushes.Black, rect, rightAlign);
                rect.X += rect.Width + 10;
                rect.Width = 50;
                g.DrawString(wp.runway, font, Brushes.Black, rect);
                rect.X += rect.Width;
                rect.Width = 60;
                g.DrawString(wp.landable == true && wp.length != -1 ? string.Format("{0} m", wp.length) : "", font, Brushes.Black, rect, rightAlign);
                rect.X += rect.Width + 10;
                rect.Width = 60;
                g.DrawString(wp.landable == true ? wp.surface.ToString() : "", font, Brushes.Black, rect);
                rect.X += rect.Width;
                rect.Width = 80;
                g.DrawString(wp.country.ToString(), font, Brushes.Black, rect);
                rect.X += rect.Width;
                rect.Width = 100;
                g.DrawString(wp.description, font, Brushes.Black, rect);
                rect.X += rect.Width;
                rect.Width = ev.PageBounds.Width - 50 - rect.X;
                g.DrawString(wp.comment, font, Brushes.Black, rect);
                rect.Y += stringSize.Height;
            }
            pagenum++;
            ev.HasMorePages = start < VisibleWaypointsCount - 1;
        }
    }
}
