// name   : WgsArea.cs
// author : Harald Maier
// date   : 13.11.2003
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.Drawing;

namespace SoaringDotNET.Data
{
    interface IWgsArea {
        Point TopLeft {
            get;
            set;
        }
        Point TopRight {
            get;
            set;
        }
        Point BottomRight {
            get;
            set;
        }
        Point BottomLeft {
            get;
            set;
        }
        int TopLat {
            get;
            set;
        }
        int BottomLat {
            get;
            set;
        }
        int LeftLon {
            get;
            set;
        }
        int RightLon {
            get;
            set;
        }
        int Width {
            get;
        }
        int Height {
            get;
        }
    }

	/// <summary>
	/// 
	/// </summary>
	public class WgsArea : IWgsArea
	{
        private int bottomLat = 0;
        private int topLat = 0;
        private int leftLon = 0 ;
        private int rightLon = 0;
        private bool eastWest = false;

        public WgsArea() : this(0, 0, 0, 0) {}
        public WgsArea(WgsArea src) : this(src.TopLat, src.LeftLon, src.BottomLat, src.RightLon){}
        public WgsArea(int topLat, int leftLon, int bottomLat, int rightLon) {
            this.topLat = topLat;
            this.leftLon = leftLon;
            this.bottomLat = bottomLat;
            this.rightLon = rightLon;
        }

        #region Public Functions
        public bool PointInArea(Point p) {
            bool isTrue;
            if (leftLon < rightLon) {
                isTrue = (
                    p.X >= leftLon && p.X <= rightLon && p.Y >= topLat && p.Y <= bottomLat);
            }
            else {
                isTrue = (
                    (p.X >= leftLon || p.X <= rightLon) && p.Y >= topLat && p.Y <= bottomLat);
            }
            return isTrue;
        }

        public bool PointInArea(WgsPoint p) {
            bool isTrue;
            if (leftLon < rightLon) {
                isTrue = (
                    p.Longitude >= leftLon && p.Longitude <= rightLon && p.Latitude >= topLat && p.Latitude <= bottomLat);
            }
            else {
                isTrue = (
                    (p.Longitude >= leftLon || p.Longitude <= rightLon) && p.Latitude >= topLat && p.Latitude <= bottomLat);
            }
            return isTrue;
        }

        public bool AreaIntersect(WgsArea a) {
            bool isTrue;
            isTrue = PointInArea(a.TopLeft) || PointInArea(a.TopRight) || PointInArea(a.BottomLeft) || PointInArea(a.BottomRight);
            if (!isTrue) {
                if (leftLon <= rightLon) {
                    if (a.LeftLon <= a.RightLon) {
                        // completly covered by a
                        if(a.LeftLon <= leftLon && a.RightLon >= rightLon && a.TopLat <= topLat && a.BottomLat >= bottomLat) {
                            isTrue = true;
                        }
                        else if (a.TopLat <= topLat && a.BottomLat >= bottomLat) {
                            if ((a.LeftLon >= leftLon && a.LeftLon <= rightLon) ||
                                (a.RightLon >= leftLon && a.RightLon <= rightLon)) {
                                isTrue = true;
                            }
                        }
                        else if (a.LeftLon <= leftLon && a.RightLon >= rightLon) {
                            if ((a.TopLat >= topLat && a.TopLat <= bottomLat) ||
                                (a.BottomLat >= topLat && a.BottomLat <= bottomLat)) {
                                isTrue = true;
                            }
                        }
                    }
                    else {
                        // a cover E and W
                        // completly covered by a?
                        if ((a.LeftLon <= leftLon || a.RightLon >= rightLon) && a.TopLat <= topLat && a.BottomLat >= bottomLat) {
                            isTrue = true;
                        }
                        else if (a.TopLat <= topLat && a.BottomLat >= bottomLat) {
                            if ((a.LeftLon >= leftLon && a.LeftLon <= rightLon) ||
                                (a.RightLon >= leftLon && a.RightLon <= rightLon)) {
                                isTrue = true;
                            }
                        }
                        else if ((a.LeftLon >= leftLon && a.RightLon >= rightLon) ||
                            (a.LeftLon <= leftLon && a.RightLon <= rightLon)) {
                            if ((a.TopLat >= topLat && a.TopLat <= bottomLat) ||
                                (a.BottomLat >= topLat && a.BottomLat <= bottomLat)) {
                                isTrue = true;
                            }
                        }
                    }
                }
                else {
                    // a completly in E or W section
                    if (a.LeftLon <= a.RightLon) {
                        // completly covered by a ?
                        if(a.LeftLon <= leftLon && a.RightLon >= rightLon && a.TopLat <= topLat && a.BottomLat >= bottomLat) {
                            isTrue = true;
                        }
                        else if (a.TopLat <= topLat && a.BottomLat >= bottomLat) {
                            if ((a.LeftLon >= leftLon && a.LeftLon <= rightLon) ||
                                (a.RightLon >= leftLon && a.RightLon <= rightLon)) {
                                isTrue = true;
                            }
                        }
                        else if (a.LeftLon <= rightLon || a.RightLon >= leftLon) {
                            if ((a.TopLat >= topLat && a.TopLat <= bottomLat) ||
                                (a.BottomLat >= topLat && a.BottomLat <= bottomLat)) {
                                isTrue = true;
                            }
                        }
                    }
                    else {
                        // completly covered by a
                        if(a.LeftLon <= leftLon && a.RightLon >= rightLon && a.TopLat <= topLat && a.BottomLat >= bottomLat) {
                            isTrue = true;
                        }
                        else if ((a.TopLat <= topLat && a.BottomLat >= bottomLat) ||
                            (a.TopLat >= topLat && a.TopLat <= bottomLat) ||
                            (a.BottomLat >= topLat && a.BottomLat <= bottomLat)) {
                            // only lat must lie within view, lon by default due to W/E in area and view
                            isTrue = true;
                        }
                    }
                }
            }
            return isTrue;
        }

        public void Union(WgsArea other) {
            BottomLat = Math.Max(bottomLat, other.bottomLat);
            TopLat = Math.Min(topLat, other.topLat);
            ExtentLon(other.LeftLon, other.RightLon);
        }

        public void Clear() {
            bottomLat = leftLon = topLat = rightLon = 0;
            eastWest = false;
        }

        public void ExtentLon(int p1, int p2) {
            ExtentLon(p1);
            ExtentLon(p2);
        }

        public void ExtentLon(int p1) {
            if (!eastWest && (Math.Abs(leftLon - p1) > 64800000)) {
                if (p1 < 0) {
                    RightLon = p1;
                }
                else {
                    LeftLon = p1;
                }
                eastWest = true;
            }
            else {
                if (eastWest) {
                    if (p1 >= 0) {
                        LeftLon = Math.Min(leftLon, p1);
                    }
                    else {
                        RightLon = Math.Max(rightLon, p1);
                    }
                }
                else {
                    LeftLon = Math.Min(leftLon, p1);
                    RightLon = Math.Max(rightLon, p1);
                }
            }
        }
        #endregion

        #region IWgsArea Members

        public Point TopLeft {
            get {
                // TODO:  Add WgsArea.TopLeft getter implementation
                return new Point(leftLon, topLat);
            }
            set {
                // TODO:  Add WgsArea.TopLeft setter implementation
                LeftLon = value.X;
                TopLat = value.Y;
            }
        }

        public Point TopRight {
            get {
                // TODO:  Add WgsArea.TopLeft getter implementation
                return new Point(rightLon, topLat);
            }
            set {
                // TODO:  Add WgsArea.TopLeft setter implementation
                RightLon = value.X;
                TopLat = value.Y;
            }
        }

        public Point BottomRight {
            get {
                // TODO:  Add WgsArea.BottomRight getter implementation
                return new Point(rightLon, bottomLat);
            }
            set {
                // TODO:  Add WgsArea.BottomRight setter implementation
                RightLon = value.X;
                BottomLat = value.Y;
            }
        }

        public Point BottomLeft {
            get {
                // TODO:  Add WgsArea.BottomRight getter implementation
                return new Point(leftLon, bottomLat);
            }
            set {
                // TODO:  Add WgsArea.BottomRight setter implementation
                LeftLon = value.X;
                BottomLat = value.Y;
            }
        }

        public int TopLat {
            get {
                // TODO:  Add WgsArea.LeftLat getter implementation
                return topLat;
            }
            set {
                // TODO:  Add WgsArea.LeftLat setter implementation
                topLat = Math.Max(value, -32400000 /*90 N */);
                //if (topLat != 0 && bottomLat != 0 && topLat > bottomLat) {
                //    int tmp = topLat;
                //    topLat = bottomLat;
                //    bottomLat = tmp;
                //}
            }
        }

        public int BottomLat {
            get {
                // TODO:  Add WgsArea.RightLat getter implementation
                return bottomLat;
            }
            set {
                // TODO:  Add WgsArea.RightLat setter implementation
                bottomLat = Math.Min(value, 32400000/* 90 S*/);
                //if (topLat != 0 && bottomLat != 0 && topLat > bottomLat) {
                //    int tmp = topLat;
                //    topLat = bottomLat;
                //    bottomLat = tmp;
                //}
            }
        }

        public int LeftLon {
            get {
                // TODO:  Add WgsArea.LeftLon getter implementation
                return leftLon;
            }
            set {
                // TODO:  Add WgsArea.LeftLon setter implementation
                leftLon = value;
                if (leftLon <= -64800000) {
                    leftLon += 129600000;
                }
                else if (leftLon > 64800000) {
                    leftLon -= 129600000;
                }
            }
        }

        public int RightLon {
            get {
                // TODO:  Add WgsArea.RightLon getter implementation
                return rightLon;
            }
            set {
                // TODO:  Add WgsArea.RightLon setter implementation
                rightLon = value;
                if (rightLon <= -64800000) {
                    rightLon += 129600000;
                }
                else if (rightLon > 64800000) {
                    rightLon -= 129600000;
                }
            }
        }

        public int Width {
            get {
                if (rightLon >= leftLon) {
                    int diff = rightLon - leftLon;
                    if (diff > 64800000) {
                        diff = 129600000 - diff;
                    }
                    return diff;
                }
                else {
                    return 129600000 - Math.Abs(leftLon) - Math.Abs(rightLon);
                }
            }
        }

        public int Height {
            get {
                return Math.Abs(bottomLat - topLat);
            }
        }
        #endregion

        #region Attributes

        public bool IsEmpty {
            get {
                return (bottomLat == 0 && leftLon == 0 && topLat == 0 && rightLon == 0);
            }
        }

        public WgsArea Area {
            get {
                return new WgsArea(topLat, leftLon, bottomLat, rightLon);
            }
            set {
                LeftLon = value.LeftLon;
                RightLon = value.RightLon;
                // do NOT use Getter/Setter may accidently switch top/bottom if new area is outside old onw
                topLat = value.TopLat;
                bottomLat = value.bottomLat;
            }
        }

        public WgsPoint Center {
            get {
                int latCenter = (bottomLat + topLat) / 2;
                int lonCenter;
                if (leftLon <= rightLon) {
                    lonCenter = (leftLon + rightLon) / 2;
                }
                else {
                    lonCenter = (Width / 2) + leftLon;
                    if (lonCenter <= -64800000) {
                        lonCenter += 129600000;
                    }
                    else if (lonCenter > 64800000) {
                        lonCenter -= 129600000;
                    }
                }
                return new WgsPoint(latCenter, lonCenter);
            }
        }

        public bool EastWest {
            get {
                return eastWest;
            }
        }
        #endregion
    }
}
