// name   : WgsPoint.cs
// author : Harald Maier
// date   : 13.11.2003
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.Drawing;
using System.Text.RegularExpressions;

namespace SoaringDotNET.Data
{
	/// <summary>
	/// class WgsPoint
	/// represent 3D point on WGS sphere
	/// latitude <= 0 represent northern, > 0 southern  hemisphere
	/// longitude <= 0 represent western, > 0 estern hemisphere
	/// </summary>
	public class WgsPoint
	{
        static double RADIUS = 6378137.000; // WGS84
        private int latitude = 0;
        private int longitude = 0;
        private int elevation = -1;

		public WgsPoint() : this(0, 0, -1) {}
        public WgsPoint(int latitude, int longitude) : this(latitude, longitude, -1) {}
        public WgsPoint(int latitude, int longitude, int elevation) {
            this.Latitude = latitude;
            this.Longitude = longitude;
            this.Elevation = elevation;
        }

        #region Public Fuctions
        public static int RadToInternal(double rad) {
            return (int)(-rad * 64800000.0 / Math.PI);
        }

        public static double InternalToRad(int koord) {
            return -(double)koord * Math.PI / 64800000.0;
        }

        public static int FromString(string latOrLong, bool isLat) {
            int tmp = 0;
            Match m;
            Regex coord;
            GroupCollection coll;

            if (isLat) {
                coord = new Regex("([0-9]{2})([0-9]{2})'([0-9]{2})\"[ \t]*([N|S])");
            }
            else {
                coord = new Regex("([0-9]{3})([0-9]{2})'([0-9]{2})\"[ \t]*([W|E])");
            }

            m = coord.Match(latOrLong.ToUpper());
            if (m.Success) {
                coll = m.Groups;
                tmp = (int.Parse(coll[1].Value) * 360000) +
                    (int.Parse(coll[2].Value) * 6000) + 
                    (int.Parse(coll[3].Value) * 100);

                if (isLat && tmp > 32400000) {
                    tmp -= 64800000;
                }
                else if (!isLat && tmp > 64800000) {
                    tmp -= 129600000;
                }

                if (coll[4].Value == "N" || coll[4].Value == "W") {
                    tmp = -tmp;
                }
            }
            else {
                throw new Exception("malformed Lat or Long " + latOrLong);
            }
            return tmp;
        }

        public static int FromColonString(string latOrLong, bool isLat) {
            int tmp = 0;
            string []tt;
            char latLonChar;

            latOrLong = latOrLong.Trim();
            latLonChar = latOrLong[latOrLong.Length - 1];

            tt = latOrLong.Substring(0, latOrLong.Length - 2).Split(':');

            if (tt.Length >= 2) {
                tmp = int.Parse(tt[0]) * 360000;
                if (tt.Length == 2) {
                    tmp += (int)(double.Parse(tt[1], AppContents.ni) * 6000);
                }
                else {
                    tmp += (int.Parse(tt[1]) * 6000) + (int)(double.Parse(tt[2], AppContents.ni) * 100);
                }
                if (isLat && tmp > 32400000) {
                    tmp -= 64800000;
                }
                else if (!isLat && tmp > 64800000) {
                    tmp -= 129600000;
                }

                if (latLonChar == 'N' || latLonChar == 'W' || latLonChar == 'n' || latLonChar == 'w') {
                    tmp = -tmp;
                }
            }
            else {
                throw new Exception("malformed Lat or Long " + latOrLong);
            }
            return tmp;
        }

        public static int FromCompactString(string latOrLong, bool isLat) {
            int tmp = 0;
            Match m;
            Regex coord;
            GroupCollection coll;

            if (isLat) {
                coord = new Regex("([N|S])([0-9]{2})([0-9]{2})([0-9]{2})");
            }
            else {
                coord = new Regex("([W|E])([0-9]{3})([0-9]{2})([0-9]{2})");
            }

            m = coord.Match(latOrLong.ToUpper());
            if (m.Success) {
                coll = m.Groups;
                tmp = (int.Parse(coll[2].Value) * 360000) +
                    (int.Parse(coll[3].Value) * 6000) + 
                    (int.Parse(coll[4].Value) * 100);

                if (isLat && tmp > 32400000) {
                    tmp -= 64800000;
                }
                else if (!isLat && tmp > 64800000) {
                    tmp -= 129600000;
                }

                if (coll[1].Value == "N" || coll[1].Value == "W") {
                    tmp = -tmp;
                }
            }
            else {
                throw new Exception("malformed Lat or Long " + latOrLong);
            }
            return tmp;
        }

        // calc distance to next point in radians
        public double distance(WgsPoint to) {
            double lonDiff2, latDiff2;
            double lat1, lat2, lon1, lon2;

            lat1 = WgsPoint.InternalToRad(this.latitude);
            lon1 = WgsPoint.InternalToRad(this.longitude);
            lat2 = WgsPoint.InternalToRad(to.latitude);
            lon2 = WgsPoint.InternalToRad(to.longitude);

            latDiff2 = (lat1 - lat2) / 2.0;
            lonDiff2 = (lon1 - lon2) / 2.0;

            return 2.0 * Math.Asin(Math.Sqrt(Math.Pow(Math.Sin(latDiff2), 2.0) + 
                (Math.Cos(lat1) * Math.Cos(lat2) * Math.Pow(Math.Sin(lonDiff2), 2.0))));
        }

        // calc distance to next point in km
        public double distanceKM(WgsPoint to) {
            return Math.Round(RADIUS * distance(to) / 1000.0, 2);
        }

        // calc distance to next point in m
        public double distanceM(WgsPoint to) {
            return RADIUS * distance(to);
        }

        // calc distance to next point in nm
        public double distanceNM(WgsPoint to) {
            return (180.0 * 60.0) / Math.PI * distance(to);
        }

        // calc bearing to next point in radians
        public double bearing(WgsPoint to) {
            double trueCourse;
            double lat1, lat2, lon1, lon2;

            double dist = distance(to);

            if (dist < double.Epsilon) {
                trueCourse = 0.0;
            }
            else {
                lat1 = WgsPoint.InternalToRad(this.latitude);
                lon1 = WgsPoint.InternalToRad(this.longitude);
                lat2 = WgsPoint.InternalToRad(to.latitude);
                lon2 = WgsPoint.InternalToRad(to.longitude);

                trueCourse = Math.Acos((Math.Sin(lat2) - Math.Sin(lat1) * Math.Cos(dist)) / (Math.Sin(dist) * Math.Cos(lat1)));

                if (Math.Sin(lon2 - lon1) > 0.0) {
                    trueCourse = 2.0 * Math.PI - trueCourse;
                }

                if (trueCourse < double.Epsilon || lon2 == lon1) {
                    trueCourse = 0.0;
                }
            }
            return trueCourse;
        }

        // calc bearing to next point in deg
        public double bearingDeg(WgsPoint to) {
            return bearing(to) * 180.0 / Math.PI;
        }

        public WgsPoint PosOfDistAndBearing(double bearing, double dist) {
            double lat1, lat2, lon1, lon2;
            double tmp;

            lat1 = WgsPoint.InternalToRad(this.latitude);
            lon1 = WgsPoint.InternalToRad(this.longitude);

            dist = dist * 1000.0 / RADIUS;

            lat2 = Math.Asin(Math.Sin(lat1) * Math.Cos(dist) + Math.Cos(lat1) * Math.Sin(dist) * Math.Cos(bearing));
            tmp = Math.Atan2(Math.Sin(bearing) * Math.Sin(dist) * Math.Cos(lat1),
                Math.Cos(dist) - Math.Sin(lat1) * Math.Sin(lat1));

            lon2 = Math.IEEERemainder(lon1 - tmp + Math.PI, 2.0 * Math.PI) - Math.PI;

            return new WgsPoint(WgsPoint.RadToInternal(lat2), WgsPoint.RadToInternal(lon2));
        }

        public void Clear() {
            latitude = 0;
            longitude = 0;
            elevation = -1;
        }

        public string ToStringLat() {
            return WgsPoint.ToString(this.latitude, true);
        }

        public string ToStringLong() {
            return WgsPoint.ToString(this.longitude, false);
        }

        static public string ToString(int p, bool isLat) {
            string t;
            int deg, min, sec;
            int tmp;
            tmp = Math.Abs(p);

            deg = tmp / 360000;
            tmp = tmp % 360000;
            min = tmp / 6000;
            tmp = tmp % 6000;
            sec = (int)Math.Round(tmp / 100.0, 0);
            if (sec == 60) {
                min++;
                sec = 0;
                if (min == 60) {
                    deg++;
                    min = 0;
                }
            }
            if (isLat) {
                t = string.Format("{0:00}{1:00}'{2:00}\" {3}", deg, min, sec, p <= 0 ? "N" : "S");
            }
            else {
                t = string.Format("{0:000}{1:00}'{2:00}\" {3}", deg, min, sec, p < 0 ? "W" : "E");
            }
            return t;
        }

        static public string ToCompactString(int p, bool isLat) {
            string t;
            int deg, min, sec;
            int tmp;
            tmp = Math.Abs(p);

            deg = tmp / 360000;
            tmp = tmp % 360000;
            min = tmp / 6000;
            tmp = tmp % 6000;
            sec = (int)Math.Round(tmp / 100.0, 0);

            if (isLat) {
                t = string.Format("{3}{0:00}{1:00}{2:00}", deg, min, sec, p <= 0 ? "N" : "S");
            }
            else {
                t = string.Format("{3}{0:000}{1:00}{2:00}", deg, min, sec, p < 0 ? "W" : "E");
            }
            return t;
        }

        static public string ToColonString(int p, bool isLat) {
            string t;
            int deg, min, sec;
            int tmp;
            tmp = Math.Abs(p);

            deg = tmp / 360000;
            tmp = tmp % 360000;
            min = tmp / 6000;
            tmp = tmp % 6000;
            sec = (int)Math.Round(tmp / 100.0, 0);

            if (isLat) {
                t = string.Format(AppContents.ni, "{0:00}:{1:00}:{2:00} {3}", deg, min, sec, p <= 0 ? "N" : "S");
            }
            else {
                t = string.Format(AppContents.ni, "{0:000}:{1:00}:{2:00} {3}", deg, min, sec, p < 0 ? "W" : "E");
            }
            return t;
        }

        static public double GetSectorAngle(double w1, double w2) {
            double alpha = 0;
            if (w2 < w1) {
                alpha = (w1 - ((w1 - w2) / 2.0) + 180);
                if ((w1 - w2) < 180.0) {
                    alpha -= 180;
                }
            }
            else {
                alpha = (w1 + ((w2 - w1) / 2.0));
                if ((w2 - w1) > 180.0) {
                    alpha += 180;
                }
            }
            return alpha;
        }

        #endregion

        #region Attributes
        public int Latitude {
            get {
                return latitude;
            }
            set {
                latitude = Math.Min(Math.Max(value, -32400000 /*90 N */), 32400000 /*90 S */);
            }
        }

        public int Longitude {
            get {
                return longitude;
            }
            set {
                longitude = value;
                if (longitude <= -64800000) {
                    longitude += 129600000;
                }
                else if (longitude > 64800000) {
                    longitude -= 129600000;
                }
            }
        }

        public virtual int Elevation {
            get {
                return elevation;
            }
            set {
                elevation = value;
            }
        }
        #endregion
    }
}
