using System;
using System.Collections;
using System.Text;
using System.Windows.Forms;
using System.IO;

using SoaringDotNET.Data;
using SoaringDotNET.GUI;
using SoaringDotNET.FileFormats;

namespace SoaringDotNET.Recorders
{
    public class XCSoar : FileSystemRecorder {
        public XCSoar() {
            options = RecorderOptions.DownloadTasks |
                RecorderOptions.DownloadWaypoints |
                RecorderOptions.UploadTask |
                RecorderOptions.UploadWaypoints |
                RecorderOptions.UploadSUA;

            capacity.maxNrTasks = (int)CapacityNumbers.Unlimited;
            capacity.maxNrWaypoints = (int)CapacityNumbers.Unlimited;
            capacity.maxNrWaypointsPerTask = (int)CapacityNumbers.Unlimited;

            name = "XCSoar";
            ManufacurerCode = "XXX";
            ManufacurerLetter = "X";

        }

        #region Protected Functions
        protected override void DownloadTasks() {
            throw new System.NotImplementedException();
        }

        protected override void DownloadWaypoints() {
            currentCatalogue = new WaypointCatalog("XCSoar" + SerialNo);
            CAIAndWinPilotFileHandler fh = new CAIAndWinPilotFileHandler();
            bool success = false;
            WayPoint wp;

            IsInterrupted = false;
            try {
                ArrayList file = ReadFile("waypoints.dat");
                success = file.Count > 1;
                foreach (string line in file) {
                    if ((wp = fh.ParseLine(line)) != null) {
                        currentCatalogue.Add(wp);
                    }
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.Message, "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                success = false;
            }
            finally {
                if (RecorderBase.DownloadWaypointsThreadNotify != null) {
                    RecorderBase.DownloadWaypointsThreadNotify(success);
                }
            }
        }

        protected ArrayList ReadFile(string fileName) {
            // TODO:  Add SoaringPilotFile.ReadFile implementation
            string line;
            ArrayList file = new ArrayList();
            OpenFileDialog fd = new OpenFileDialog();
            fd.CheckFileExists = true;
            fd.Filter = "All files (*.*)|*.*|Waypoints (*.dat)|*.dat|Tasks (*.spt|*.spt|SUA (*.txt)|*.txt";
            fd.FilterIndex = 0;
            fd.FileName = fileName;

            if (fd.ShowDialog() == DialogResult.OK) {
                try {
                    using (StreamReader inStream = new StreamReader(fd.FileName)) {
                        while ((line = inStream.ReadLine()) != null) {
                            file.Add(line);
                        }
                    }
                }
                catch (Exception e) {
                    MessageBox.Show(e.ToString(), "Read Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
            return file;
        }

        protected void WriteFile(ArrayList file, string fileName) {
            // TODO:  Add SoaringPilotFile.WriteFile implementation
            SaveFileDialog fd = new SaveFileDialog();
            fd.CheckPathExists = true;
            fd.OverwritePrompt = false;
            fd.ValidateNames = true;
            fd.FileName = fileName;
            fd.Filter = "All files (*.*)|*.*|Waypoints (*.dat)|*.dat|Tasks (*.spt|*.spt|SUA (*.txt)|*.txt";
            fd.FilterIndex = 0;
            StreamWriter outStream = null;
            if (fd.ShowDialog() == DialogResult.OK) {
                try {
                    outStream = new StreamWriter(fd.FileName, false);
                    foreach (string line in file) {
                        outStream.WriteLine(line);
                    }
                }
                catch (Exception e) {
                    MessageBox.Show(e.ToString(), "Write Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
                finally {
                    if (outStream != null) {
                        outStream.Close();
                    }
                }
            }
        }

        protected override void UploadSUAs() {
            bool success = false;
            ArrayList file;
            IsInterrupted = false;
            SUAFileHandlerTimNewport fileHandler = new SUAFileHandlerTimNewport();
            try {
                file = fileHandler.WriteFile(suas);
                WriteFile(file, "suadata.txt");
                success = true;
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                success = false;
            }
            finally {
                if (RecorderBase.UploadSUAsThreadNotify != null) {
                    RecorderBase.UploadSUAsThreadNotify(success);
                }
            }
        }

        protected override void UploadTasks() {
            throw new System.NotImplementedException();
        }

        protected override void UploadWaypoints() {
            CAIAndWinPilotFileHandler fh = new CAIAndWinPilotFileHandler();
            bool success = true;

            IsInterrupted = false;
            try {
                if (currentCatalogue != null) {
                    ArrayList file = new ArrayList(currentCatalogue.Count);

                    foreach (WayPoint w in currentCatalogue.VisibleWaypoints) {
                        file.Add(fh.BuildLine(w));
                    }

                    WriteFile(file, "waypoints.dat");
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                success = false;
            }
            finally {
                if (RecorderBase.UploadWaypointsThreadNotify != null) {
                    RecorderBase.UploadWaypointsThreadNotify(success);
                }
            }
        }
        #endregion

        #region Attributes
        public override string Description {
            get {
                return string.Format(@"Adapter for XCSoar PDA Software

This is a file based adapter.

You can read or write files in the XCSoar required formats.

Download procedure  : None, just select the files from the file system.
Upload procedure    : None, just select the files from the file system.
") + base.OptionsString;
            }
        }
        #endregion
    }
}
