// name   : AirspaceElement.cs
// author : Harald Maier
// date   : 23.01.2004
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.Collections;
using System.Drawing;
using System.Drawing.Drawing2D;

using SoaringDotNET.Data;

namespace SoaringDotNET.GUI
{
	/// <summary>
	/// 
	/// </summary>
	public abstract class AirspaceElement : SoaringDotNET.GUI.MapAreaElement
	{
        public Region region;
        public Airspace airspace;

        public AirspaceElement(Airspace space) :
            base(MapElementTypes.Airspace) {
			// 
			// TODO: Add constructor logic here
			//
            airspace = space;
            pen = new Pen(Brushes.Black, 1);
		}

        public new AirspaceSegment this[int pos] {
            get {
                return (AirspaceSegment)airspace.points[pos];
            }
        }
    
        public new IEnumerator GetEnumerator() {
            return airspace.points.GetEnumerator();
        }

        public new int Count {
            get {
                return airspace.points.Count;
            }
        }

        public AirspaceTypes AirspaceType {
            get {
                return airspace.airspaceType;
            }
        }

        public AirspaceClasses AirspaceClass {
            get {
                return airspace.airspaceClass;
            }
        }

        public string AirspaceTypeString {
            get {
                string cl = "";
                switch (airspace.airspaceType) {
                case AirspaceTypes.ControlZone:
                    cl = "CTLZ";
                    break;
                case AirspaceTypes.ControlArea:
                    cl = "CTA";
                    break;
                case AirspaceTypes.TerminalControlArea:
                    cl = "TCA";
                    break;
                }
                return cl;
            }
        }

        public string AirspaceClassString {
            get {
                string cl = null;
                switch (airspace.airspaceClass) {
                case AirspaceClasses.A:
                    cl = "Class A";
                    break;
                case AirspaceClasses.B:
                    cl = "Class B";
                    break;
                case AirspaceClasses.C:
                    cl = "Class C";
                    break;
                case AirspaceClasses.D:
                    cl = "Class D";
                    break;
                case AirspaceClasses.E:
                    cl = "Class E";
                    break;
                case AirspaceClasses.F:
                    cl = "Class F";
                    break;
                case AirspaceClasses.Danger:
                    cl = "Danger";
                    break;
                case AirspaceClasses.Prohibited:
                    cl = "Prohibited";
                    break;
                case AirspaceClasses.Restricted:
                    cl = "Restricted";
                    break;
                case AirspaceClasses.Alert:
                    cl = "Alert";
                    break;
                case AirspaceClasses.Military:
                    cl = "Military Ops";
                    break;
                case AirspaceClasses.Temporary:
                    cl = "Temp. reserved";
                    break;
                case AirspaceClasses.Warning:
                    cl = "Warning";
                    break;
                case AirspaceClasses.Soaring:
                    cl = "Soaring";
                    break;
                case AirspaceClasses.Other:
                    cl = "Other";
                    break;
                default:
                    cl = "Unknown";
                    break;
                }
                return cl;
            }
        }

        public override void Draw(Graphics g, WgsArea view, int width, int height) {
            // TODO:  Add AirspaceElement.Draw implementation
            long lat, lat2;
            long lon, lon2;
            long latRange, lonRange;
            long minLat, minLon;
            long diffLat, diffLon;
            long diff;
            float radius;
            float startAngle, sweepAngle;
            double dist;

            minLat = view.TopLat;
            latRange = view.Height;
            minLon = view.LeftLon;
            lonRange = view.Width;

            if (view.AreaIntersect(Area)) {
                GraphicsPath gp = new GraphicsPath();
                foreach (AirspaceSegment s in this) {
                    switch (s.type) {
                    case AirspaceSegmentTypes.Line:
                        lat = (s.latFrom - minLat) * height / latRange;
                        diff = s.lonFrom - minLon;
                        if (diff <= -64800000) {
                            diff += 129600000;
                        }
                        lon = diff * width / lonRange;

                        lat2 = (s.latTo - minLat) * height / latRange;
                        diff = s.lonTo - minLon;
                        if (diff <= -64800000) {
                            diff += 129600000;
                        }
                        lon2 = diff * width / lonRange;

                        gp.AddLine(lon, lat, lon2, lat2);
                        break;
                    case AirspaceSegmentTypes.Circle:
                        radius = (s.radius / 60.0f) * 360000.0f * height / latRange;
                        lat = (s.latCenter - minLat) * height / latRange;
                        diff = s.lonCenter - minLon;
                        if (diff <= -64800000) {
                            diff += 129600000;
                        }
                        lon = diff * width / lonRange;
                        gp.AddEllipse(lon - radius, lat - radius, 2.0f * radius, 2.0f * radius);
                        break;
                    case AirspaceSegmentTypes.ArcClock:
                    case AirspaceSegmentTypes.ArcCounterClock:
                        // calculate center and offset for upper left corner of arc rectangle
                        radius = (float)Math.Round(((s.radius / 60.0) * 360000.0) * height / latRange);
                        lat = (s.latCenter - minLat) * height / latRange;
                        diff = s.lonCenter - minLon;
                        if (diff <= -64800000) {
                            diff += 129600000;
                        }
                        lon = diff * width / lonRange;

                        // calc pos for start point and angle
                        lat2 = (s.latFrom - minLat) * height / latRange;
                        diff = s.lonFrom - minLon;
                        if (diff <= -64800000) {
                            diff += 129600000;
                        }
                        lon2 = diff * width / lonRange;

                        diffLat = lat2 - lat;
                        diffLon = lon2 - lon;
                        dist = Math.Sqrt(Math.Pow(diffLon, 2) + Math.Pow(diffLat, 2));
                        if (diffLat < 0) {
                            startAngle = (float)(Math.Asin(diffLon / dist) * 180.0 / Math.PI) + 270.0f;
                        }
                        else {
                            startAngle = 90.0f - (float)(Math.Asin(diffLon / dist) * 180.0 / Math.PI);
                        }

                        // calc pos for dest point and angle
                        lat2 = (s.latTo - minLat) * height / latRange;
                        diff = s.lonTo - minLon;
                        if (diff <= -64800000) {
                            diff += 129600000;
                        }
                        lon2 = diff * width / lonRange;

                        diffLat = lat2 - lat;
                        diffLon = lon2 - lon;
                        dist = Math.Sqrt(Math.Pow(diffLon, 2) + Math.Pow(diffLat, 2));
                        if (diffLat < 0) {
                            sweepAngle = (float)(Math.Asin(diffLon / dist) * 180.0 / Math.PI) + 270.0f;
                        }
                        else {
                            sweepAngle = 90.0f - (float)(Math.Asin(diffLon / dist) * 180.0 / Math.PI);
                        }

                        if (s.type == AirspaceSegmentTypes.ArcClock) {
                            if (sweepAngle < startAngle) {
                                sweepAngle = 360.0f - startAngle + sweepAngle;
                            }
                            else {
                                sweepAngle -= startAngle;
                            }
                        }
                        else if (s.type == AirspaceSegmentTypes.ArcCounterClock) {
                            if (sweepAngle > startAngle)
                                sweepAngle = -(360.0f - sweepAngle + startAngle);
                            else {
                                sweepAngle -= startAngle;
                            }
                        }

                        if (radius > 0.0) {
                            gp.AddArc(lon - radius, lat - radius, 2 * radius, 2 * radius, startAngle, sweepAngle);
                        }
                        break;
                    }
                }

                region = new Region(gp);
                g.FillRegion(this.brush, region);
                g.DrawPath(this.pen, gp);
            }
            else {
                region = null;
            }
        }
    
        public new void CalcArea() {
            // TODO:  Add AirspaceElement.CalcArea implementation
            area.Clear();
            int radius;
            WgsPoint center;
            int from;
            int to;
            int w, ww;

            foreach (AirspaceSegment s in airspace) {
                if (area.IsEmpty) {
                    switch (s.type) {
                    case AirspaceSegmentTypes.ArcClock:
                        radius = (int)Math.Ceiling((s.radius / 60.0f) * 360000.0f);
                        center = new WgsPoint(s.latCenter, s.lonCenter);
                        from = (int)Math.Floor(center.bearingDeg(new WgsPoint(s.latFrom, s.lonFrom)));
                        to = (int)Math.Ceiling(center.bearingDeg(new WgsPoint(s.latTo, s.lonTo)));
                        if (to < from) {
                            to += 360;
                        }
                        for (w = from; w < to; w += 90) {
                            ww = w % 360;
                            if (ww < 90 && to > 90) {
                                area.RightLon = s.lonCenter + radius;
                            }
                            else if (ww < 180 && to > 180) {
                                area.BottomLat = s.latCenter + radius;
                            }
                            else if (ww < 270 && to > 270) {
                                area.LeftLon = s.lonCenter - radius;
                            }
                            else if (ww < 360 && to > 360) {
                                area.TopLat = s.latCenter - radius;
                            }
                        }
                        if (area.BottomLat != 0) {
                            area.BottomLat = Math.Max(Math.Max(s.latFrom, s.latTo), area.BottomLat);
                        }
                        else {
                            area.BottomLat = Math.Max(s.latFrom, s.latTo);
                        }
                        if (area.TopLat != 0) {
                            area.TopLat = Math.Min(Math.Min(s.latFrom, s.latTo), area.TopLat);
                        }
                        else {
                            area.TopLat = Math.Min(s.latFrom, s.latTo);
                        }
                        if (area.LeftLon != 0) {
                            area.LeftLon = Math.Min(Math.Min(s.lonFrom, s.lonTo), area.LeftLon);
                        }
                        else {
                            area.LeftLon = Math.Min(s.lonFrom, s.lonTo);
                        }
                        if (area.RightLon != 0) {
                            area.RightLon = Math.Max(Math.Max(s.lonFrom, s.lonTo), area.RightLon);
                        }
                        else {
                            area.RightLon = Math.Max(s.lonFrom, s.lonTo);
                        }
                        break;
                    case AirspaceSegmentTypes.ArcCounterClock:
                        radius = (int)Math.Ceiling((s.radius / 60.0f) * 360000.0f);
                        center = new WgsPoint(s.latCenter, s.lonCenter);
                        from = (int)Math.Ceiling(center.bearingDeg(new WgsPoint(s.latFrom, s.lonFrom)));
                        to = (int)Math.Floor(center.bearingDeg(new WgsPoint(s.latTo, s.lonTo)));
                        if (from < to) {
                            from += 360;
                        }
                        for (w = from; w > to; w -= 90) {
                            ww = w % 360;
                            if (ww < 90 && to > 270) {
                                area.TopLat = s.latCenter - radius;
                            }
                            else if (ww < 180 && to < 90) {
                                area.RightLon = s.lonCenter + radius;
                            }
                            else if (ww < 270 && to < 180) {
                                area.BottomLat = s.latCenter + radius;
                            }
                            else if (ww < 360 && to < 270) {
                                area.LeftLon = s.lonCenter - radius;                                
                            }
                        }
                        if (area.BottomLat != 0) {
                            area.BottomLat = Math.Max(Math.Max(s.latFrom, s.latTo), area.BottomLat);
                        }
                        else {
                            area.BottomLat = Math.Max(s.latFrom, s.latTo);
                        }
                        if (area.TopLat != 0) {
                            area.TopLat = Math.Min(Math.Min(s.latFrom, s.latTo), area.TopLat);
                        }
                        else {
                            area.TopLat = Math.Min(s.latFrom, s.latTo);
                        }
                        if (area.LeftLon != 0) {
                            area.LeftLon = Math.Min(Math.Min(s.lonFrom, s.lonTo), area.LeftLon);
                        }
                        else {
                            area.LeftLon = Math.Min(s.lonFrom, s.lonTo);
                        }
                        if (area.RightLon != 0) {
                            area.RightLon = Math.Max(Math.Max(s.lonFrom, s.lonTo), area.RightLon);
                        }
                        else {
                            area.RightLon = Math.Max(s.lonFrom, s.lonTo);
                        }
                        break;
                    case AirspaceSegmentTypes.Circle:
                        radius = (int)((s.radius / 60.0f) * 360000.0);
                        area.LeftLon = s.lonCenter - radius;
                        area.RightLon = s.lonCenter + radius;
                        area.TopLat = s.latCenter - radius;
                        area.BottomLat = s.latCenter + radius;
                        break;
                    case AirspaceSegmentTypes.Line:
                        area.BottomLat = Math.Max(s.latFrom, s.latTo);
                        area.TopLat = Math.Min(s.latFrom, s.latTo);
                        area.LeftLon = Math.Min(s.lonFrom, s.lonTo);
                        area.RightLon = Math.Max(s.lonFrom, s.lonTo);
                        break;
                    }
                }
                else {
                    switch (s.type) {
                    case AirspaceSegmentTypes.ArcClock:
                        radius = (int)Math.Ceiling((s.radius / 60.0f) * 360000.0f);
                        center = new WgsPoint(s.latCenter, s.lonCenter);
                        from = (int)Math.Floor(center.bearingDeg(new WgsPoint(s.latFrom, s.lonFrom)));
                        to = (int)Math.Ceiling(center.bearingDeg(new WgsPoint(s.latTo, s.lonTo)));
                        if (to < from) {
                            to += 360;
                        }
                        for (w = from; w < to; w += 90) {
                            ww = w % 360;
                            if (ww < 90 && to > 90) {
                                area.RightLon = Math.Max(s.lonCenter + radius, area.RightLon);
                            }
                            else if (ww < 180 && to > 180) {
                                area.BottomLat = Math.Max(s.latCenter + radius, area.BottomLat);
                            }
                            else if (ww < 270 && to > 270) {
                                area.LeftLon = area.LeftLon = Math.Min(s.lonCenter - radius, area.LeftLon);
                            }
                            else if (ww < 360 && to > 360) {
                                area.TopLat = Math.Min(s.latCenter - radius, area.TopLat);
                            }
                        }
                        goto case AirspaceSegmentTypes.Line;
                    case AirspaceSegmentTypes.ArcCounterClock:
                        radius = (int)Math.Ceiling((s.radius / 60.0f) * 360000.0f);
                        center = new WgsPoint(s.latCenter, s.lonCenter);
                        from = (int)Math.Ceiling(center.bearingDeg(new WgsPoint(s.latFrom, s.lonFrom)));
                        to = (int)Math.Floor(center.bearingDeg(new WgsPoint(s.latTo, s.lonTo)));
                        if (from < to) {
                            from += 360;
                        }
                        for (w = from; w > to; w -= 90) {
                            ww = w % 360;
                            if (ww < 90 && to > 270) {
                                area.TopLat = s.latCenter - radius;
                            }
                            else if (ww < 180 && to < 90) {
                                area.RightLon = s.lonCenter + radius;
                            }
                            else if (ww < 270 && to < 180) {
                                area.BottomLat = s.latCenter + radius;
                            }
                            else if (ww < 360 && to < 270) {
                                area.LeftLon = s.lonCenter - radius;                                
                            }
                        }
                        goto case AirspaceSegmentTypes.Line;
                    case AirspaceSegmentTypes.Circle:
                        radius = (int)((s.radius / 60.0f) * 360000.0f);
                        area.LeftLon = Math.Min(s.lonCenter - radius, area.LeftLon);
                        area.RightLon = Math.Max(s.lonCenter + radius, area.RightLon);
                        area.TopLat = Math.Min(s.latCenter - radius, area.TopLat);
                        area.BottomLat = Math.Max(s.latCenter + radius, area.BottomLat);
                        break;
                    case AirspaceSegmentTypes.Line:
                        area.BottomLat = Math.Max(Math.Max(s.latFrom, s.latTo), area.BottomLat);
                        area.TopLat = Math.Min(Math.Min(s.latFrom, s.latTo), area.TopLat);
                        area.LeftLon = Math.Min(Math.Min(s.lonFrom, s.lonTo), area.LeftLon);
                        area.RightLon = Math.Max(Math.Max(s.lonFrom, s.lonTo), area.RightLon);
                        break;
                    }
                }
            }
        }
    
        public override string ToString() {
            // TODO:  Add AirspaceElement.ToString implementation
            string cl = null, txt;

            cl = AirspaceClassString;

            txt = string.Format("{0}{1} {2}\n{3}->{4}", airspace.name, AirspaceType == AirspaceTypes.SpecialUsedAirspace ? " (" + airspace.id + ")" : "", cl, airspace.lowerBound, airspace.upperBound);
            if (airspace.radioFrequency.Length > 0 || airspace.commName.Length > 0) {
                txt += "\nCOM: " + airspace.commName + " " + airspace.radioFrequency;
            }
            if (airspace.remark.Length > 0) {
                txt += "\nREM:" + airspace.remark;
            }
            return txt;
        }
    }
}
