// name   : AppContents.cs
// author : Harald Maier
// date   : 17.08.2003
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.Collections;
using System.Xml;
using System.IO;
using System.Windows.Forms;
using System.Globalization;
using System.Drawing;
using Microsoft.Win32;

using SoaringDotNET.GUI;

namespace SoaringDotNET.Data
{
	/// <summary>
	/// 
	/// </summary>
    public enum WayPointTypeId {Unknown = -1, Airport, Glidersite, Outlanding, Landmark};
    public enum SurfaceTypeId {Unknown = -1, Grass, Hard};
    public enum CountryTypeId {Unknown = -1, Andorra, UnitedArabEmirates, Afghanistan, AntiguaBarbuda, Anguilla, 
        Albania, Armenia, NetherlandsAntilles, Angola, Antarctica, Argentina, AmericanSamoa, Austria, Australia, 
        Aruba, AlandIslands, Azerbaijan, BosniaHerzegovina, Barbados, Bangladesh, 
        Belgium, BurkinaFaso, Bulgaria, Bahrain, Burundi, 
        Benin, Bermuda, BruneiDarussalam, Bolivia, Brazil, Bahamas, Bhutan, BouvetIsland, Botswana, Belarus, Belize, Canada, 
        CocosKeelingIslands, DemocraticRepublicCongo, CentralAfricanRepublic, Congo, Switzerland, CoteDIvoire, 
        CookIslands, Chile, Cameroon, China, Colombia, CostaRica, SerbiaMontenegro, Cuba, CapeVerde, ChristmasIsland, 
        Cyprus, CzechRepublic, Germany, Djibouti, Denmark, Dominica, DominicanRepublic, Algeria, Ecuador, Estonia, 
        Egypt, WesternSahara, Eritrea, Spain, Ethiopia, Finland, Fiji, FalklandIslands, FederatedStatesMicronesia, 
        FaroeIslands, France, FranceMetropolitan, Gabon, GreatBritain, Grenada, Georgia, FrenchGuiana, Ghana, Gibraltar, Greenland, 
        Gambia, Guinea, Guadeloupe, EquatorialGuinea, Greece, SGeorgiaSSandwichIslands, Guatemala, Guam, GuineaBissau, 
        Guyana, HongKong, HeardIslandMcDonaldIslands, Honduras, Croatia, Haiti, Hungary, Indonesia, Ireland, Israel, 
        India, BritishIndianOceanTerritory, Iraq, Iran, Iceland, Italy, Jamaica, Jordan, Japan, Kenya, Kyrgyzstan, 
        Cambodia, Kiribati, Comoros, SaintKittsNevis, KoreaNorth, KoreaSouth, Kuwait, CaymanIslands, Kazakhstan, Laos, 
        Lebanon, SaintLucia, Liechtenstein, SriLanka, Liberia, Lesotho, Lithuania, Luxembourg, Latvia, Libya, Morocco, 
        Monaco, Moldova, Madagascar, MarshallIslands, Macedonia, Mali, Myanmar, Mongolia, Macao, NorthernMarianaIslands, 
        Martinique, Mauritania, Montserrat, Malta, Mauritius, Maldives, Malawi, Mexico, Malaysia, Mozambique, Namibia, 
        NewCaledonia, Niger, NorfolkIsland, Nigeria, Nicaragua, Netherlands, Norway, Nepal, Nauru, Niue, NewZealand, 
        Oman, Panama, Peru, FrenchPolynesia, PapuaNewGuinea, Philippines, Pakistan, Poland, SaintPierreMiquelon, Pitcairn, 
        PuertoRico, PalestinianTerritory, Portugal, Palau, Paraguay, Qatar, Reunion, Romania, RussianFederation, Rwanda, 
        SaudiArabia, SolomonIslands, Seychelles, Sudan, Sweden, Singapore, SaintHelena, Slovenia, SvalbardJanMayen, 
        Slovakia, SierraLeone, SanMarino, Senegal, Somalia, Suriname, SaoTomePrincipe, USSR, ElSalvador, Syria, Swaziland, 
        TurksCaicosIslands, Chad, FrenchSouthernTerritories, Togo, Thailand, Tajikistan, Tokelau, TimorLeste, Turkmenistan, 
        Tunisia, Tonga, EastTimor, Turkey, TrinidadTobago, Tuvalu, Taiwan, Tanzania, Ukraine, Uganda, UnitedKingdom, 
        UnitedStatesMinorOutlyingIslands, UnitedStates, Uruguay, Uzbekistan, VaticanCityState, SaintVincentGrenadines, 
        Venezuela, VirginIslandsBritish, VirginIslandsUS, VietNam, Vanuatu, WallisFutuna, Samoa, Yemen, Mayotte, Yugoslavia, 
        SouthAfrica, Zambia, Zaire, Zimbabwe};

    public enum FileTypeId {All, Flight, Task, Waypoint};
    public enum TurnpointTypes {Start, Turnpoint, Finish};

    public class AppContents {
        public TranslationList waypointTypes;
        public TranslationList surfaceTypes;
        public TranslationList countryTypes;

        public WayPoint homePoint;
        public static ArrayList waypointCatalogs = new ArrayList();
        public static ArrayList flights = new ArrayList();
        public static ArrayList tasks = new ArrayList();
        public static ArrayList pilots = new ArrayList();
        public static ArrayList airspaces = new ArrayList();
        public static ArrayList compCountries = new ArrayList();
        public static ArrayList loadedAirspaceCountries = new ArrayList();
        public static ArrayList allColors = new ArrayList();

        public static Hashtable gliderIndexes = new Hashtable();
        public static Hashtable countryTypesByCode = new Hashtable();
        public static Hashtable airspaceColours = new Hashtable();
        public static Hashtable countryTypesById = new Hashtable();
        public static Hashtable airspacesByCountry = new Hashtable();

        public static MapColor classAColor;
        public static MapColor classBColor;
        public static MapColor classCColor;
        public static MapColor classCCTRColor;
        public static MapColor classDColor;
        public static MapColor classDCTRColor;
        public static MapColor classEHighColor;
        public static MapColor classELowColor;
        public static MapColor classFColor;
        public static MapColor classAlertColor;
        public static MapColor classDangerColor;
        public static MapColor classMilitaryColor;
        public static MapColor classProhibitedColor;
        public static MapColor classRestrictedColor;
        public static MapColor classTempColor;
        public static MapColor classWarningColor;
        public static MapColor classOtherColor;
        public static MapColor classSoaringColor;

        public static MapColor taskColor;
        public static MapColor taskColorFAI;
        public static MapColor flightColor;

        public static  object[] compClasses = new object[0];

        public static NumberFormatInfo ni = new NumberFormatInfo();

        public static Hashtable defaultSectors = new Hashtable();

        public AppContents() {
            // 
            // TODO: Add constructor logic here
            //
            waypointTypes = new TranslationList();
            waypointTypes.Add(new IdType(WayPointTypeId.Unknown, "Unknown"));
            waypointTypes.Add(new IdType(WayPointTypeId.Airport, "Airport"));
            waypointTypes.Add(new IdType(WayPointTypeId.Glidersite, "Glidersite"));
            waypointTypes.Add(new IdType(WayPointTypeId.Outlanding, "Outlanding field"));
            waypointTypes.Add(new IdType(WayPointTypeId.Landmark, "Landmark"));

            surfaceTypes = new TranslationList();
            surfaceTypes.Add(new IdType(SurfaceTypeId.Unknown, "Unknown"));
            surfaceTypes.Add(new IdType(SurfaceTypeId.Grass, "Grass"));
            surfaceTypes.Add(new IdType(SurfaceTypeId.Hard, "Asphalt/Concrete"));

            countryTypes = new TranslationList();

            RegisterCountries();

            foreach (string s in countryTypesByCode.Keys) {
                countryTypesById[countryTypesByCode[s]] = s;
            }

            compClasses = new object [] {new CompetitionClass("15m FAI", 5), 
                                            new CompetitionClass("18m", 6),
                                            new CompetitionClass("Club", 2),
                                            new CompetitionClass("Doubleseater", 7),
                                            new CompetitionClass("Open", 8),
                                            new CompetitionClass("Standard", 3),
                                            new CompetitionClass("World", 1)};
                                                 
            ni.NumberDecimalSeparator = ".";
            ni.NumberGroupSeparator = "";

            ReadPilots();
            ReadGliders();
            ReadColours();
            ReadAirspace();
            ReadSectors();
            ReadCompetitionCountries();
        }

        #region Public Functions

        public void SaveColors() {
            foreach (MapColor c in allColors) {
                c.Save();
            }
        }

        public void SaveSectors() {
            RegistryKey currUser = Registry.CurrentUser.CreateSubKey("Software\\SoaringDotNet\\Sector");
            RegistryKey s;

            foreach (TurnpointTypes type in defaultSectors.Keys) {
                SectorDefinition sector = (SectorDefinition)defaultSectors[type];

                s = currUser.CreateSubKey(((int)type).ToString());
                s.SetValue("Type", (int)sector.sectorType);
                s.SetValue("Radius 1", sector.radius1);
                s.SetValue("Radius 2", sector.radius2);
            }
        }

        public void SavePilots() {
            string fileName = Application.StartupPath + @"\pilots.xml";
            XmlNode n;
            XmlElement root;
            XmlAttribute attr;
            try {
                XmlDocument doc = new XmlDocument();
                doc.AppendChild(doc.CreateDocumentType("SoaringDotNetPilots", null, null, null));
                root = doc.CreateElement("Pilots");
                foreach (Pilot p in pilots) {
                    n = doc.CreateElement("Pilot");

                    attr = doc.CreateAttribute("FirstName");
                    attr.Value = p.FirstName;
                    n.Attributes.Append(attr);
                    attr = doc.CreateAttribute("LastName");
                    attr.Value = p.LastName;
                    n.Attributes.Append(attr);
                    attr = doc.CreateAttribute("BirthDay");
                    attr.Value = p.BirthDayString;
                    n.Attributes.Append(attr);
                    attr = doc.CreateAttribute("Street");
                    attr.Value = p.Street;
                    n.Attributes.Append(attr);
                    attr = doc.CreateAttribute("Address");
                    attr.Value = p.Address;
                    n.Attributes.Append(attr);
                    attr = doc.CreateAttribute("Phone");
                    attr.Value = p.Phone;
                    n.Attributes.Append(attr);
                    attr = doc.CreateAttribute("eMail");
                    attr.Value = p.Email;
                    n.Attributes.Append(attr);

                    root.AppendChild(n);
                }
                doc.AppendChild(root);
                doc.Save(fileName);
            }
            catch (Exception e) {
                MessageBox.Show("Error occured while writing pilots file\n\n" + fileName + "\n\n" +
                    e.Message + "\n\n" + e.StackTrace, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        public WaypointCatalog NewCatalog() {
            int i = 1;
            bool found;

            do {
                found = false;
                foreach (WaypointCatalog c in waypointCatalogs) {
                    if (c.Name == "Waypoint" + i.ToString()) {
                        found = true;
                        i++;
                        break;
                    }
                }
            } while (found);

            WaypointCatalog cc = new WaypointCatalog("Waypoint" + i.ToString());
            waypointCatalogs.Add(cc);
            return cc;
        }

        public WaypointCatalog FindCatalog(string name) {
            WaypointCatalog cc = null;
            foreach (WaypointCatalog c in waypointCatalogs) {
                if (c.Name == name) {
                    cc = c;
                    break;
                }
            }
            return cc;
        }

        public void ReloadAirspace() {
            airspaces.Clear();
            airspacesByCountry.Clear();
            loadedAirspaceCountries.Clear();
            ReadColours();
            ReadAirspace();
        }

        #endregion

        #region Private Functions

        private void ReadColours() {
            classAColor = new MapColor("Class A", Color.DarkGreen);
            classBColor = new MapColor("Class B", Color.DarkGreen);
            classCColor = new MapColor("Class C", Color.DarkGreen);
            classCCTRColor = new MapColor("Class C (CTR)", Color.DarkRed);
            classDColor = new MapColor("Class D", Color.DarkGreen);
            classDCTRColor = new MapColor("Class D (CTR)", Color.DarkRed);
            classEHighColor = new MapColor("Class E (High)", Color.Pink);
            classELowColor = new MapColor("Class E (Low)", Color.LightSkyBlue);
            classFColor = new MapColor("Class F", Color.LightBlue);
            classAlertColor = new MapColor("Class Alert", Color.Red);
            classDangerColor = new MapColor("Class Danger", Color.DarkBlue);
            classMilitaryColor = new MapColor("Class Military Ops.", Color.Red);
            classProhibitedColor = new MapColor("Class Prohibited", Color.DarkBlue);
            classRestrictedColor = new MapColor("Class Restricted", Color.DarkBlue);
            classTempColor = new MapColor("Class Temp. Reserved", Color.Wheat);
            classWarningColor = new MapColor("Class Warning", Color.Red);
            classOtherColor = new MapColor("Class Other", Color.Gray);
            classSoaringColor = new MapColor("Class Soaring", Color.SteelBlue);

            taskColor = new MapColor("Task", Color.BlueViolet);
            taskColorFAI = new MapColor("Task (FAI)", Color.Lime);
            flightColor = new MapColor("Flight Path", Color.Goldenrod);

            allColors.Clear();
            allColors.AddRange(new object [] {classAColor, classBColor, classCColor, classCCTRColor, 
                                                 classDColor, classDCTRColor, classEHighColor, classELowColor, 
                                                 classFColor, classAlertColor, classDangerColor, classMilitaryColor, 
                                                 classProhibitedColor, classRestrictedColor, classTempColor, classWarningColor,
                                                 classOtherColor, classSoaringColor,
                                                 taskColor, taskColorFAI, flightColor});

            foreach (MapColor c in allColors) {
                c.Load();
            }
        }

        private void ReadSectors() {
            RegistryKey currUser = Registry.CurrentUser.CreateSubKey("Software\\SoaringDotNet\\Sector");
            RegistryKey s;
            SectorDefinition sector;

            s = currUser.CreateSubKey(((int)TurnpointTypes.Start).ToString());
            sector = new SectorDefinition(TurnpointTypes.Start);
            sector.sectorType = (SectorTypes)s.GetValue("Type", SectorTypes.Sector);
            sector.radius1 = (int)s.GetValue("Radius 1", 3000);
            sector.radius2 = (int)s.GetValue("Radius 2", 0);
            defaultSectors.Add(TurnpointTypes.Start, sector);

            s = currUser.CreateSubKey(((int)TurnpointTypes.Turnpoint).ToString());
            sector = new SectorDefinition(TurnpointTypes.Turnpoint);
            sector.sectorType = (SectorTypes)s.GetValue("Type", SectorTypes.Sector);
            sector.radius1 = (int)s.GetValue("Radius 1", 3000);
            sector.radius2 = (int)s.GetValue("Radius 2", 0);
            defaultSectors.Add(TurnpointTypes.Turnpoint, sector);
        
            s = currUser.CreateSubKey(((int)TurnpointTypes.Finish).ToString());
            sector = new SectorDefinition(TurnpointTypes.Finish);
            sector.sectorType = (SectorTypes)s.GetValue("Type", SectorTypes.Sector);
            sector.radius1 = (int)s.GetValue("Radius 1", 3000);
            sector.radius2 = (int)s.GetValue("Radius 2", 0);
            defaultSectors.Add(TurnpointTypes.Finish, sector);
        }

        private void ReadCompetitionCountries() {
            compCountries.Add(new CompetitionCountry("Deutschland", "http://www.segelflugszene.de/olc-cgi/olc-d/olc"));
            compCountries.Add(new CompetitionCountry("sterreich", "http://www.segelflugszene.de/olc-cgi/olc-at/olc"));
            compCountries.Add(new CompetitionCountry("Schweiz", "http://www.segelflugszene.de/olc-cgi/olc-ch/olc"));
            compCountries.Add(new CompetitionCountry("Slovenia", "http://www.segelflugszene.de/olc-cgi/olc-si/olc"));
            compCountries.Add(new CompetitionCountry("Czech Republic", "http://www.segelflugszene.de/olc-cgi/olc-cz/olc"));
            compCountries.Add(new CompetitionCountry("France", "http://www.segelflugszene.de/olc-cgi/olc-fr/olc"));
            compCountries.Add(new CompetitionCountry("Canada", "http://www.segelflugszene.de/olc-cgi/olc-ca/olc"));
            compCountries.Add(new CompetitionCountry("Afrika", "http://www.segelflugszene.de/olc-cgi/olc-af/olc"));
            compCountries.Add(new CompetitionCountry("Australia", "http://www.segelflugszene.de/olc-cgi/olc-au/olc"));
            compCountries.Add(new CompetitionCountry("Belgium", "http://www.segelflugszene.de/olc-cgi/olc-be/olc"));
            compCountries.Add(new CompetitionCountry("Brasil", "http://www.segelflugszene.de/olc-cgi/olc-br/olc"));
            compCountries.Add(new CompetitionCountry("Denmark", "http://www.segelflugszene.de/olc-cgi/olc-dk/olc"));
            compCountries.Add(new CompetitionCountry("Finland", "http://www.segelflugszene.de/olc-cgi/olc-fi/olc"));
            compCountries.Add(new CompetitionCountry("Greece", "http://www.segelflugszene.de/olc-cgi/olc-gr/olc"));
            compCountries.Add(new CompetitionCountry("Hungary", "http://www.segelflugszene.de/olc-cgi/olc-hu/olc"));
            compCountries.Add(new CompetitionCountry("Italy", "http://www.segelflugszene.de/olc-cgi/olc-it/olc"));
            compCountries.Add(new CompetitionCountry("Japan", "http://www.segelflugszene.de/olc-cgi/olc-jp/olc"));
            compCountries.Add(new CompetitionCountry("The Netherlands", "http://www.segelflugszene.de/olc-cgi/olc-nl/olc"));
            compCountries.Add(new CompetitionCountry("New Zealand", "http://www.segelflugszene.de/olc-cgi/olc-nz/olc"));
            compCountries.Add(new CompetitionCountry("Norway", "http://www.segelflugszene.de/olc-cgi/olc-no/olc"));
            compCountries.Add(new CompetitionCountry("Poland", "http://www.segelflugszene.de/olc-cgi/olc-pl/olc"));
            compCountries.Add(new CompetitionCountry("Spain", "http://www.segelflugszene.de/olc-cgi/olc-es/olc"));
            compCountries.Add(new CompetitionCountry("Sweden", "http://www.segelflugszene.de/olc-cgi/olc-se/olc"));
            compCountries.Add(new CompetitionCountry("United Kingdom", "http://www.segelflugszene.de/olc-cgi/olc-uk/olc"));
            compCountries.Add(new CompetitionCountry("USA", "http://www.segelflugszene.de/olc-cgi/olc-usa/olc"));
            compCountries.Add(new CompetitionCountry("Argentinia", "http://www.segelflugszene.de/olc-cgi/olc-ar/olc"));
            compCountries.Add(new CompetitionCountry("Luxembourg", "http://www.segelflugszene.de/olc-cgi/olc-lu/olc"));
            compCountries.Add(new CompetitionCountry("Portugal", "http://www.segelflugszene.de/olc-cgi/olc-pt/olc"));
        }

        private void ReadGliders() {
            string fileName = Application.StartupPath + @"\gliders.txt";
            string line;
            string []tokens;
            char []splits = {',', '='};
            Glider g;
            try {
                FileInfo fi = new FileInfo(fileName);
                if (fi.Exists) {
                    using (StreamReader reader = new StreamReader(fileName)) {
                        while ((line = reader.ReadLine()) != null) {
                            line.Trim();
                            tokens = line.Split(splits);
                            if (tokens.Length == 3) {
                                g = new Glider(tokens[0], tokens[2], int.Parse(tokens[1]));
                                gliderIndexes.Add(tokens[0], g);
                            }
                        }
                    }
                }
            }
            catch (Exception e) {
                MessageBox.Show("Error occured while reading glider index file\n\n" + fileName + "\n\n" +
                    e.Message + "\n\n" + e.StackTrace, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void ReadPilots() {
            string fileName = Application.StartupPath + @"\pilots.xml";
            XmlAttributeCollection attrs;

            Pilot p;
            try {
                FileInfo fi = new FileInfo(fileName);
                if (fi.Exists) {
                    using (StreamReader reader = new StreamReader(fileName)) {
                        XmlDocument doc = new XmlDocument();
                        doc.Load(reader);
                        foreach (XmlNode n in doc.GetElementsByTagName("Pilot")) {
                            p = new Pilot();
                            attrs = n.Attributes;
                            p.FirstName = attrs["FirstName"].Value;
                            p.LastName = attrs["LastName"].Value;
                            p.BirthDay = DateTime.ParseExact(attrs["BirthDay"].Value, "dd.MM.yyyy", null);
                            p.Street = attrs["Street"].Value;
                            p.Address = attrs["Address"].Value;
                            p.Phone = attrs["Phone"].Value;
                            p.Email = attrs["eMail"].Value;

                            pilots.Add(p);
                        }
                    }
                }
            }
            catch (Exception e) {
                MessageBox.Show("Error occured while reading pilots file\n\n" + fileName + "\n\n" +
                    e.Message + "\n\n" + e.StackTrace, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void ReadAirspace() {
            string dirName = null;
            string t, country;
            int airspaceCount, segmentCount;
            int i, j;
            DafifCountry c = null;
            Airspace space;
            AirspaceElement element;
            AirspaceSegment segment;
            ArrayList countryContents;

            RegistryKey currUser = Registry.CurrentUser.CreateSubKey("Software\\SoaringDotNet\\Path");
            dirName = (string)currUser.GetValue("AirspacePath", Application.StartupPath + @"\airspace");

            //DateTime validTo;
            if (Directory.Exists(dirName)) {
                string [] fileNames = Directory.GetFiles(dirName);
                foreach (string file in fileNames) {
                    try {
                        FileInfo fi = new FileInfo(file);
                        using (BinaryReader reader = new BinaryReader(fi.OpenRead())) {
                            if (reader.ReadByte() == Database.AirspaceGenerator.MAGIC_AIRSPACE_NUMBER) {
                                t = reader.ReadString();
/*                                validTo = new DateTime(int.Parse(t.Substring(4, 4)), int.Parse(t.Substring(2, 2)), int.Parse(t.Substring(0, 2)));
                                if (validTo < DateTime.Today) {
                                    MessageBox.Show(string.Format("The airspace file {0} has been expired on {1}!\nPlease generate or download the latest version.", fi.Name, validTo), "Warning", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                                }*/
                                country = t.Substring(8);
                                c = new DafifCountry();
                                c.code = country;
                                i = fi.Name.IndexOf(".");
                                c.name = fi.Name.Substring(0, i > -1 ? i : fi.Name.Length);
                                loadedAirspaceCountries.Add(c);
                                airspaceCount = reader.ReadInt32();
                                countryContents = new ArrayList(airspaceCount);
                                airspacesByCountry.Add(country, countryContents);
                                for (i = 0; i < airspaceCount; i++) {
                                    space = new Airspace();
                                    space.id = reader.ReadString();
                                    space.name = reader.ReadString();
                                    space.airspaceClass = (AirspaceClasses)reader.ReadChar();
                                    space.airspaceType = (AirspaceTypes)reader.ReadChar();
                                    space.lowerBound = reader.ReadString();
                                    space.upperBound = reader.ReadString();
                                    space.remark = reader.ReadString();
                                    space.commName = reader.ReadString();
                                    space.radioFrequency = reader.ReadString();
                                    space.country = country;
                                    segmentCount = reader.ReadInt32();
                                    for (j = 0; j < segmentCount; j++) {
                                        segment = new AirspaceSegment();
                                        segment.type = (AirspaceSegmentTypes)reader.ReadChar();
                                        segment.latFrom = reader.ReadInt32();
                                        segment.lonFrom = reader.ReadInt32();
                                        segment.latTo = reader.ReadInt32();
                                        segment.lonTo = reader.ReadInt32();
                                        segment.latCenter = reader.ReadInt32();
                                        segment.lonCenter = reader.ReadInt32();
                                        segment.radius = reader.ReadSingle();

                                        space.points.Add(segment);
                                    }

                                    switch (space.airspaceClass) {
                                    case AirspaceClasses.A:
                                        element = new ClassA(space);
                                        break;
                                    case AirspaceClasses.B:
                                        element = new ClassB(space);
                                        break;
                                    case AirspaceClasses.C:
                                        element = new ClassC(space);
                                        break;
                                    case AirspaceClasses.D:
                                        element = new ClassD(space);
                                        break;
                                    case AirspaceClasses.E:
                                        if (space.lowerBound.IndexOf("1000") != -1) {
                                            element = new ClassELow(space);
                                        }
                                        else {
                                            element = new ClassEHigh(space);
                                        }
                                        break;
                                    case AirspaceClasses.Danger:
                                        element = new ClassDanger(space);
                                        break;
                                    case AirspaceClasses.Prohibited:
                                        element = new ClassProhibited(space);
                                        break;
                                    case AirspaceClasses.Restricted:
                                        element = new ClassRestricted(space);
                                        break;
                                    case AirspaceClasses.Military:
                                        element = new ClassMilitary(space);
                                        break;
                                    case AirspaceClasses.Temporary:
                                        element = new ClassTempReserved(space);
                                        break;
                                    case AirspaceClasses.Warning:
                                        element = new ClassWarning(space);
                                        break;
                                    case AirspaceClasses.Alert:
                                        element = new ClassAlert(space);
                                        break;
                                    case AirspaceClasses.Soaring:
                                        element = new ClassSoaring(space);
                                        break;
                                    case AirspaceClasses.Other:
                                        element = new ClassOther(space);
                                        break;
                                    default:
                                        continue;
                                    }

                                    element.CalcArea();
                                    airspaces.Add(element);
                                    countryContents.Add(element);
                                }
                            }
                        }
                    }
                    catch (Exception e) {
                        MessageBox.Show("Error occured while reading airspace file\n\nPlease regenerate or download the latest version!\n\n" + file + "\n\n" +
                            e.Message + "\n\n" + e.StackTrace, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);

                        airspaces.Clear();
                        break;
                    }

                    IComparer comp = new AirspaceSorter();
                    airspaces.Sort(comp);
                }
            }
            else {
                try {
                    Directory.CreateDirectory(dirName);
                    MessageBox.Show("The airspace direcory has been created in\n\n" + dirName + "\n\n" +
                        "Please generate or download your requested airspace databases\nand copy the files into this directory.", "Information", MessageBoxButtons.OK, MessageBoxIcon.Information);
                }
                catch (Exception e) {
                    MessageBox.Show("Error occured while creating airspace directory\n\n" + dirName + "\n\n" +
                        e.Message + "\n\n" + e.StackTrace, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
        }

        private void RegisterCountries() {
            countryTypes.Add(new IdType(CountryTypeId.Unknown, "XX  Unknown"));
            countryTypes.Add(new IdType(CountryTypeId.Andorra, "AD  Andorra"));
            countryTypes.Add(new IdType(CountryTypeId.UnitedArabEmirates, "AE  United Arab Emirates"));
            countryTypes.Add(new IdType(CountryTypeId.Afghanistan, "AF  Afghanistan"));
            countryTypes.Add(new IdType(CountryTypeId.AntiguaBarbuda, "AG  Antigua and Barbuda"));
            countryTypes.Add(new IdType(CountryTypeId.Anguilla, "AI  Anguilla"));
            countryTypes.Add(new IdType(CountryTypeId.Albania, "AL  Albania"));
            countryTypes.Add(new IdType(CountryTypeId.Armenia, "AM  Armenia"));
            countryTypes.Add(new IdType(CountryTypeId.NetherlandsAntilles, "AN  Netherlands Antilles"));
            countryTypes.Add(new IdType(CountryTypeId.Angola, "AO  Angola"));
            countryTypes.Add(new IdType(CountryTypeId.Antarctica, "AQ  Antarctica"));
            countryTypes.Add(new IdType(CountryTypeId.Argentina, "AR  Argentina"));
            countryTypes.Add(new IdType(CountryTypeId.AmericanSamoa, "AS  American Samoa"));
            countryTypes.Add(new IdType(CountryTypeId.Austria, "AT  Austria"));
            countryTypes.Add(new IdType(CountryTypeId.Australia, "AU  Australia"));
            countryTypes.Add(new IdType(CountryTypeId.Aruba, "AW  Aruba"));
            countryTypes.Add(new IdType(CountryTypeId.AlandIslands, "AX  Aland Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Azerbaijan, "AZ  Azerbaijan"));
            countryTypes.Add(new IdType(CountryTypeId.BosniaHerzegovina, "BA  Bosnia and Herzegovina"));
            countryTypes.Add(new IdType(CountryTypeId.Barbados, "BB  Barbados"));
            countryTypes.Add(new IdType(CountryTypeId.Bangladesh, "BD  Bangladesh"));
            countryTypes.Add(new IdType(CountryTypeId.Belgium, "BE  Belgium"));
            countryTypes.Add(new IdType(CountryTypeId.BurkinaFaso, "BF  Burkina Faso"));
            countryTypes.Add(new IdType(CountryTypeId.Bulgaria, "BG  Bulgaria"));
            countryTypes.Add(new IdType(CountryTypeId.Bahrain, "BH  Bahrain"));
            countryTypes.Add(new IdType(CountryTypeId.Burundi, "BI  Burundi"));
            countryTypes.Add(new IdType(CountryTypeId.Benin, "BJ  Benin"));
            countryTypes.Add(new IdType(CountryTypeId.Bermuda, "BM  Bermuda"));
            countryTypes.Add(new IdType(CountryTypeId.BruneiDarussalam, "BN  Brunei Darussalam"));
            countryTypes.Add(new IdType(CountryTypeId.Bolivia, "BO  Bolivia"));
            countryTypes.Add(new IdType(CountryTypeId.Brazil, "BR  Brazil"));
            countryTypes.Add(new IdType(CountryTypeId.Bahamas, "BS  Bahamas"));
            countryTypes.Add(new IdType(CountryTypeId.Bhutan, "BT  Bhutan"));
            countryTypes.Add(new IdType(CountryTypeId.BouvetIsland, "BV  Bouvet Island"));
            countryTypes.Add(new IdType(CountryTypeId.Botswana, "BW  Botswana"));
            countryTypes.Add(new IdType(CountryTypeId.Belarus, "BY  Belarus"));
            countryTypes.Add(new IdType(CountryTypeId.Belize, "BZ  Belize"));
            countryTypes.Add(new IdType(CountryTypeId.Canada, "CA  Canada"));
            countryTypes.Add(new IdType(CountryTypeId.CocosKeelingIslands, "CC  Cocos (Keeling) Islands"));
            countryTypes.Add(new IdType(CountryTypeId.DemocraticRepublicCongo, "CD  Democratic Republic of the Congo"));
            countryTypes.Add(new IdType(CountryTypeId.CentralAfricanRepublic, "CF  Central African Republic"));
            countryTypes.Add(new IdType(CountryTypeId.Congo, "CG  Congo"));
            countryTypes.Add(new IdType(CountryTypeId.Switzerland, "CH  Switzerland"));
            countryTypes.Add(new IdType(CountryTypeId.CoteDIvoire, "CI  Cote D'Ivoire (Ivory Coast)"));
            countryTypes.Add(new IdType(CountryTypeId.CookIslands, "CK  Cook Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Chile, "CL  Chile"));
            countryTypes.Add(new IdType(CountryTypeId.Cameroon, "CM  Cameroon"));
            countryTypes.Add(new IdType(CountryTypeId.China, "CN  China"));
            countryTypes.Add(new IdType(CountryTypeId.Colombia, "CO  Colombia"));
            countryTypes.Add(new IdType(CountryTypeId.CostaRica, "CR  Costa Rica"));
            countryTypes.Add(new IdType(CountryTypeId.SerbiaMontenegro, "CS  Serbia and Montenegro"));
            countryTypes.Add(new IdType(CountryTypeId.Cuba, "CU  Cuba"));
            countryTypes.Add(new IdType(CountryTypeId.CapeVerde, "CV  Cape Verde"));
            countryTypes.Add(new IdType(CountryTypeId.ChristmasIsland, "CX  Christmas Island"));
            countryTypes.Add(new IdType(CountryTypeId.Cyprus, "CY  Cyprus"));
            countryTypes.Add(new IdType(CountryTypeId.CzechRepublic, "CZ  Czech Republic"));
            countryTypes.Add(new IdType(CountryTypeId.Germany, "DE  Germany"));
            countryTypes.Add(new IdType(CountryTypeId.Djibouti, "DJ  Djibouti"));
            countryTypes.Add(new IdType(CountryTypeId.Denmark, "DK  Denmark"));
            countryTypes.Add(new IdType(CountryTypeId.Dominica, "DM  Dominica"));
            countryTypes.Add(new IdType(CountryTypeId.DominicanRepublic, "DO  Dominican Republic"));
            countryTypes.Add(new IdType(CountryTypeId.Algeria, "DZ  Algeria"));
            countryTypes.Add(new IdType(CountryTypeId.Ecuador, "EC  Ecuador"));
            countryTypes.Add(new IdType(CountryTypeId.Estonia, "EE  Estonia"));
            countryTypes.Add(new IdType(CountryTypeId.Egypt, "EG  Egypt"));
            countryTypes.Add(new IdType(CountryTypeId.WesternSahara, "EH  Western Sahara"));
            countryTypes.Add(new IdType(CountryTypeId.Eritrea, "ER  Eritrea"));
            countryTypes.Add(new IdType(CountryTypeId.Spain, "ES  Spain"));
            countryTypes.Add(new IdType(CountryTypeId.Ethiopia, "ET  Ethiopia"));
            countryTypes.Add(new IdType(CountryTypeId.Finland, "FI  Finland"));
            countryTypes.Add(new IdType(CountryTypeId.Fiji, "FJ  Fiji"));
            countryTypes.Add(new IdType(CountryTypeId.FalklandIslands, "FK  Falkland Islands (Malvinas)"));
            countryTypes.Add(new IdType(CountryTypeId.FederatedStatesMicronesia, "FM  Federated States of Micronesia"));
            countryTypes.Add(new IdType(CountryTypeId.FaroeIslands, "FO  Faroe Islands"));
            countryTypes.Add(new IdType(CountryTypeId.France, "FR  France"));
            countryTypes.Add(new IdType(CountryTypeId.FranceMetropolitan, "FX   France, Metropolitan"));
            countryTypes.Add(new IdType(CountryTypeId.Gabon, "GA  Gabon"));
            countryTypes.Add(new IdType(CountryTypeId.GreatBritain, "GB  Great Britain (UK)"));
            countryTypes.Add(new IdType(CountryTypeId.Grenada, "GD  Grenada"));
            countryTypes.Add(new IdType(CountryTypeId.Georgia, "GE  Georgia"));
            countryTypes.Add(new IdType(CountryTypeId.FrenchGuiana, "GF  French Guiana"));
            countryTypes.Add(new IdType(CountryTypeId.Ghana, "GH  Ghana"));
            countryTypes.Add(new IdType(CountryTypeId.Gibraltar, "GI  Gibraltar"));
            countryTypes.Add(new IdType(CountryTypeId.Greenland, "GL  Greenland"));
            countryTypes.Add(new IdType(CountryTypeId.Gambia, "GM  Gambia"));
            countryTypes.Add(new IdType(CountryTypeId.Guinea, "GN  Guinea"));
            countryTypes.Add(new IdType(CountryTypeId.Guadeloupe, "GP  Guadeloupe"));
            countryTypes.Add(new IdType(CountryTypeId.EquatorialGuinea, "GQ  Equatorial Guinea"));
            countryTypes.Add(new IdType(CountryTypeId.Greece, "GR  Greece"));
            countryTypes.Add(new IdType(CountryTypeId.SGeorgiaSSandwichIslands, "GS  S. Georgia and S. Sandwich Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Guatemala, "GT  Guatemala"));
            countryTypes.Add(new IdType(CountryTypeId.Guam, "GU  Guam"));
            countryTypes.Add(new IdType(CountryTypeId.GuineaBissau, "GW  Guinea-Bissau"));
            countryTypes.Add(new IdType(CountryTypeId.Guyana, "GY  Guyana"));
            countryTypes.Add(new IdType(CountryTypeId.HongKong, "HK  Hong Kong"));
            countryTypes.Add(new IdType(CountryTypeId.HeardIslandMcDonaldIslands, "HM  Heard Island and McDonald Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Honduras, "HN  Honduras"));
            countryTypes.Add(new IdType(CountryTypeId.Croatia, "HR  Croatia (Hrvatska)"));
            countryTypes.Add(new IdType(CountryTypeId.Haiti, "HT  Haiti"));
            countryTypes.Add(new IdType(CountryTypeId.Hungary, "HU  Hungary"));
            countryTypes.Add(new IdType(CountryTypeId.Indonesia, "ID  Indonesia"));
            countryTypes.Add(new IdType(CountryTypeId.Ireland, "IE  Ireland"));
            countryTypes.Add(new IdType(CountryTypeId.Israel, "IL  Israel"));
            countryTypes.Add(new IdType(CountryTypeId.India, "IN  India"));
            countryTypes.Add(new IdType(CountryTypeId.BritishIndianOceanTerritory, "IO  British Indian Ocean Territory"));
            countryTypes.Add(new IdType(CountryTypeId.Iraq, "IQ  Iraq"));
            countryTypes.Add(new IdType(CountryTypeId.Iran, "IR  Iran"));
            countryTypes.Add(new IdType(CountryTypeId.Iceland, "IS  Iceland"));
            countryTypes.Add(new IdType(CountryTypeId.Italy, "IT  Italy"));
            countryTypes.Add(new IdType(CountryTypeId.Jamaica, "JM  Jamaica"));
            countryTypes.Add(new IdType(CountryTypeId.Jordan, "JO  Jordan"));
            countryTypes.Add(new IdType(CountryTypeId.Japan, "JP  Japan"));
            countryTypes.Add(new IdType(CountryTypeId.Kenya, "KE  Kenya"));
            countryTypes.Add(new IdType(CountryTypeId.Kyrgyzstan, "KG  Kyrgyzstan"));
            countryTypes.Add(new IdType(CountryTypeId.Cambodia, "KH  Cambodia"));
            countryTypes.Add(new IdType(CountryTypeId.Kiribati, "KI  Kiribati"));
            countryTypes.Add(new IdType(CountryTypeId.Comoros, "KM  Comoros"));
            countryTypes.Add(new IdType(CountryTypeId.SaintKittsNevis, "KN  Saint Kitts and Nevis"));
            countryTypes.Add(new IdType(CountryTypeId.KoreaNorth, "KP Korea (North)"));
            countryTypes.Add(new IdType(CountryTypeId.KoreaSouth, "KR Korea (South)"));
            countryTypes.Add(new IdType(CountryTypeId.Kuwait, "KW  Kuwait"));
            countryTypes.Add(new IdType(CountryTypeId.CaymanIslands, "KY  Cayman Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Kazakhstan, "KZ  Kazakhstan"));
            countryTypes.Add(new IdType(CountryTypeId.Laos, "LA  Laos"));
            countryTypes.Add(new IdType(CountryTypeId.Lebanon, "LB  Lebanon"));
            countryTypes.Add(new IdType(CountryTypeId.SaintLucia, "LC  Saint Lucia"));
            countryTypes.Add(new IdType(CountryTypeId.Liechtenstein, "LI  Liechtenstein"));
            countryTypes.Add(new IdType(CountryTypeId.SriLanka, "LK  Sri Lanka"));
            countryTypes.Add(new IdType(CountryTypeId.Liberia, "LR  Liberia"));
            countryTypes.Add(new IdType(CountryTypeId.Lesotho, "LS  Lesotho"));
            countryTypes.Add(new IdType(CountryTypeId.Lithuania, "LT  Lithuania"));
            countryTypes.Add(new IdType(CountryTypeId.Luxembourg, "LU  Luxembourg"));
            countryTypes.Add(new IdType(CountryTypeId.Latvia, "LV  Latvia"));
            countryTypes.Add(new IdType(CountryTypeId.Libya, "LY  Libya"));
            countryTypes.Add(new IdType(CountryTypeId.Morocco, "MA  Morocco"));
            countryTypes.Add(new IdType(CountryTypeId.Monaco, "MC  Monaco"));
            countryTypes.Add(new IdType(CountryTypeId.Moldova, "MD  Moldova"));
            countryTypes.Add(new IdType(CountryTypeId.Madagascar, "MG  Madagascar"));
            countryTypes.Add(new IdType(CountryTypeId.MarshallIslands, "MH  Marshall Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Macedonia, "MK  Macedonia"));
            countryTypes.Add(new IdType(CountryTypeId.Mali, "ML  Mali"));
            countryTypes.Add(new IdType(CountryTypeId.Myanmar, "MM  Myanmar"));
            countryTypes.Add(new IdType(CountryTypeId.Mongolia, "MN  Mongolia"));
            countryTypes.Add(new IdType(CountryTypeId.Macao, "MO  Macao"));
            countryTypes.Add(new IdType(CountryTypeId.NorthernMarianaIslands, "MP  Northern Mariana Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Martinique, "MQ  Martinique"));
            countryTypes.Add(new IdType(CountryTypeId.Mauritania, "MR  Mauritania"));
            countryTypes.Add(new IdType(CountryTypeId.Montserrat, "MS  Montserrat"));
            countryTypes.Add(new IdType(CountryTypeId.Malta, "MT  Malta"));
            countryTypes.Add(new IdType(CountryTypeId.Mauritius, "MU  Mauritius"));
            countryTypes.Add(new IdType(CountryTypeId.Maldives, "MV  Maldives"));
            countryTypes.Add(new IdType(CountryTypeId.Malawi, "MW  Malawi"));
            countryTypes.Add(new IdType(CountryTypeId.Mexico, "MX  Mexico"));
            countryTypes.Add(new IdType(CountryTypeId.Malaysia, "MY  Malaysia"));
            countryTypes.Add(new IdType(CountryTypeId.Mozambique, "MZ  Mozambique"));
            countryTypes.Add(new IdType(CountryTypeId.Namibia, "NA  Namibia"));
            countryTypes.Add(new IdType(CountryTypeId.NewCaledonia, "NC  New Caledonia"));
            countryTypes.Add(new IdType(CountryTypeId.Niger, "NE  Niger"));
            countryTypes.Add(new IdType(CountryTypeId.NorfolkIsland, "NF  Norfolk Island"));
            countryTypes.Add(new IdType(CountryTypeId.Nigeria, "NG  Nigeria"));
            countryTypes.Add(new IdType(CountryTypeId.Nicaragua, "NI  Nicaragua"));
            countryTypes.Add(new IdType(CountryTypeId.Netherlands, "NL  Netherlands"));
            countryTypes.Add(new IdType(CountryTypeId.Norway, "NO  Norway"));
            countryTypes.Add(new IdType(CountryTypeId.Nepal, "NP  Nepal"));
            countryTypes.Add(new IdType(CountryTypeId.Nauru, "NR  Nauru"));
            countryTypes.Add(new IdType(CountryTypeId.Niue, "NU  Niue"));
            countryTypes.Add(new IdType(CountryTypeId.NewZealand, "NZ  New Zealand (Aotearoa)"));
            countryTypes.Add(new IdType(CountryTypeId.Oman, "OM  Oman"));
            countryTypes.Add(new IdType(CountryTypeId.Panama, "PA  Panama"));
            countryTypes.Add(new IdType(CountryTypeId.Peru, "PE  Peru"));
            countryTypes.Add(new IdType(CountryTypeId.FrenchPolynesia, "PF  French Polynesia"));
            countryTypes.Add(new IdType(CountryTypeId.PapuaNewGuinea, "PG  Papua New Guinea"));
            countryTypes.Add(new IdType(CountryTypeId.Philippines, "PH  Philippines"));
            countryTypes.Add(new IdType(CountryTypeId.Pakistan, "PK  Pakistan"));
            countryTypes.Add(new IdType(CountryTypeId.Poland, "PL  Poland"));
            countryTypes.Add(new IdType(CountryTypeId.SaintPierreMiquelon, "PM  Saint Pierre and Miquelon"));
            countryTypes.Add(new IdType(CountryTypeId.Pitcairn, "PN  Pitcairn"));
            countryTypes.Add(new IdType(CountryTypeId.PuertoRico, "PR  Puerto Rico"));
            countryTypes.Add(new IdType(CountryTypeId.PalestinianTerritory, "PS  Palestinian Territory"));
            countryTypes.Add(new IdType(CountryTypeId.Portugal, "PT  Portugal"));
            countryTypes.Add(new IdType(CountryTypeId.Palau, "PW  Palau"));
            countryTypes.Add(new IdType(CountryTypeId.Paraguay, "PY  Paraguay"));
            countryTypes.Add(new IdType(CountryTypeId.Qatar, "QA  Qatar"));
            countryTypes.Add(new IdType(CountryTypeId.Reunion, "RE  Reunion"));
            countryTypes.Add(new IdType(CountryTypeId.Romania, "RO  Romania"));
            countryTypes.Add(new IdType(CountryTypeId.RussianFederation, "RU  Russian Federation"));
            countryTypes.Add(new IdType(CountryTypeId.Rwanda, "RW  Rwanda"));
            countryTypes.Add(new IdType(CountryTypeId.SaudiArabia, "SA  Saudi Arabia"));
            countryTypes.Add(new IdType(CountryTypeId.SolomonIslands, "SB  Solomon Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Seychelles, "SC  Seychelles"));
            countryTypes.Add(new IdType(CountryTypeId.Sudan, "SD  Sudan"));
            countryTypes.Add(new IdType(CountryTypeId.Sweden, "SE  Sweden"));
            countryTypes.Add(new IdType(CountryTypeId.Singapore, "SG  Singapore"));
            countryTypes.Add(new IdType(CountryTypeId.SaintHelena, "SH  Saint Helena"));
            countryTypes.Add(new IdType(CountryTypeId.Slovenia, "SI  Slovenia"));
            countryTypes.Add(new IdType(CountryTypeId.SvalbardJanMayen, "SJ  Svalbard and Jan Mayen"));
            countryTypes.Add(new IdType(CountryTypeId.Slovakia, "SK  Slovakia"));
            countryTypes.Add(new IdType(CountryTypeId.SierraLeone, "SL  Sierra Leone"));
            countryTypes.Add(new IdType(CountryTypeId.SanMarino, "SM  San Marino"));
            countryTypes.Add(new IdType(CountryTypeId.Senegal, "SN  Senegal"));
            countryTypes.Add(new IdType(CountryTypeId.Somalia, "SO  Somalia"));
            countryTypes.Add(new IdType(CountryTypeId.Suriname, "SR  Suriname"));
            countryTypes.Add(new IdType(CountryTypeId.SaoTomePrincipe, "ST  Sao Tome and Principe"));
            countryTypes.Add(new IdType(CountryTypeId.USSR, "SU  USSR (former)"));
            countryTypes.Add(new IdType(CountryTypeId.ElSalvador, "SV  El Salvador"));
            countryTypes.Add(new IdType(CountryTypeId.Syria, "SY  Syria"));
            countryTypes.Add(new IdType(CountryTypeId.Swaziland, "SZ  Swaziland"));
            countryTypes.Add(new IdType(CountryTypeId.TurksCaicosIslands, "TC  Turks and Caicos Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Chad, "TD  Chad"));
            countryTypes.Add(new IdType(CountryTypeId.FrenchSouthernTerritories, "TF  French Southern Territories"));
            countryTypes.Add(new IdType(CountryTypeId.Togo, "TG  Togo"));
            countryTypes.Add(new IdType(CountryTypeId.Thailand, "TH  Thailand"));
            countryTypes.Add(new IdType(CountryTypeId.Tajikistan, "TJ  Tajikistan"));
            countryTypes.Add(new IdType(CountryTypeId.Tokelau, "TK  Tokelau"));
            countryTypes.Add(new IdType(CountryTypeId.TimorLeste, "TL  Timor-Leste"));
            countryTypes.Add(new IdType(CountryTypeId.Turkmenistan, "TM  Turkmenistan"));
            countryTypes.Add(new IdType(CountryTypeId.Tunisia, "TN  Tunisia"));
            countryTypes.Add(new IdType(CountryTypeId.Tonga, "TO  Tonga"));
            countryTypes.Add(new IdType(CountryTypeId.EastTimor, "TP  East Timor"));
            countryTypes.Add(new IdType(CountryTypeId.Turkey, "TR  Turkey"));
            countryTypes.Add(new IdType(CountryTypeId.TrinidadTobago, "TT  Trinidad and Tobago"));
            countryTypes.Add(new IdType(CountryTypeId.Tuvalu, "TV  Tuvalu"));
            countryTypes.Add(new IdType(CountryTypeId.Taiwan, "TW  Taiwan"));
            countryTypes.Add(new IdType(CountryTypeId.Tanzania, "TZ  Tanzania"));
            countryTypes.Add(new IdType(CountryTypeId.Ukraine, "UA  Ukraine"));
            countryTypes.Add(new IdType(CountryTypeId.Uganda, "UG  Uganda"));
            countryTypes.Add(new IdType(CountryTypeId.UnitedKingdom, "UK  United Kingdom"));
            countryTypes.Add(new IdType(CountryTypeId.UnitedStatesMinorOutlyingIslands, "UM  United States Minor Outlying Islands"));
            countryTypes.Add(new IdType(CountryTypeId.UnitedStates, "US  United States"));
            countryTypes.Add(new IdType(CountryTypeId.Uruguay, "UY  Uruguay"));
            countryTypes.Add(new IdType(CountryTypeId.Uzbekistan, "UZ  Uzbekistan"));
            countryTypes.Add(new IdType(CountryTypeId.VaticanCityState, "VA  Vatican City State (Holy See)"));
            countryTypes.Add(new IdType(CountryTypeId.SaintVincentGrenadines, "VC  Saint Vincent and the Grenadines"));
            countryTypes.Add(new IdType(CountryTypeId.Venezuela, "VE  Venezuela"));
            countryTypes.Add(new IdType(CountryTypeId.VirginIslandsBritish, "VG  Virgin Islands (British)"));
            countryTypes.Add(new IdType(CountryTypeId.VirginIslandsUS, "VI  Virgin Islands (U.S.)"));
            countryTypes.Add(new IdType(CountryTypeId.VietNam, "VN  Viet Nam"));
            countryTypes.Add(new IdType(CountryTypeId.Vanuatu, "VU  Vanuatu"));
            countryTypes.Add(new IdType(CountryTypeId.WallisFutuna, "WF  Wallis and Futuna"));
            countryTypes.Add(new IdType(CountryTypeId.Samoa, "WS  Samoa"));
            countryTypes.Add(new IdType(CountryTypeId.Yemen, "YE  Yemen"));
            countryTypes.Add(new IdType(CountryTypeId.Mayotte, "YT  Mayotte"));
            countryTypes.Add(new IdType(CountryTypeId.Yugoslavia, "YU  Yugoslavia (former)"));
            countryTypes.Add(new IdType(CountryTypeId.SouthAfrica, "ZA  South Africa"));
            countryTypes.Add(new IdType(CountryTypeId.Zambia, "ZM  Zambia"));
            countryTypes.Add(new IdType(CountryTypeId.Zaire, "ZR  Zaire (former)"));
            countryTypes.Add(new IdType(CountryTypeId.Zimbabwe, "ZW  Zimbabwe"));

            countryTypesByCode["AD"] = CountryTypeId.Andorra;
            countryTypesByCode["AE"] = CountryTypeId.UnitedArabEmirates;
            countryTypesByCode["AF"] = CountryTypeId.Afghanistan;
            countryTypesByCode["AG"] = CountryTypeId.AntiguaBarbuda;
            countryTypesByCode["AI"] = CountryTypeId.Anguilla;
            countryTypesByCode["AL"] = CountryTypeId.Albania;
            countryTypesByCode["AM"] = CountryTypeId.Armenia;
            countryTypesByCode["AN"] = CountryTypeId.NetherlandsAntilles;
            countryTypesByCode["AO"] = CountryTypeId.Angola;
            countryTypesByCode["AQ"] = CountryTypeId.Antarctica;
            countryTypesByCode["AR"] = CountryTypeId.Argentina;
            countryTypesByCode["AS"] = CountryTypeId.AmericanSamoa;
            countryTypesByCode["AT"] = CountryTypeId.Austria;
            countryTypesByCode["AU"] = CountryTypeId.Australia;
            countryTypesByCode["AW"] = CountryTypeId.Aruba;
            countryTypesByCode["AX"] = CountryTypeId.AlandIslands;
            countryTypesByCode["AZ"] = CountryTypeId.Azerbaijan;
            countryTypesByCode["BA"] = CountryTypeId.BosniaHerzegovina;
            countryTypesByCode["BB"] = CountryTypeId.Barbados;
            countryTypesByCode["BD"] = CountryTypeId.Bangladesh;
            countryTypesByCode["BE"] = CountryTypeId.Belgium;
            countryTypesByCode["BF"] = CountryTypeId.BurkinaFaso;
            countryTypesByCode["BG"] = CountryTypeId.Bulgaria;
            countryTypesByCode["BH"] = CountryTypeId.Bahrain;
            countryTypesByCode["BI"] = CountryTypeId.Burundi;
            countryTypesByCode["BJ"] = CountryTypeId.Benin;
            countryTypesByCode["BM"] = CountryTypeId.Bermuda;
            countryTypesByCode["BN"] = CountryTypeId.BruneiDarussalam;
            countryTypesByCode["BO"] = CountryTypeId.Bolivia;
            countryTypesByCode["BR"] = CountryTypeId.Brazil;
            countryTypesByCode["BS"] = CountryTypeId.Bahamas;
            countryTypesByCode["BT"] = CountryTypeId.Bhutan;
            countryTypesByCode["BV"] = CountryTypeId.BouvetIsland;
            countryTypesByCode["BW"] = CountryTypeId.Botswana;
            countryTypesByCode["BY"] = CountryTypeId.Belarus;
            countryTypesByCode["BZ"] = CountryTypeId.Belize;
            countryTypesByCode["CA"] = CountryTypeId.Canada;
            countryTypesByCode["CC"] = CountryTypeId.CocosKeelingIslands;
            countryTypesByCode["CD"] = CountryTypeId.DemocraticRepublicCongo;
            countryTypesByCode["CF"] = CountryTypeId.CentralAfricanRepublic;
            countryTypesByCode["CG"] = CountryTypeId.Congo;
            countryTypesByCode["CH"] = CountryTypeId.Switzerland;
            countryTypesByCode["CI"] = CountryTypeId.CoteDIvoire;
            countryTypesByCode["CK"] = CountryTypeId.CookIslands;
            countryTypesByCode["CL"] = CountryTypeId.Chile;
            countryTypesByCode["CM"] = CountryTypeId.Cameroon;
            countryTypesByCode["CN"] = CountryTypeId.China;
            countryTypesByCode["CO"] = CountryTypeId.Colombia;
            countryTypesByCode["CR"] = CountryTypeId.CostaRica;
            countryTypesByCode["CS"] = CountryTypeId.SerbiaMontenegro;
            countryTypesByCode["CU"] = CountryTypeId.Cuba;
            countryTypesByCode["CV"] = CountryTypeId.CapeVerde;
            countryTypesByCode["CX"] = CountryTypeId.ChristmasIsland;
            countryTypesByCode["CY"] = CountryTypeId.Cyprus;
            countryTypesByCode["CZ"] = CountryTypeId.CzechRepublic;
            countryTypesByCode["DE"] = CountryTypeId.Germany;
            countryTypesByCode["DJ"] = CountryTypeId.Djibouti;
            countryTypesByCode["DK"] = CountryTypeId.Denmark;
            countryTypesByCode["DM"] = CountryTypeId.Dominica;
            countryTypesByCode["DO"] = CountryTypeId.DominicanRepublic;
            countryTypesByCode["DZ"] = CountryTypeId.Algeria;
            countryTypesByCode["EC"] = CountryTypeId.Ecuador;
            countryTypesByCode["EE"] = CountryTypeId.Estonia;
            countryTypesByCode["EG"] = CountryTypeId.Egypt;
            countryTypesByCode["EH"] = CountryTypeId.WesternSahara;
            countryTypesByCode["ER"] = CountryTypeId.Eritrea;
            countryTypesByCode["ES"] = CountryTypeId.Spain;
            countryTypesByCode["ET"] = CountryTypeId.Ethiopia;
            countryTypesByCode["FI"] = CountryTypeId.Finland;
            countryTypesByCode["FJ"] = CountryTypeId.Fiji;
            countryTypesByCode["FK"] = CountryTypeId.FalklandIslands;
            countryTypesByCode["FM"] = CountryTypeId.FederatedStatesMicronesia;
            countryTypesByCode["FO"] = CountryTypeId.FaroeIslands;
            countryTypesByCode["FR"] = CountryTypeId.France;
            countryTypesByCode["FX"] = CountryTypeId.FranceMetropolitan;
            countryTypesByCode["GA"] = CountryTypeId.Gabon;
            countryTypesByCode["GB"] = CountryTypeId.GreatBritain;
            countryTypesByCode["GD"] = CountryTypeId.Grenada;
            countryTypesByCode["GE"] = CountryTypeId.Georgia;
            countryTypesByCode["GF"] = CountryTypeId.FrenchGuiana;
            countryTypesByCode["GH"] = CountryTypeId.Ghana;
            countryTypesByCode["GI"] = CountryTypeId.Gibraltar;
            countryTypesByCode["GL"] = CountryTypeId.Greenland;
            countryTypesByCode["GM"] = CountryTypeId.Gambia;
            countryTypesByCode["GN"] = CountryTypeId.Guinea;
            countryTypesByCode["GP"] = CountryTypeId.Guadeloupe;
            countryTypesByCode["GQ"] = CountryTypeId.EquatorialGuinea;
            countryTypesByCode["GR"] = CountryTypeId.Greece;
            countryTypesByCode["GS"] = CountryTypeId.SGeorgiaSSandwichIslands;
            countryTypesByCode["GT"] = CountryTypeId.Guatemala;
            countryTypesByCode["GU"] = CountryTypeId.Guam;
            countryTypesByCode["GW"] = CountryTypeId.GuineaBissau;
            countryTypesByCode["GY"] = CountryTypeId.Guyana;
            countryTypesByCode["HK"] = CountryTypeId.HongKong;
            countryTypesByCode["HM"] = CountryTypeId.HeardIslandMcDonaldIslands;
            countryTypesByCode["HN"] = CountryTypeId.Honduras;
            countryTypesByCode["HR"] = CountryTypeId.Croatia;
            countryTypesByCode["HT"] = CountryTypeId.Haiti;
            countryTypesByCode["HU"] = CountryTypeId.Hungary;
            countryTypesByCode["ID"] = CountryTypeId.Indonesia;
            countryTypesByCode["IE"] = CountryTypeId.Ireland;
            countryTypesByCode["IL"] = CountryTypeId.Israel;
            countryTypesByCode["IN"] = CountryTypeId.India;
            countryTypesByCode["IO"] = CountryTypeId.BritishIndianOceanTerritory;
            countryTypesByCode["IQ"] = CountryTypeId.Iraq;
            countryTypesByCode["IR"] = CountryTypeId.Iran;
            countryTypesByCode["IS"] = CountryTypeId.Iceland;
            countryTypesByCode["IT"] = CountryTypeId.Italy;
            countryTypesByCode["JM"] = CountryTypeId.Jamaica;
            countryTypesByCode["JO"] = CountryTypeId.Jordan;
            countryTypesByCode["JP"] = CountryTypeId.Japan;
            countryTypesByCode["KE"] = CountryTypeId.Kenya;
            countryTypesByCode["KG"] = CountryTypeId.Kyrgyzstan;
            countryTypesByCode["KH"] = CountryTypeId.Cambodia;
            countryTypesByCode["KI"] = CountryTypeId.Kiribati;
            countryTypesByCode["KM"] = CountryTypeId.Comoros;
            countryTypesByCode["KN"] = CountryTypeId.SaintKittsNevis;
            countryTypesByCode["KP"] = CountryTypeId.KoreaNorth;
            countryTypesByCode["KR"] = CountryTypeId.KoreaSouth;
            countryTypesByCode["KW"] = CountryTypeId.Kuwait;
            countryTypesByCode["KY"] = CountryTypeId.CaymanIslands;
            countryTypesByCode["KZ"] = CountryTypeId.Kazakhstan;
            countryTypesByCode["LA"] = CountryTypeId.Laos;
            countryTypesByCode["LB"] = CountryTypeId.Lebanon;
            countryTypesByCode["LC"] = CountryTypeId.SaintLucia;
            countryTypesByCode["LI"] = CountryTypeId.Liechtenstein;
            countryTypesByCode["LK"] = CountryTypeId.SriLanka;
            countryTypesByCode["LR"] = CountryTypeId.Liberia;
            countryTypesByCode["LS"] = CountryTypeId.Lesotho;
            countryTypesByCode["LT"] = CountryTypeId.Lithuania;
            countryTypesByCode["LU"] = CountryTypeId.Luxembourg;
            countryTypesByCode["LV"] = CountryTypeId.Latvia;
            countryTypesByCode["LY"] = CountryTypeId.Libya;
            countryTypesByCode["MA"] = CountryTypeId.Morocco;
            countryTypesByCode["MC"] = CountryTypeId.Monaco;
            countryTypesByCode["MD"] = CountryTypeId.Moldova;
            countryTypesByCode["MG"] = CountryTypeId.Madagascar;
            countryTypesByCode["MH"] = CountryTypeId.MarshallIslands;
            countryTypesByCode["MK"] = CountryTypeId.Macedonia;
            countryTypesByCode["ML"] = CountryTypeId.Mali;
            countryTypesByCode["MM"] = CountryTypeId.Myanmar;
            countryTypesByCode["MN"] = CountryTypeId.Mongolia;
            countryTypesByCode["MO"] = CountryTypeId.Macao;
            countryTypesByCode["MP"] = CountryTypeId.NorthernMarianaIslands;
            countryTypesByCode["MQ"] = CountryTypeId.Martinique;
            countryTypesByCode["MR"] = CountryTypeId.Mauritania;
            countryTypesByCode["MS"] = CountryTypeId.Montserrat;
            countryTypesByCode["MT"] = CountryTypeId.Malta;
            countryTypesByCode["MU"] = CountryTypeId.Mauritius;
            countryTypesByCode["MV"] = CountryTypeId.Maldives;
            countryTypesByCode["MW"] = CountryTypeId.Malawi;
            countryTypesByCode["MX"] = CountryTypeId.Mexico;
            countryTypesByCode["MY"] = CountryTypeId.Malaysia;
            countryTypesByCode["MZ"] = CountryTypeId.Mozambique;
            countryTypesByCode["NA"] = CountryTypeId.Namibia;
            countryTypesByCode["NC"] = CountryTypeId.NewCaledonia;
            countryTypesByCode["NE"] = CountryTypeId.Niger;
            countryTypesByCode["NF"] = CountryTypeId.NorfolkIsland;
            countryTypesByCode["NG"] = CountryTypeId.Nigeria;
            countryTypesByCode["NI"] = CountryTypeId.Nicaragua;
            countryTypesByCode["NL"] = CountryTypeId.Netherlands;
            countryTypesByCode["NO"] = CountryTypeId.Norway;
            countryTypesByCode["NP"] = CountryTypeId.Nepal;
            countryTypesByCode["NR"] = CountryTypeId.Nauru;
            countryTypesByCode["NU"] = CountryTypeId.Niue;
            countryTypesByCode["NZ"] = CountryTypeId.NewZealand;
            countryTypesByCode["OM"] = CountryTypeId.Oman;
            countryTypesByCode["PA"] = CountryTypeId.Panama;
            countryTypesByCode["PE"] = CountryTypeId.Peru;
            countryTypesByCode["PF"] = CountryTypeId.FrenchPolynesia;
            countryTypesByCode["PG"] = CountryTypeId.PapuaNewGuinea;
            countryTypesByCode["PH"] = CountryTypeId.Philippines;
            countryTypesByCode["PK"] = CountryTypeId.Pakistan;
            countryTypesByCode["PL"] = CountryTypeId.Poland;
            countryTypesByCode["PM"] = CountryTypeId.SaintPierreMiquelon;
            countryTypesByCode["PN"] = CountryTypeId.Pitcairn;
            countryTypesByCode["PR"] = CountryTypeId.PuertoRico;
            countryTypesByCode["PS"] = CountryTypeId.PalestinianTerritory;
            countryTypesByCode["PT"] = CountryTypeId.Portugal;
            countryTypesByCode["PW"] = CountryTypeId.Palau;
            countryTypesByCode["PY"] = CountryTypeId.Paraguay;
            countryTypesByCode["QA"] = CountryTypeId.Qatar;
            countryTypesByCode["RE"] = CountryTypeId.Reunion;
            countryTypesByCode["RO"] = CountryTypeId.Romania;
            countryTypesByCode["RU"] = CountryTypeId.RussianFederation;
            countryTypesByCode["RW"] = CountryTypeId.Rwanda;
            countryTypesByCode["SA"] = CountryTypeId.SaudiArabia;
            countryTypesByCode["SB"] = CountryTypeId.SolomonIslands;
            countryTypesByCode["SC"] = CountryTypeId.Seychelles;
            countryTypesByCode["SD"] = CountryTypeId.Sudan;
            countryTypesByCode["SE"] = CountryTypeId.Sweden;
            countryTypesByCode["SG"] = CountryTypeId.Singapore;
            countryTypesByCode["SH"] = CountryTypeId.SaintHelena;
            countryTypesByCode["SI"] = CountryTypeId.Slovenia;
            countryTypesByCode["SJ"] = CountryTypeId.SvalbardJanMayen;
            countryTypesByCode["SK"] = CountryTypeId.Slovakia;
            countryTypesByCode["SL"] = CountryTypeId.SierraLeone;
            countryTypesByCode["SM"] = CountryTypeId.SanMarino;
            countryTypesByCode["SN"] = CountryTypeId.Senegal;
            countryTypesByCode["SO"] = CountryTypeId.Somalia;
            countryTypesByCode["SR"] = CountryTypeId.Suriname;
            countryTypesByCode["ST"] = CountryTypeId.SaoTomePrincipe;
            countryTypesByCode["SU"] = CountryTypeId.USSR;
            countryTypesByCode["SV"] = CountryTypeId.ElSalvador;
            countryTypesByCode["SY"] = CountryTypeId.Syria;
            countryTypesByCode["SZ"] = CountryTypeId.Swaziland;
            countryTypesByCode["TC"] = CountryTypeId.TurksCaicosIslands;
            countryTypesByCode["TD"] = CountryTypeId.Chad;
            countryTypesByCode["TF"] = CountryTypeId.FrenchSouthernTerritories;
            countryTypesByCode["TG"] = CountryTypeId.Togo;
            countryTypesByCode["TH"] = CountryTypeId.Thailand;
            countryTypesByCode["TJ"] = CountryTypeId.Tajikistan;
            countryTypesByCode["TK"] = CountryTypeId.Tokelau;
            countryTypesByCode["TL"] = CountryTypeId.TimorLeste;
            countryTypesByCode["TM"] = CountryTypeId.Turkmenistan;
            countryTypesByCode["TN"] = CountryTypeId.Tunisia;
            countryTypesByCode["TO"] = CountryTypeId.Tonga;
            countryTypesByCode["TP"] = CountryTypeId.EastTimor;
            countryTypesByCode["TR"] = CountryTypeId.Turkey;
            countryTypesByCode["TT"] = CountryTypeId.TrinidadTobago;
            countryTypesByCode["TV"] = CountryTypeId.Tuvalu;
            countryTypesByCode["TW"] = CountryTypeId.Taiwan;
            countryTypesByCode["TZ"] = CountryTypeId.Tanzania;
            countryTypesByCode["UA"] = CountryTypeId.Ukraine;
            countryTypesByCode["UG"] = CountryTypeId.Uganda;
            countryTypesByCode["UK"] = CountryTypeId.UnitedKingdom;
            countryTypesByCode["UM"] = CountryTypeId.UnitedStatesMinorOutlyingIslands;
            countryTypesByCode["US"] = CountryTypeId.UnitedStates;
            countryTypesByCode["UY"] = CountryTypeId.Uruguay;
            countryTypesByCode["UZ"] = CountryTypeId.Uzbekistan;
            countryTypesByCode["VA"] = CountryTypeId.VaticanCityState;
            countryTypesByCode["VC"] = CountryTypeId.SaintVincentGrenadines;
            countryTypesByCode["VE"] = CountryTypeId.Venezuela;
            countryTypesByCode["VG"] = CountryTypeId.VirginIslandsBritish;
            countryTypesByCode["VI"] = CountryTypeId.VirginIslandsUS;
            countryTypesByCode["VN"] = CountryTypeId.VietNam;
            countryTypesByCode["VU"] = CountryTypeId.Vanuatu;
            countryTypesByCode["WF"] = CountryTypeId.WallisFutuna;
            countryTypesByCode["WS"] = CountryTypeId.Samoa;
            countryTypesByCode["XX"] = CountryTypeId.Unknown;
            countryTypesByCode["YE"] = CountryTypeId.Yemen;
            countryTypesByCode["YT"] = CountryTypeId.Mayotte;
            countryTypesByCode["YU"] = CountryTypeId.Yugoslavia;
            countryTypesByCode["ZA"] = CountryTypeId.SouthAfrica;
            countryTypesByCode["ZM"] = CountryTypeId.Zambia;
            countryTypesByCode["ZR"] = CountryTypeId.Zaire;
            countryTypesByCode["ZW"] = CountryTypeId.Zimbabwe;
        }
        #endregion

        #region Attributes
        #endregion
    }

    public class AirspaceSorter : IComparer {
    
        int IComparer.Compare(object x, object y) {
            // TODO:  Add AirspaceSorter.Compare implementation
            int result;
            AirspaceClasses p1 = ((AirspaceElement)x).AirspaceClass;
            AirspaceClasses p2 = ((AirspaceElement)y).AirspaceClass;
            
            if (p1 == AirspaceClasses.D && ((AirspaceElement)x).AirspaceType == AirspaceTypes.ControlZone) {
                result = 1;
            }
            else if (p2 == AirspaceClasses.D && ((AirspaceElement)y).AirspaceType == AirspaceTypes.ControlZone) {
                result = -1;
            }
            else if (p1 == p2) {
                result = 0;
            }
            else if (p1 > p2) {
                result = -1;
            }
            else {
                result = 1;
            }
            return result;
        }
    }

    public class WayPointSorter : IComparer {
        public int Compare(object x, object y) {
            // TODO:  Add WayPointSorter.Compare implementation
            return ((WayPoint)x).longName.CompareTo(((WayPoint)y).longName);
        }
    }

}
