// name   : DMSt2004FAI.cs
// author : Harald Maier
// date   : 02.04.2004
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.Collections;
using System.Windows.Forms;

using SoaringDotNET.GUI;
using SoaringDotNET.Data;

namespace SoaringDotNET.Optimization
{
	/// <summary>
	/// 
	/// </summary>
	
	public class DMSt2004FAI : SoaringDotNET.Optimization.OptimizationBase
	{
        private const int LEGS = 3;

        public DMSt2004FAI(Flight flight) : 
            base(flight)
		{
			// 
			// TODO: Add constructor logic here
			//
            weight = new double [] {1.15, 1.15, 1.15};
            optimizedTask.OptimizedFor = OptimizeTypes.DMST;
		}
	
        public override void Optimize() {
            // TODO:  Add DMSt2004FAI.Optimize implementation
            int N;
            int i, ii;
            uint j, k, temp;
            int start, stop, step;
            int [] triangleStart = {0, 0, 0};
            int [] triangleStop = {0, 0, 0};
            uint curNumSteps, minNumSteps;
            ArrayList sections = new ArrayList();
            ArrayList triangles = new ArrayList();
            FAITriangle []calculatedPoints = new FAITriangle[25];
            FAITriangle triangle;
            FAITriangle best = new FAITriangle();
            FlightPoint p;
            WayPoint wp;

            start = flight.Start != -1 ? flight.Start : 0;
            stop = flight.Stop != -1 ? flight.Stop : flight.Count;

            status = OptimizataionStatus.Running;

            // first look for "closed" sections
            // DMSt rules say distance must be <= 1km
            optimizedTask.Clear();
            for (i = start; i < stop; i++) {
                for (ii = stop - 1; ii > i; ii--) {
                    if (flight[i].distanceKM(flight[ii]) <= 1.0 && (ii - i) > 10) {
                        //found a closed section
                        sections.Add(new int[]{i, ii});
                        i = ii;
                        break;
                    }
                }
            }

            for (i = 0; i < 25; i++) {
                triangles.Add(new FAITriangle());
            }
            // calculate largest FAI for each section
            foreach (int [] s in sections) {
                N = (s[1] - s[0] + 1);
                step = 1;
                minNumSteps = uint.MaxValue;

                // calc step size of each section
                for (k = 1; k < 100; k++) {
                    // ( N / k ) is the number of points used for the
                    // first optimization-run. If this number is smaller than 3, we
                    // cannot create a triangle.
                        
                    if (N / k < 3) {
                        break;
                    }
                    curNumSteps = 1;
                    temp = 1;

                    /*
                    * Calculating the step-width for the first run.
                    */
                    for (j = 3; j < N / k; j++) {
                        temp += j;
                        curNumSteps += temp;
                    }
                    temp = 1;

                    /*
                    * Calculating the number of steps for the second run.
                    */
                    for (j = 3; j < (k - 1) * 6; j++) {
                        temp += j;
                        curNumSteps += temp + j;
                    }

                    if (curNumSteps < minNumSteps) {
                        minNumSteps = curNumSteps;
                        step = (int)k;
                    }
                }

                triangleStart[0] = s[0];
                triangleStart[1] = s[0] + step - 1;
                triangleStart[2] = s[0] + (2 * step) - 1;
                triangleStop[0] = s[1] - (2 * step);
                triangleStop[1] = s[1] - step;
                triangleStop[2] = s[1];

                // first iteration step
                CalculateBestTask(triangleStart, triangleStop, step, s[1], triangles, true);
                triangles.CopyTo(calculatedPoints);
                // now refine iteration for every triangle
                foreach (FAITriangle t in calculatedPoints) {
                    triangleStart[0] = Math.Max(t.p1 - step, s[0]);
                    triangleStart[1] = t.p2 - step;
                    triangleStart[2] = Math.Min(t.p3 + step, s[1]) - (2 * step);

                    triangleStop[0] = triangleStart[0] + (2 * step);
                    triangleStop[1] = triangleStart[1] + (2 * step);
                    triangleStop[2] = Math.Min(triangleStart[2] + (2 * step), s[1]);

                    if (triangleStart[0] > 0 && triangleStart[1] > 0 && triangleStart[2] > 0 &&
                        triangleStop[0] < flight.Count && triangleStop[1] < flight.Count && triangleStop[2] < flight.Count) {
                        CalculateBestTask(triangleStart, triangleStop, 1, s[1], triangles, false);
                    }
                }

                triangle = (FAITriangle)triangles[0];
                if (triangle.length > best.length) {
                    best.length = triangle.length;
                    best.p1 = triangle.p1;
                    best.p2 = triangle.p2;
                    best.p3 = triangle.p3;
                    best.idx1 = s[0];
                    best.idx2 = s[1];
                }
            }

            if (best.length > 0.0) {
                if (best.p1 > best.p2 || best.p1 > best.p3 || best.p2 > best.p3) {
                    MessageBox.Show("No valid or finished FAI triangle found!", "Inforamtion", MessageBoxButtons.OK, MessageBoxIcon.Information);
                }
                else {
                    p = flight[flight.Takeoff];
                    wp = new WayPoint(p.Latitude, p.Longitude, p.pAlt, "Takeoff", WayPointTypeId.Unknown);
                    wp.reachPoint = p;
                    optimizedTask.Add(wp);

                    p = flight[best.idx1];
                    wp = new WayPoint(p.Latitude, p.Longitude, p.pAlt, "Start", WayPointTypeId.Unknown);
                    wp.reachPoint = p;
                    optimizedTask.Add(wp);

                    p = flight[best.p1];
                    wp = new WayPoint(p.Latitude, p.Longitude, p.pAlt, "TP 1", WayPointTypeId.Unknown);
                    wp.reachPoint = p;
                    optimizedTask.Add(wp);

                    p = flight[best.p2];
                    wp = new WayPoint(p.Latitude, p.Longitude, p.pAlt, "TP 2", WayPointTypeId.Unknown);
                    wp.reachPoint = p;
                    optimizedTask.Add(wp);

                    p = flight[best.p3];
                    wp = new WayPoint(p.Latitude, p.Longitude, p.pAlt, "TP 3", WayPointTypeId.Unknown);
                    wp.reachPoint = p;
                    optimizedTask.Add(wp);

                    p = flight[best.idx2];
                    wp = new WayPoint(p.Latitude, p.Longitude, p.pAlt, "Finish", WayPointTypeId.Unknown);
                    wp.reachPoint = p;
                    optimizedTask.Add(wp);

                    p = flight[flight.Landing];
                    wp = new WayPoint(p.Latitude, p.Longitude, p.pAlt, "Landing", WayPointTypeId.Unknown);
                    wp.reachPoint = p;
                    optimizedTask.Add(wp);
                }
            }

            status = OptimizataionStatus.Idle;            
        }

        private void CalculateBestTask(int [] triangleStart, int [] triangleStop, int step, int lastPoint, ArrayList triangles, bool firstRun) {
            int loopA, loopB, loopC, i, maxAB;
            double [][] lengthMatrix = flight.lengthMatrix;
            double [] lengthVector;
            double length, s1, s2, s3;
            FAITriangle tmpTriangle;

            for (loopA = triangleStart[0]; loopA < Math.Min(triangleStop[0], lastPoint); loopA += step) {
                if (firstRun) {
                    triangleStart[1] = loopA + step - 1;
                }
                
                for (loopB = triangleStart[1]; loopB < Math.Min(triangleStop[1], lastPoint); loopB += step) {
                    maxAB = Math.Max(loopA, loopB);
                    s1 = lengthMatrix[maxAB][Math.Min(loopA, loopB)];//p1.distanceKM(p2);
                    if (firstRun) {
                        triangleStart[2] = loopB + step - 1;
                    }
                    
                    for (loopC = triangleStart[2]; loopC < Math.Min(triangleStop[2], lastPoint); loopC += step) {
                        lengthVector = lengthMatrix[Math.Max(maxAB, loopC)];
                        s2 = lengthVector[loopB];//p2.distanceKM(p3);
                        s3 = lengthVector[loopA];//p3.distanceKM(p1);
                        length = s1 + s2 + s3;
                        if (Task.isFAI(length, s1, s2, s3)) {
                            for (i = triangles.Count - 1; i >= 0; i--) {
                                tmpTriangle = (FAITriangle)triangles[i];
                                if (length > tmpTriangle.length) {
                                    if (i == triangles.Count - 1) {
                                        // insert least one
                                        tmpTriangle.length = length;
                                        tmpTriangle.p1 = loopA;
                                        tmpTriangle.p2 = loopB;
                                        tmpTriangle.p3 = loopC;
                                    }
                                    else {
                                        //swap
                                        triangles[i] = triangles[i + 1];
                                        triangles[i + 1] = tmpTriangle;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        public override string ToString() {
            // TODO:  Add DMSt2004FAI.ToString implementation
            return "DMSt 2004 FAI Triangle";
        }
    }
}
