using System;
using System.IO;
using System.Text.RegularExpressions;
using System.Runtime.InteropServices;

using SoaringDotNET.FileFormats;

namespace SoaringDotNET.Database
{
	/// <summary>
	/// 
	/// </summary>
    public class SoaringPilotTerrainGenerator {
        private string inputPath = "";
        private string [] DEMNames = {"W180N90.DEM", "W140N90.DEM", "W100N90.DEM", "W060N90.DEM", "W020N90.DEM", 
                                         "E020N90.DEM", "E060N90.DEM", "E100N90.DEM", "E140N90.DEM", "W180N40.DEM",
                                         "W140N40.DEM", "W100N40.DEM", "W060N40.DEM", "W020N40.DEM", "E020N40.DEM", 
                                         "E060N40.DEM", "E100N40.DEM", "E140N40.DEM", "W180S10.DEM", "W140S10.DEM",
                                         "W100S10.DEM", "W060S10.DEM", "W020S10.DEM", "E020S10.DEM", "E060S10.DEM",
                                         "E100S10.DEM", "E140S10.DEM"}; 
        
        private const double GCONST = 50.0/6000.0;
        private const int TerHeaderLen = 12;
        private const int TerLineSize = 4800; // 4800 shorts
        private const int TerLineSizeInBytes = TerLineSize * 2; // 4800 shorts

        private enum SectorRegions {OneSector, TwoSectorsVertial, TwoSectorsHorizontal, FourSectors};

        public SoaringPilotTerrainGenerator() {
            // 
            // TODO: Add constructor logic here
            //
            FindDEMSector(49, 9);
        }

        #region Public Functions
        
        /// <summary>
        // Calculates the approximate size of the file that would be created
        // for the region bounded by the inputted lat/long values.
        // Output is in kilobytes
        /// <\summary>
        public int CalcOutputSize(double topLat, double leftLon, double bottomLat, double rightLon) {
            int numRows, numCols, outputSize;

            numRows = (int)Math.Abs(((topLat - bottomLat) / GCONST)) + 1;
            numCols = (int)Math.Abs(((leftLon - rightLon) / GCONST)) + 1;
            outputSize = numRows * numCols * 16 / 8 / 1024;
            //sprintf(tempmsg, "Output file will be approximately %ikb.", outputsize);
            //DisplayMessage(tempmsg, "Info");
	
            return outputSize;
        }

        /*struct TerHeader {
            float  ullat; // 4 bytes
            float  ullon; // 4 bytes
            short    numrows; // 2 bytes
            short    numcols; // 2 bytes
        } */

        public void BuildSPTerrainDB(string fileName, double topLat, double leftLon, double bottomLat, double rightLon) {
            SectorRegions splitType;
            int  ulSect, lrSect, urSect, llSect;
            double terTopLat = 0, terLeftLon = 0, terBottomLat = 0, terRightLon = 0;
            ushort numRows, numCols, numRecords;
            int numRowsFromUpper, numRowsFromLower;
            int numColsFromLeft, numColsFromRight;
            short startRow = 0, startCol = 0;
            int i, x, j;
            int infLen;
            int rowOffset;
            
            string [] demFileNames = new string[4]; // Input file names //
            BinaryReader [] readers = {null, null, null, null};
            PDBFile writer = null;
            byte [] terline = new byte[TerLineSizeInBytes];//, terline2, outline;
            byte [] terline2 = new byte[TerLineSizeInBytes];//, terline2, outline;
            short [] outLine = new short[TerLineSize];
            string dbName = "SoaringPilot_terrain_db";   // Database name on Palm //
            /*
            sprintf(tempmsg, "ullat-|%f| ullon-|%f|", ullat, ullon);
            DisplayMessage(tempmsg, tempmsg);
            sprintf(tempmsg, "lrlat-|%f| lrlon-|%f|", lrlat, lrlon);
            DisplayMessage(tempmsg, tempmsg);
            */

            try {
                if (fileName == null || fileName.Length == 0) {
                    fileName = dbName + ".pdb";
                }

                if (bottomLat >= topLat) {
                    throw new Exception("Latitude out-of-bounds\nInput lower lat greater than or equal to upper lat");
                }

                if (rightLon <= leftLon) {
                    throw new Exception("Longitude out-of-bounds\nInput lower lon less than or equal to upper lon");
                }

                ulSect = FindDEMSector(topLat, leftLon);
                lrSect = FindDEMSector(bottomLat, rightLon);
                urSect = FindDEMSector(topLat, rightLon);
                llSect = FindDEMSector(bottomLat, leftLon);

                demFileNames[0] = DEMNames[ulSect];
                readers[0] = new BinaryReader(new FileStream(inputPath + "\\" + demFileNames[0], FileMode.Open, FileAccess.Read));
                switch(lrSect - ulSect) {
                case 0:
                    splitType = SectorRegions.OneSector;
                    break;
                case 1:
                    splitType = SectorRegions.TwoSectorsHorizontal;
                    demFileNames[1] = DEMNames[lrSect];
                    readers[1] = new BinaryReader(new FileStream(inputPath + "\\" + demFileNames[1], FileMode.Open, FileAccess.Read));
                    break;
                case 9:
                    splitType = SectorRegions.TwoSectorsVertial;
                    demFileNames[1] = DEMNames[lrSect];
                    readers[1] = new BinaryReader(new FileStream(inputPath + "\\" + demFileNames[1], FileMode.Open, FileAccess.Read));
                    break;
                case 10:
                    splitType = SectorRegions.FourSectors;
                    demFileNames[1] = DEMNames[urSect];
                    demFileNames[2] = DEMNames[llSect];
                    demFileNames[3] = DEMNames[lrSect];
                    readers[1] = new BinaryReader(new FileStream(inputPath + "\\" + demFileNames[1], FileMode.Open, FileAccess.Read));
                    readers[2] = new BinaryReader(new FileStream(inputPath + "\\" + demFileNames[2], FileMode.Open, FileAccess.Read));
                    readers[3] = new BinaryReader(new FileStream(inputPath + "\\" + demFileNames[3], FileMode.Open, FileAccess.Read));
                    break;
                default:
                    throw new Exception("Requested area is too large!");
                    break;
                }

                // Determine the upper left lat/long of the GTOPO3 file being read
                GetLatLonFormFileName(demFileNames[0], out terTopLat, out terLeftLon);
                // Determine the lower right lat/lon of the GTOPO3 file being read
	
                switch(splitType) {
                case SectorRegions.OneSector:
                    terBottomLat = terTopLat - 50.0;
                    terRightLon = terLeftLon + 40.0;
                    break;
                case SectorRegions.TwoSectorsVertial:
                    terBottomLat = terTopLat - 100.0;
                    terRightLon = terLeftLon + 40.0;
                    break;
                case SectorRegions.TwoSectorsHorizontal:
                    terBottomLat = terTopLat - 50.0;
                    terRightLon = terLeftLon + 80.0;
                    break;
                case SectorRegions.FourSectors:
                    terBottomLat = terTopLat - 100.0;
                    terRightLon = terLeftLon + 80.0;
                    break;
                }
                //sprintf(tempmsg, "terullat-|%f| terullon-|%f|", terullat, terullon);
                //	DisplayMessage(tempmsg, tempmsg);
                //sprintf(tempmsg, "terlrlat-|%f| terlrlon-|%f|", terlrlat, terlrlon);
                //	DisplayMessage(tempmsg, tempmsg);

                startRow = (short)Math.Abs(((terTopLat - topLat) / GCONST));
                startCol = (short)Math.Abs(((terLeftLon - leftLon) / GCONST));
                //sprintf(tempmsg, "startrow %i", startrow);
                //	DisplayMessage(tempmsg, tempmsg);
                //sprintf(tempmsg, "startcol %i", startcol);
                //	DisplayMessage(tempmsg, tempmsg);

                // Create PDB file header. //
                //time(&today);

                //outtext(dbname, kMaxPDBNameSize); // name //
                //outshort(pdbflags); 	      // flags //
                //outshort(10);		      // version //
                //outlong(today + timeOffset);      // creationTime //
                //outlong(today + timeOffset);      // modificationTime //
                //outlong(0); 		      // backupTime //
                //outlong(0); 		      // modificationNumber //
                //outlong(0); 		      // appInfoOffset //
                //outlong(0); 		      // sortInfoOffset //
                //outbytes(dtype, 4); 	      // type //
                //outbytes(creator, 4);	      // Creator //
                //outlong(0); 		      // uniqueID //
                //outlong(0); 		      // nextRecordID //

                // Have to add one to numrows so that the terrain data is inclusive
                // of the requested terrain area
                numRows = (ushort)(Math.Abs(((terTopLat - bottomLat) / GCONST)) - startRow + 1);
                // numrecords must be one more than rows to account for the first
                // records being the index/header records
                numRecords = (ushort)(numRows + 1);
                //sprintf(tempmsg, "numrows %i", numrows);
                //	DisplayMessage(tempmsg, tempmsg);
                //sprintf(tempmsg, "numrecords %i", numrecords);
                //	DisplayMessage(tempmsg, tempmsg);
                
                //outshort(numrecords);   // numRecords in the actual .pdb file //
                writer = new PDBFile(new FileStream(fileName, FileMode.Create, FileAccess.Write));
                writer.header.fileName = dbName;
                writer.header.version = 10;
                writer.header.type = "STer";
                writer.header.creator = "Soar";
                writer.header.flags = 0; // Database flags //
                writer.header.uniqueID = 0;
                writer.header.numRecords = numRecords;

                // start writing
                writer.WriteHeader();
                
                // Have to add one to numcols so that the terrain data is inclusive
                // of the requested terrain area
                numCols = (ushort)(Math.Abs(((terLeftLon - rightLon) / GCONST)) - startCol + 1);
                //sprintf(tempmsg, "numcols %i", numcols);
                //	DisplayMessage(tempmsg, tempmsg);

                // Create PDB record entry. //
                infLen = numCols * 2;
                //	sprintf(tempmsg, "inflen %i", inflen);
                //	DisplayMessage(tempmsg, tempmsg);
                rowOffset = PDBFile.PDBHeaderSize + (numRecords * PDBFile.PDBRecordEntrySize) + 2;

                for (i = 0; i < numRecords; i++) {
                    writer.Write(rowOffset); // offset 
                    writer.Write(i); // attr + uniqueID:24 
                    //outlong(roffset);		 
                    //outlong(i); 		 
                    if (i == 0) {
                        rowOffset += TerHeaderLen;
                    } else {
                        rowOffset += infLen;
                    }
                }

                writer.Write((byte) 0);
                writer.Write((byte) 0);
                //outbyte(0);
                //outbyte(0);

                // Output the first record with the upper left lat/long
                // and the number of rows and columns in the file.
                writer.Write((float)topLat);
                writer.Write((float)leftLon);
                writer.Write(numRows);
                writer.Write(numCols);
                /*terhead.ullat = ullat;
                ByteSwapFloat(&terhead.ullat);

                terhead.ullon = ullon;
                ByteSwapFloat(&terhead.ullon);

                terhead.numrows = numrows;
                Swap2Byte(&terhead.numrows);

                terhead.numcols = numcols;
                Swap2Byte(&terhead.numcols);

                outbytes((char *)&terhead, sizeof(TerHeader));
                */
                readers[0].BaseStream.Seek(startRow * terline.Length, SeekOrigin.Current);
                //fseek(fi, startrow*sizeof(TerLine), SEEK_CUR);
                if (splitType == SectorRegions.FourSectors || splitType == SectorRegions.TwoSectorsHorizontal) {
                    readers[1].BaseStream.Seek(startRow * terline.Length, SeekOrigin.Current);
                    //fseek(fi2, startrow*sizeof(TerLine), SEEK_CUR);
                }

                if (splitType == SectorRegions.FourSectors || splitType == SectorRegions.TwoSectorsVertial) {
                    numRowsFromUpper = (short)(Math.Abs(((terTopLat - terBottomLat) / 2.0 / GCONST)) - startRow);
                    numRowsFromLower = (short)(Math.Abs(((terTopLat - 50.0 - bottomLat) / GCONST)));
                } else {
                    numRowsFromUpper = numRows;
                    numRowsFromLower = 0;
                }
                //sprintf(tempmsg, "numrowsfmupper-|%i| numrowsfmlower-|%i|", numrowsfmupper, numrowsfmlower);
                //	DisplayMessage(tempmsg, tempmsg);

                if (splitType == SectorRegions.FourSectors || splitType == SectorRegions.TwoSectorsHorizontal) {
                    numColsFromLeft = (short)(Math.Acos(((terLeftLon - terRightLon) / 2.0 / GCONST)) - startCol);
                    numColsFromRight = (short)(Math.Abs(((rightLon - (terLeftLon + 40.0)) / GCONST)));
                } else {
                    numColsFromLeft = numCols;
                    numColsFromRight = 0;
                }
                //sprintf(tempmsg, "numcolsfmleft-|%i| numcolsfmright-|%i|", numcolsfmleft, numcolsfmright);
                //	DisplayMessage(tempmsg, tempmsg);
	
                // From now on we're creating the content of the database. //
                for (i = 0; i < numRows; i++) {
                    if (numRowsFromUpper > 0) {
                        if (splitType == SectorRegions.OneSector || splitType == SectorRegions.TwoSectorsVertial) {
                            // This reads the whole line from the GTOPO file
                            readers[0].Read(terline, 0, terline.Length);
                            //fread(&terline, sizeof(TerLine), 1, fi);
                        } 
                        else {
                            readers[0].Read(terline, 0, terline.Length);
                            readers[1].Read(terline2, 0, terline2.Length);
                            //fread(&terline, sizeof(TerLine), 1, fi);
                            //fread(&terline2, sizeof(TerLine), 1, fi2);
                        }
                        numRowsFromUpper--;
                    } 
                    else if (numRowsFromLower > 0) {
                        if (splitType == SectorRegions.TwoSectorsVertial) {
                            // This is the TWOSECTV Case
                            readers[1].Read(terline, 0, terline.Length);
                            //fread(&terline, sizeof(TerLine), 1, fi2);
                        } else {
                            // This is the FOURSECT Case
                            readers[2].Read(terline, 0, terline.Length);
                            readers[3].Read(terline2, 0, terline2.Length);
                            //fread(&terline, sizeof(TerLine), 1, fi3);
                            //fread(&terline2, sizeof(TerLine), 1, fi4);
                        }
                    }

                    j = numColsFromLeft;
                    for (x = 0; x < numCols; x++) {
                        if (j > 0) {
                            outLine[x] = (short)(terline[(startCol * 2) + (x * 2)] << 8 | 
                                                 terline[(startCol * 2) + (x * 2) + 1]);
                            j--;
                        } 
                        else if (numColsFromRight > 0) {
                            outLine[x] = (short)(terline2[(numColsFromRight * 2) + (x * 2) - (numCols * 2)] << 8 | 
                                                 terline2[(numColsFromRight * 2) + (x * 2) - (numCols * 2) + 1]);
                            //outline.terints[x] = terline2.terints[numcolsfmright+x-numcols];
                        }
                    }
                    writer.Write(outLine, numCols);
                }
            }
            finally {
                foreach (BinaryReader r in readers) {
                    if (r != null) {
                        r.Close();
                    }
                }
                if (writer != null) {
                    writer.Close();
                }
            }
        }

        #endregion

        #region Private Functions
        int FindDEMSector(double lat, double lon) {
            int i;
            double topLat, leftLon, bottomLat, rightLon;
            int sector = -1;
            
            for (i = 0; i < DEMNames.Length; i++) {
                // Determine the upper left lat/long of the GTOPO3 file being read
                GetLatLonFormFileName(DEMNames[i], out topLat, out leftLon);
                // Determine the lower right lat/lon of the GTOPO3 file being read
                bottomLat = topLat - 50.0;
                rightLon = leftLon + 40.0;
                if (topLat >= lat && bottomLat <= lat && leftLon <= lon && rightLon >= lon) {
                    sector = i;
                    break;
                    //sprintf(tempmsg, "Its Inside-|%s| retsector=|%i|", demnames[i], retsector);
                    // Comment this one out for final
                    //DisplayMessage(tempmsg, tempmsg);
                }
            }
            return sector;
        }

        void GetLatLonFormFileName(string fName, out double lat, out double lon) {
            Regex coord = new Regex("([W|E])([0-9]{3})([N|S])([0-9]{2})\\.DEM");
            Match m;
            GroupCollection coll;
            m = coord.Match(fName);
            if (m.Success) {
                coll = m.Groups;
                lat = double.Parse(coll[4].Value);
                if (coll[3].Value == "S") {
                    lat = -lat;
                }
                lon = double.Parse(coll[2].Value);
                if (coll[1].Value == "W") {
                    lon = -lon;
                }
            }
            else {
                throw new Exception(string.Format("Invalid filename format for DEM file {0}", fName));
            }
        }

        #endregion

        #region Attributes
        public string DEMFilePath {
            get {
                return inputPath;
            }
            set {
                inputPath = value;
            }
        }
        #endregion
    }
}
