// name   : Task.cs
// author : Harald Maier
// date   : 17.10.2003
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.Collections;
using System.Drawing;

using SoaringDotNET.Data;
using SoaringDotNET.Optimization;

namespace SoaringDotNET.GUI
{
    public enum TaskTypes {Unknown, FreeOut, FreeOutAndReturn, Triangle, TriangleStartOnLeg};
    /// <summary>
	/// Summary description for Task.
	/// </summary>
	public class Task : SoaringDotNET.GUI.MapLineElement
	{
        private string name;
        private static Font font = new Font("Arial Black", 8.0f, FontStyle.Regular);
        private OptimizeTypes optimizationType;
        private TaskTypes taskType;
        private ArrayList sectors;
        private bool isErasing;
        private bool isArea;

        public Task(string name) : 
            this() {
            this.name = name;
        }

        public Task(Task t) :
            this(t.Name) {
            foreach (WayPoint wp in t) {
                this.Add(wp);
            }
        }

        public Task() :
            base(MapElementTypes.Task) {
			//
			// TODO: Add constructor logic here
			//
            taskType = TaskTypes.Unknown;
            optimizationType = OptimizeTypes.Unknown;

            pen = new Pen(AppContents.taskColor.DrawColor);
            sectors = new ArrayList();
            isErasing = false;
            isArea = false;
		}

        #region Public Functions
        public void Add(WayPoint t) {
            this.Insert(Count, t);
        }

        public void Swap(int first, int second) {
            object tmp;
            if ((first >= 0 && first < this.Count) && (second >= 0 && second < this.Count)) {
                tmp = points[first];
                points[first] = points[second];
                points[second] = tmp;

                tmp = sectors[first];
                sectors[first] = sectors[second];
                sectors[second] = tmp;

                SetSectorTypes();
                SetTaskType();
            }
        }
/*
        public void Insert(int pos, WayPoint t) {
            Insert(pos, t);
        }
*/
        public void Insert(int pos, WayPoint t/*, bool recalc*/) {
            points.Insert(pos, t);
            if (pos == 1) {
                sectors.Insert(pos, new SectorDefinition((SectorDefinition)AppContents.defaultSectors[TurnpointTypes.Start]));
            }
            else if (sectors.Count > 1 && pos == sectors.Count - 1) {
                sectors.Insert(pos, new SectorDefinition((SectorDefinition)AppContents.defaultSectors[TurnpointTypes.Finish]));
                SetSector(sectors.Count - 3, new SectorDefinition((SectorDefinition)AppContents.defaultSectors[TurnpointTypes.Turnpoint]));
            }
            else {
                sectors.Insert(pos, new SectorDefinition((SectorDefinition)AppContents.defaultSectors[TurnpointTypes.Turnpoint]));
                if (sectors.Count >= 4) {
                    SetSector(sectors.Count - 2, new SectorDefinition((SectorDefinition)AppContents.defaultSectors[TurnpointTypes.Finish]));
                    if (sectors.Count > 4) {
                        SetSector(sectors.Count - 3, new SectorDefinition((SectorDefinition)AppContents.defaultSectors[TurnpointTypes.Turnpoint]));
                    }
                }
            }

//            if (recalc) {
                CalcArea();
  //          }
            SetSectorTypes();
            SetTaskType();
        }

        public void Delete(int pos) {
            points.RemoveAt(pos);
            sectors.RemoveAt(pos);
            SetSectorTypes();
            SetTaskType();
        }

        public void Clear() {
            points.Clear();
            sectors.Clear();
            SetTaskType();
        }

        public override void Draw(Graphics g, Pen pen, Brush brush, WgsArea view, int width, int height) {
            // TODO:  Add Task.Draw implementation
            Pen oldPen = this.pen;
            this.pen = pen;
            isErasing = true;
            this.brush = brush;
            this.Draw(g, view, width, height);
            isErasing = false;
            this.pen = oldPen;
        }

        public override void Draw(Graphics g, WgsArea view, int width, int height) {
            long lat, lon, lat2, lon2, x2, y2, x3, y3;
            long latRange, lonRange;
            long minLat, minLon;
            long diff, r, r2;
            int i, sweepAngle;
            double alpha;
            double w1, w2;
            WayPoint wp;
            WayPoint wp1, wp2;

            minLat = view.TopLat;
            latRange = view.Height;
            minLon = view.LeftLon;
            lonRange = view.Width;

            mapPos.Clear();            
            if (!isErasing) {
                this.pen.Color = IsFAI ? AppContents.taskColorFAI.DrawColor : AppContents.taskColor.DrawColor;
                this.brush = Brushes.Black;
            }

            if (points.Count > 0) {
                wp = this[0];
                lat = (wp.Latitude - minLat) * height / latRange;
                diff = wp.Longitude - minLon;
                if (diff <= -64800000) {
                    diff += 129600000;
                }
                lon = diff * width / lonRange;

                mapPos[0] = new Point((int)lon, (int)lat);
                
                if (view.PointInArea(wp)) {
                    g.DrawEllipse(this.pen, lon - 5, lat - 5, 11, 11);
                    g.DrawString(wp.longName, font, brush, lon + 10, lat);
                }

                if (points.Count > 1) {
                    for (i = 1; i < points.Count; i++) {
                        wp = this[i];
                        lat2 = (wp.Latitude - minLat) * height / latRange;
                        diff = wp.Longitude - minLon;
                        if (diff <= -64800000) {
                            diff += 129600000;
                        }
                        lon2 = diff * width / lonRange;

                        g.DrawLine(this.pen, lon, lat, lon2, lat2);
                        mapPos[i] = new Point((int)lon2, (int)lat2);

                        if (view.PointInArea(wp)) {
                            if (i == points.Count - 1) {
                                g.DrawEllipse(this.pen, lon2 - 5, lat2 - 5, 11, 11);
                            }
                            else {
                                SectorDefinition sector = (SectorDefinition)sectors[i];

                                wp1 = wp2 = null;

                                if (i == 1) {
                                    wp1 = this[i + 1];
                                }
                                else if (i == Count - 2) {
                                    wp1 = this[i - 1];
                                }
                                else {
                                    wp1 = this[i - 1];
                                    wp2 = this[i + 1];
                                }

                                if (wp2 == null) {
                                    wp2 = wp1;
                                }

                                // draw sector
                                if (sector.directionFrom == 360) {
                                    w1 = wp.bearingDeg(wp1);
                                    w2 = wp.bearingDeg(wp2);
                                    if (w2 < w1) {
                                        alpha = (w1 - ((w1 - w2) / 2.0) + 180);// % 360;
                                        if ((w1 - w2) < 180.0) {
                                            alpha -= 180;
                                        }
                                    }
                                    else {
                                        alpha = (w1 + ((w2 - w1) / 2.0));// % 360;
                                        if ((w2 - w1) > 180.0) {
                                            alpha += 180;
                                        }
                                    }
                                }
                                else {
                                    alpha = sector.directionFrom;
                                }

                                switch (sector.sectorType) {
                                case SectorTypes.Line:
                                    alpha += 180.0;
                                    r = (long)(height * sector.radius1 / ConvertionFactors.NAUTICMILES2METERS / 60.0 * 360000.0 / latRange);
                                    x2 = (long)(Math.Cos(Math.PI * alpha / 180.0) * r);
                                    y2 = (long)(Math.Sin(Math.PI * alpha / 180.0) * r);
                                    g.DrawLine(pen, lon2 + x2, lat2 + y2, lon2 - x2, lat2 - y2);
                                    break;
                                case SectorTypes.Sector:
                                    alpha = (alpha + 45.0) % 360.0;
                                    if (sector.radius1 > 0) {
                                        r = (long)(height * sector.radius1 / ConvertionFactors.NAUTICMILES2METERS / 60.0 * 360000.0 / latRange);
                                        if (r > 0) {
                                            g.DrawPie(pen, lon2 - r, lat2 - r, 2 * r, 2 * r, (float)alpha + 90, -90);
                                        }
                                    }
                                    if (sector.radius2 > 0) {
                                        r = (long)(height * sector.radius2 / ConvertionFactors.NAUTICMILES2METERS / 60.0 * 360000.0 / latRange);
                                        if (r > 0) {
                                            g.DrawEllipse(pen, lon2 - r, lat2 - r, 2 * r, 2 * r);
                                        }
                                    }
                                    break;
                                case SectorTypes.Area:
                                    if (sector.directionFrom == sector.directionTo) {
                                        // draw circle for both radius
                                        if (sector.radius1 > 0) {
                                            r = (long)(height * sector.radius1 / ConvertionFactors.NAUTICMILES2METERS / 60.0 * 360000.0 / latRange);
                                            g.DrawEllipse(pen, lon2 - r, lat2 - r, 2 * r, 2 * r);
                                        }
                                        if (sector.radius2 > 0) {
                                            r = (long)(height * sector.radius2 / ConvertionFactors.NAUTICMILES2METERS / 60.0 * 360000.0 / latRange);
                                            g.DrawEllipse(pen, lon2 - r, lat2 - r, 2 * r, 2 * r);                                        }
                                    }
                                    else {
                                        // calculate center and offset for upper left corner of arc rectangle
                                        if (sector.directionFrom < sector.directionTo) {
                                            sweepAngle = sector.directionTo - sector.directionFrom;
                                        }
                                        else {
                                            sweepAngle = 360 - sector.directionFrom + sector.directionTo;
                                        }
                                        r = (long)(height * sector.radius1 / ConvertionFactors.NAUTICMILES2METERS / 60.0 * 360000.0 / latRange);
                                        if (r > 0) {
                                            g.DrawArc(pen, lon2 - r, lat2 - r, 2 * r, 2 * r, sector.directionFrom - 90, sweepAngle);
                                        }

                                        r = (long)(height * sector.radius2 / ConvertionFactors.NAUTICMILES2METERS / 60.0 * 360000.0 / latRange);
                                        if (r > 0) {
                                            g.DrawArc(pen, lon2 - r, lat2 - r, 2 * r, 2 * r, sector.directionFrom - 90, sweepAngle);
                                        }

                                        // draw segments
                                        r = (long)(height * sector.radius1 / ConvertionFactors.NAUTICMILES2METERS / 60.0 * 360000.0 / latRange);
                                        r2 = (long)(height * sector.radius2 / ConvertionFactors.NAUTICMILES2METERS / 60.0 * 360000.0 / latRange);

                                        x2 = (long)(Math.Cos(Math.PI * (sector.directionFrom + 90) / 180.0) * r);
                                        y2 = (long)(Math.Sin(Math.PI * (sector.directionFrom + 90) / 180.0) * r);
                                        x3 = (long)(Math.Cos(Math.PI * (sector.directionFrom + 90) / 180.0) * r2);
                                        y3 = (long)(Math.Sin(Math.PI * (sector.directionFrom + 90) / 180.0) * r2);
                                        g.DrawLine(pen, lon2 - x3, lat2 - y3, lon2 - x2, lat2 - y2);
                                        
                                        x2 = (long)(Math.Cos(Math.PI * (sector.directionTo + 90) / 180.0) * r);
                                        y2 = (long)(Math.Sin(Math.PI * (sector.directionTo + 90) / 180.0) * r);
                                        x3 = (long)(Math.Cos(Math.PI * (sector.directionTo + 90) / 180.0) * r2);
                                        y3 = (long)(Math.Sin(Math.PI * (sector.directionTo + 90) / 180.0) * r2);
                                        g.DrawLine(pen, lon2 - x3, lat2 - y3, lon2 - x2, lat2 - y2);
                                    }
                                    break;
                                }
                            }
                            g.DrawString(wp.longName, font, brush, lon2 + 10, lat2);
                        }

                        lat = lat2;
                        lon = lon2;
                    }
                }
            }

            if (taskType == TaskTypes.TriangleStartOnLeg) {
                wp = this[2];
                lat = (wp.Latitude - minLat) * height / latRange;
                diff = wp.Longitude - minLon;
                if (diff <= -64800000) {
                    diff += 129600000;
                }
                lon = diff * width / lonRange;

                wp = this[4];
                lat2 = (wp.Latitude - minLat) * height / latRange;
                diff = wp.Longitude - minLon;
                if (diff <= -64800000) {
                    diff += 129600000;
                }
                lon2 = diff * width / lonRange;
                g.DrawLine(this.pen, lon, lat, lon2, lat2);
            }
        }

        public new void CalcArea() {
            base.CalcArea();
        }

        public override string ToString() {
            // TODO:  Add Task.ToString implementation
            return name;
        }

        public static bool isFAI(double length, double s1, double s2, double s3) {
            bool yes;
            double minL, maxL;
            if (length < 500) {
                minL = length * 0.28;
                yes = s1 >= minL && s2 >= minL && s3 >= minL;
            }
            else {
                minL = length * 0.25;
                maxL = length * 0.45;
                yes = (s1 >= minL && s1 <= maxL) && (s2 >= minL && s2 <= maxL) && (s3 >= minL && s3 <= maxL);
            }
            return yes;
        }

        public void SetSectorTypes() {
            int i;
            SectorDefinition sector;
            isArea = false;
            for (i = 0; i < sectors.Count; i++) {
                sector = (SectorDefinition)sectors[i];
                if (i == 1) {
                    sector.turnpointType = TurnpointTypes.Start;
                    //sector.sectorType = ((SectorDefinition)AppContents.defaultSectors[TurnpointTypes.Start]).sectorType;
                }
                else if (i == sectors.Count - 2) {
                    ((SectorDefinition )sectors[i]).turnpointType = TurnpointTypes.Finish;
                    //sector.sectorType = ((SectorDefinition)AppContents.defaultSectors[TurnpointTypes.Finish]).sectorType;
                }
                else {
                    ((SectorDefinition)sectors[i]).turnpointType = TurnpointTypes.Turnpoint;
                    //sector.sectorType = ((SectorDefinition)AppContents.defaultSectors[TurnpointTypes.Turnpoint]).sectorType;
                }

                if (sector.sectorType == SectorTypes.Area) {
                    isArea = true;
                }
            }
        }

        #endregion

        #region Private Functions
        private void SetTaskType() {
            WayPoint p1, p2;
            taskType = TaskTypes.Unknown;
            switch (this.Count) {
            case 4:
                taskType = TaskTypes.FreeOut;
                break;
            case 5:
                p1 = this[1];
                p2 = this[3];
                if (p1 == p2 || p1.distanceKM(p2) < 1.0) {
                    taskType = TaskTypes.FreeOutAndReturn;
                }
                break;
            case 6:
                p1 = this[1];
                p2 = this[4];
                if ((p1 == p2 || p1.distanceKM(p2) < 1.0) && this[2] != p1 && this[3] != p1 && this[2] != p2 && this[3] != p2) {
                    taskType = TaskTypes.Triangle;
                }
                break;
            case 7:
                p1 = this[1];
                p2 = this[5];
                if (OptimizedFor != OptimizeTypes.DMSTFreeFlight && (p1 == p2 || p1.distanceKM(p2) <= 1.0)  && this[2] != p1 && this[3] != p1 && this[4] != p1 && this[2] != p2 && this[3] != p2 && this[4] != p2 && this[2] != this[3] && this[2] != this[4] && this[3] != this[4] ) {
                    taskType = TaskTypes.TriangleStartOnLeg;
                }
                break;
            default:
                taskType = TaskTypes.Unknown;
                break;
            }
        }
        #endregion

        #region Attributes
        public string Name {
            get {
                return name;
            }
            set {
                name = value;
            }
        }

        public new WayPoint this[int pos] {
            get {
                return (WayPoint)points[pos];
            }
            set {
                if (pos >= 0 && pos < this.Count) {
                    points[pos] = value;
                    SetSectorTypes();
                    SetTaskType();
                }
            }
        }

        public SectorDefinition GetSector(int pos) {
            return (SectorDefinition) (pos >= 0 && pos < sectors.Count ? sectors[pos] : null);
        }

        public void SetSector(int pos, SectorDefinition s) {
            if (pos >= 0 && pos < sectors.Count) {
                sectors[pos] = s;
                SetSectorTypes();
            }
            else {
                throw new Exception("Internal error: sector out of range");
            }
        }

        public double TotalDistanceKM {
            get {
                double lenght = 0.0;
                int i;
                for (i = 1; i < this.Count; i++) {
                    lenght += this[i - 1].distanceKM(this[i]);
                }
                return lenght;
            }
        }

        public double TotalDistanceNM {
            get {
                return TotalDistanceKM / ConvertionFactors.NAUTICMILES2METERS;
            }
        }

        public double TaskDistanceKM {
            get {
                double lenght = 0.0;
                int i;
                if (taskType == TaskTypes.TriangleStartOnLeg) {
                    lenght += this[2].distanceKM(this[3]);
                    lenght += this[3].distanceKM(this[4]);
                    lenght += this[4].distanceKM(this[2]);
                }
                else {
                    for(i = 2; i < points.Count - 1; i++) {
                        lenght += this[i - 1].distanceKM(this[i]);
                    }
                }
                return lenght;
            }
        }

        public double TaskDistanceNM {
            get {
                return TaskDistanceKM / ConvertionFactors.NAUTICMILES2METERS;
            }        
        }

        public TaskTypes TaskType {
            get {
                return taskType;
            }
        }

        public string TaskTypeString {
            get {
                string t = "Unknown";
                switch (taskType) {
                case TaskTypes.Unknown:
                    t = "Free flight";
                    break;
                case TaskTypes.FreeOut:
                    t = "Free goal";
                    break;
                case TaskTypes.FreeOutAndReturn:
                    t = "Free goal and return distance";
                    break;
                case TaskTypes.Triangle:
                    t = "Triangle";
                    if (IsFAI) {
                        t = "FAI " + t;
                    }
                    break;
                case TaskTypes.TriangleStartOnLeg:
                    t = "Triangle (start on leg)";
                    if (IsFAI) {
                        t = "FAI " + t;
                    }
                    break;
                }
                return t;
            }
        }

        public bool IsFAI {
            get {
                bool yes = false;
                if (!IsArea) {
                    if (taskType == TaskTypes.Triangle) {
                        yes = Task.isFAI(TaskDistanceKM, this[1].distanceKM(this[2]), this[2].distanceKM(this[3]), this[3].distanceKM(this[4]));
                    }
                    else if(taskType == TaskTypes.TriangleStartOnLeg) {
                        yes = Task.isFAI(TaskDistanceKM, this[2].distanceKM(this[3]), this[3].distanceKM(this[4]), this[4].distanceKM(this[2]));
                    }
                }
                return yes;
            }
        }

        public bool IsArea {
            get {
                return isArea;
            }
        }
            public OptimizeTypes OptimizedFor {
            get {
                return optimizationType;
            }
            set {
                optimizationType = value;
            }
        }
        #endregion        
    }
}
