// name   : Volkslogger.cs
// author : Harald Maier
// date   : 25.11.2003
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.Threading;
using System.Text;
using System.IO;
using System.Collections;
using System.Windows.Forms;

using SoaringDotNET.Data;
using SoaringDotNET.GUI;

namespace SoaringDotNET.Recorders
{
	/// <summary>
	/// 
	/// </summary>
    public class Volkslogger : SoaringDotNET.Recorders.SerialConnectionRecorder {
        // wp types
        public enum VolksloggerFlags {Landable = 1, HardSurface = 2, Airport = 4, CheckPoint = 8};
        // sector types
        public enum VLSectorTypes {SECTOR = 0, LINE = 1};
        // flow- and format- control
        private enum VLFlow : byte {STX = 0x02, ETX = 0x03, ENQ = 0x05, ACK = 0x06, DLE = 0x10, NAK = 0x15, CAN = 0x18}
        // Kommandos PC -> Logger
        private enum VLCmd : byte {
            cmd_INF = 0x00,// Information
            cmd_DIR = 0x01,// Verzeichnis lesen
            cmd_GFL = 0x02,// Flug lesen mit MD4
            cmd_GFS = 0x03,// Flug lesen mit Signatur
            cmd_RDB = 0x04,// Datenbank lesen
            cmd_WPR = 0x05,// Parameter schreiben
            cmd_CFL = 0x06,// Flugspeicher lschen
            cmd_PDB = 0x07,// Datenbank schreiben
            cmd_SIG = 0x08,// Signatur berechnen und ausgeben
            cmd_ERO = 0x09,// Emergency readout (Memorydump lesen)
            cmd_RST = 0x0c // Restart Logger
        }

        /* DS-Haupttypen */
        private enum VLMainRecordType : byte {
            rectyp_msk = 0xE0, //Haupttypmaske
            rectyp_vrt = 0x00, //Variabel, timed
            rectyp_vrb = 0x20, //Variabel, untimed
            rectyp_sep = 0x40, //separator (head)
            rectyp_end = 0x60, //Security
            rectyp_pos = 0x80, //Pos-DS (Fix)
            rectyp_tnd = 0xA0, //Time&Date
            rectyp_fil = 0xC0, //Fllzeichen
            rectyp_poc = 0xE0 //komprimierter Pos-DS
        }

        /* Untertypen des Haupttyps Variabel */
        internal enum VLSubRecordType : byte {
            FLDPLT = 0x01,
            FLDPLT1 = 0x01,
            FLDPLT2 = 0x02,
            FLDPLT3 = 0x03,
            FLDPLT4 = 0x04,
            FLDGTY = 0x05,
            FLDGID = 0x06,
            FLDCID = 0x07,
            FLDCCL = 0x08,
            FLDTZN = 0x09,
            FLDNTP = 0x10,
            FLDFDT = 0x11,
            FLDTID = 0x12,
            FLDTKF = 0x20,
            FLDSTA = 0x21,
            FLDFIN = 0x22,
            FLDLDG = 0x23, // Landing gibts nicht
            FLDTP1 = 0x31,
            FLDTP2 = 0x32,
            FLDTP3 = 0x33,
            FLDTP4 = 0x34,
            FLDTP5 = 0x35,
            FLDTP6 = 0x36,
            FLDTP7 = 0x37,
            FLDTP8 = 0x38,
            FLDTP9 = 0x39,
            FLDTP10 = 0x3A,
            FLDTP11 = 0x3B,
            FLDTP12 = 0x3C,
            FLDHDR = 0x50,
            FLDEPEV = 0x60,
            FLDETKF = 0x61
        }

        // hchste, von diesem Programm verdaute Binrdateiversion
        // bfw = "binary file version"
        private const int maxBfv = 1;

        // Gre der Fix-Datenstze in den verschiedenen Binrdateiversionen
        private int [,]pos_ds_size = new int[maxBfv + 1, 2]{{11, 0}, {12, 9}};

        private ushort []Crc16Table = {
        0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50a5, 0x60c6, 0x70e7,
        0x8108, 0x9129, 0xa14a, 0xb16b, 0xc18c, 0xd1ad, 0xe1ce, 0xf1ef,
        0x1231, 0x0210, 0x3273, 0x2252, 0x52b5, 0x4294, 0x72f7, 0x62d6,
        0x9339, 0x8318, 0xb37b, 0xa35a, 0xd3bd, 0xc39c, 0xf3ff, 0xe3de,
        0x2462, 0x3443, 0x0420, 0x1401, 0x64e6, 0x74c7, 0x44a4, 0x5485,
        0xa56a, 0xb54b, 0x8528, 0x9509, 0xe5ee, 0xf5cf, 0xc5ac, 0xd58d,
        0x3653, 0x2672, 0x1611, 0x0630, 0x76d7, 0x66f6, 0x5695, 0x46b4,
        0xb75b, 0xa77a, 0x9719, 0x8738, 0xf7df, 0xe7fe, 0xd79d, 0xc7bc,
        0x48c4, 0x58e5, 0x6886, 0x78a7, 0x0840, 0x1861, 0x2802, 0x3823,
        0xc9cc, 0xd9ed, 0xe98e, 0xf9af, 0x8948, 0x9969, 0xa90a, 0xb92b,
        0x5af5, 0x4ad4, 0x7ab7, 0x6a96, 0x1a71, 0x0a50, 0x3a33, 0x2a12,
        0xdbfd, 0xcbdc, 0xfbbf, 0xeb9e, 0x9b79, 0x8b58, 0xbb3b, 0xab1a,
        0x6ca6, 0x7c87, 0x4ce4, 0x5cc5, 0x2c22, 0x3c03, 0x0c60, 0x1c41,
        0xedae, 0xfd8f, 0xcdec, 0xddcd, 0xad2a, 0xbd0b, 0x8d68, 0x9d49,
        0x7e97, 0x6eb6, 0x5ed5, 0x4ef4, 0x3e13, 0x2e32, 0x1e51, 0x0e70,
        0xff9f, 0xefbe, 0xdfdd, 0xcffc, 0xbf1b, 0xaf3a, 0x9f59, 0x8f78,
        0x9188, 0x81a9, 0xb1ca, 0xa1eb, 0xd10c, 0xc12d, 0xf14e, 0xe16f,
        0x1080, 0x00a1, 0x30c2, 0x20e3, 0x5004, 0x4025, 0x7046, 0x6067,
        0x83b9, 0x9398, 0xa3fb, 0xb3da, 0xc33d, 0xd31c, 0xe37f, 0xf35e,
        0x02b1, 0x1290, 0x22f3, 0x32d2, 0x4235, 0x5214, 0x6277, 0x7256,
        0xb5ea, 0xa5cb, 0x95a8, 0x8589, 0xf56e, 0xe54f, 0xd52c, 0xc50d,
        0x34e2, 0x24c3, 0x14a0, 0x0481, 0x7466, 0x6447, 0x5424, 0x4405,
        0xa7db, 0xb7fa, 0x8799, 0x97b8, 0xe75f, 0xf77e, 0xc71d, 0xd73c,
        0x26d3, 0x36f2, 0x0691, 0x16b0, 0x6657, 0x7676, 0x4615, 0x5634,
        0xd94c, 0xc96d, 0xf90e, 0xe92f, 0x99c8, 0x89e9, 0xb98a, 0xa9ab,
        0x5844, 0x4865, 0x7806, 0x6827, 0x18c0, 0x08e1, 0x3882, 0x28a3,
        0xcb7d, 0xdb5c, 0xeb3f, 0xfb1e, 0x8bf9, 0x9bd8, 0xabbb, 0xbb9a,
        0x4a75, 0x5a54, 0x6a37, 0x7a16, 0x0af1, 0x1ad0, 0x2ab3, 0x3a92,
        0xfd2e, 0xed0f, 0xdd6c, 0xcd4d, 0xbdaa, 0xad8b, 0x9de8, 0x8dc9,
        0x7c26, 0x6c07, 0x5c64, 0x4c45, 0x3ca2, 0x2c83, 0x1ce0, 0x0cc1,
        0xef1f, 0xff3e, 0xcf5d, 0xdf7c, 0xaf9b, 0xbfba, 0x8fd9, 0x9ff8,
        0x6e17, 0x7e36, 0x4e55, 0x5e74, 0x2e93, 0x3eb2, 0x0ed1, 0x1ef0};

        private char []base64tab = "0123456789@ABCDEFGHIJKLMNOPQRSTUVWXYZ`abcdefghijklmnopqrstuvwxyz".ToCharArray();

        // sizes of VL memory regions
        internal const int VLAPI_DBB_MEMSIZE = 16384;
        private const int VLAPI_LOG_MEMSIZE = 81920;
        private const int VL_CONV_VERSION = 424;

        private int dataBaudIdx;
        private SerialPorts.LineSpeed commandBaud;
        private SerialPorts.LineSpeed dataBaud;
        private VolksloggerDatabase DB;

        protected string port;

        public Volkslogger()
		{
			// 
			// TODO: Add constructor logic here
			//
            options = RecorderOptions.DownloadFlights |
                RecorderOptions.DownloadTasks |
                RecorderOptions.DownloadWaypoints |
                RecorderOptions.UploadTask |
                RecorderOptions.UploadWaypoints |
                RecorderOptions.UploadDeclaration |
                RecorderOptions.DownloadPilots |
                RecorderOptions.UploadPilots;

            capacity.maxNrWaypoints = 500;
            capacity.maxNrTasks = 25;
            capacity.maxNrWaypointsPerTask = 10;
            capacity.maxNrPilots = 25;

            name = "Volkslogger";
            ManufacurerCode = "GCS";
            ManufacurerLetter = "A";
            supportedLineSpeeds = new object[] {9600, 19200, 38400, 57600, 115200};
            preferedLineSpeed = 115200;
            haveDatabase = false;
        }

        #region Public Functions
        public override bool Open(string port, SerialPorts.LineSpeed speed) {
            // TODO:  Add Volkslogger.Open implementation
            bool ok;
            int i;
            byte [] c = new byte[1];
            long t;
            
            this.port = port;
            commandBaud = SerialPorts.LineSpeed.Baud_9600;
            ok = base.Open(port, commandBaud);
            SetDatabaudIdx(speed);
            
            comPort.Flush();
            // eventuell noch laufende Aktion im Logger abbrechen
            for (i = 0; i < 10; i++) {
                comPort.SendI((byte)VLFlow.CAN);
                Thread.Sleep(1);
            }

            t = DateTime.Now.Ticks;
            do { // Solange R's aussenden, bis ein L zurckkommt
                comPort.SendI((byte)'R');
                Thread.Sleep(30);
                if (DateTime.Now.Ticks - t > 50000000) {
                    throw new Exception("No response from recorder within 5 secs.!");
                }
            } while (comPort.Recv(c, 1) < 1 || c[0] != 'L');

            i = 1;
            t = DateTime.Now.Ticks;
            // Auf 4 hintereinanderfolgende L's warten
            while (i < 4) {
                if (comPort.Recv(c, 1) > 0 && c[0] == 'L') {
                    i++;
                }
                else if (DateTime.Now.Ticks - t > 50000000) {
                    throw new Exception("No response from recorder within 5 secs.!");
                }
            }
            return ok;
        }
        #endregion

        #region Protected Functions
        protected override void DownloadTasks() {
            // TODO:  Add Volkslogger.DownloadTasks implementation
            bool ok = true;
            try {
                if (!haveDatabase) {
                    ok = ReadDatabase();
                }
                if (ok) {
                    foreach (Task t in DB.tasks) {
                        tasks.Add(new Task(t));
                    }
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                ok = false;
            }
            finally {
                if (RecorderBase.DownloadTasksThreadNotify != null) {
                    RecorderBase.DownloadTasksThreadNotify(ok);
                }
            }
        }
    
        protected override void DownloadWaypoints() {
            // TODO:  Add Volkslogger.DownloadWaypoints implementation
            bool ok = true;
            try {
                if (!haveDatabase) {
                    ok = ReadDatabase();
                }
                if (ok) {
                    currentCatalogue = new WaypointCatalog("Volkslogger" + SerialNo);
                    foreach (WayPoint wp in DB.waypoints) {
                        currentCatalogue.Add(new WayPoint(wp));
                    }
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                ok = false;
            }
            finally {
                if (RecorderBase.DownloadWaypointsThreadNotify != null) {
                    RecorderBase.DownloadWaypointsThreadNotify(ok);
                }
            }
        }

        protected override void DownloadPilots() {
            // TODO:  Add Volkslogger.UploadSUAs implementation
            bool ok = true;
            try {
                if (!haveDatabase) {
                    ok = ReadDatabase();
                }
                if (ok) {
                    foreach (Pilot p in DB.pilots) {
                        pilots.Add(new Pilot(p));
                    }
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                ok = false;
            }
            finally {
                if (RecorderBase.DownloadPilotsThreadNotify != null) {
                    RecorderBase.DownloadPilotsThreadNotify(ok);
                }
            }
        }

        protected override void GetSerialNo() {
            // TODO:  Add Volkslogger.GetSerialNo implementation
            byte [] buffer = new byte[VLAPI_DBB_MEMSIZE];
            bool ok = false;
            IsInterrupted = false;
            if (SendCommand((byte)VLCmd.cmd_INF, 0, 0) == 0) {
                if (ok = ReadBlock(buffer) > 0) {
                    // Aufbau der Versions- und sonstigen Nummern
                    //vlinfo.sessionid = 256*buffer[0] + buffer[1];
                    serialNo = (uint)(256 * buffer[2]) + buffer[3];
                    //vlinfo.fwmajor = buffer[4] / 16;
                    //vlinfo.fwminor = buffer[4] % 16;
                    //vlinfo.fwbuild = buffer[7];
                }
            }
            if (RecorderBase.SerialNoThreadNotify != null) {
                RecorderBase.SerialNoThreadNotify(ok);
            }
        }
    
        protected override void UploadTasks() {
            // TODO:  Add Volkslogger.UploadTasks implementation
            bool ok = true;
            try {
                if (!haveDatabase) {
                    ok = ReadDatabase();
                }
                if (ok) {
                    DB.tasks.Clear();
                    DB.tasks.AddRange(tasks);
                    ok = WriteDatabase();
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                ok = false;
            }
            finally {
                if (RecorderBase.UploadTasksThreadNotify!= null) {
                    RecorderBase.UploadTasksThreadNotify(ok);
                }
            }
        }
    
        protected override void UploadWaypoints() {
            // TODO:  Add Volkslogger.UploadWaypoints implementation
            bool ok = true;
            try {
                if (!haveDatabase) {
                    ok = ReadDatabase();
                }
                if (ok) {
                    DB.waypoints.Clear();
                    foreach (WayPoint wp in currentCatalogue.VisibleWaypoints) {
                        DB.waypoints.Add(wp);
                    }
                    ok = WriteDatabase();
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                ok = false;
            }
            finally {
                if (RecorderBase.UploadWaypointsThreadNotify != null) {
                    RecorderBase.UploadWaypointsThreadNotify(ok);
                }
            }
        }

        protected override void UploadDeclaration() {
            // TODO:  Add Volkslogger.UploadDeclaration implementation
            bool ok = true;
            try {
                if (!haveDatabase) {
                    ok = ReadDatabase();
                }
                if (ok) {
                    DB.declarationTask = declarationTask;
                    ok = WriteDatabase();
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                ok = false;
            }
            finally {
                if (RecorderBase.UploadDeclarationThreadNotify != null) {
                    RecorderBase.UploadDeclarationThreadNotify(ok);
                }
            }
        }    

        protected override void GetFlightDir() {
            // TODO:  Add Volkslogger.GetFlightDir implementation
            byte [] buffer = new byte[VLAPI_LOG_MEMSIZE];
            bool ok = false;
            IsInterrupted = false;
            try {
                if (SendCommand((byte)VLCmd.cmd_DIR, 0, 0) == 0) {                    
                    if (ok = (ReadBlock(buffer) > 0)) {
                        ConvertToFlightDirectory(buffer);
                    }
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                ok = false;
            }
            finally {
                if (RecorderBase.FlightDirectoryThreadNotify != null) {
                    RecorderBase.FlightDirectoryThreadNotify(ok);
                }
            }
        }

        protected override void DownloadFlights() {
            // TODO:  Add Volkslogger.DownloadFlights implementation
            byte [] buffer = new byte[VLAPI_LOG_MEMSIZE];
            bool ok = false;
            IsInterrupted = false;
            int dataSize;
            DirectoryEntry de;

            try {
                foreach (int i in flightIndexes) {
                    dataSize = 0;
                    de = (DirectoryEntry) flightDirectory[i];
                    ok = false;
                    if (SendCommand(secureDownload == true ? (byte)VLCmd.cmd_GFS : (byte)VLCmd.cmd_GFL, (byte)i, (byte)dataBaudIdx) == 0) {
                        SetComSpeed(dataBaud);
                        if ((dataSize = ReadBlock(buffer)) > 0) {
                            SetComSpeed(commandBaud);
                            if (SendCommand((byte)VLCmd.cmd_SIG, 0, 0) == 0) {
                                if ((dataSize = ReadBlock(buffer, dataSize)) > 0) {
                                    de.igcFile = ConvertToIGC(buffer);
                                    ok = true;
                                }
                                else {
                                    break;
                                }
                            }
                            else {
                                break;
                            }
                        }
                        else {
                            break;
                        }
                    }
                    else {
                        break;
                    }
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                ok = false;
            }
            finally {
                if (RecorderBase.DownloadFlightsThreadNotify != null) {
                    RecorderBase.DownloadFlightsThreadNotify(ok);
                }
            }
        }

        protected override void UploadSUAs() {
            // TODO:  Add Volkslogger.UploadSUAs implementation
            if (RecorderBase.UploadSUAsThreadNotify != null) {
                RecorderBase.UploadSUAsThreadNotify(true);
            }
        }

        protected override void UploadPilots() {
            // TODO:  Add Volkslogger.UploadSUAs implementation
            bool ok = true;
            try {
                if (!haveDatabase) {
                    ok = ReadDatabase();
                }
                if (ok) {
                    DB.pilots.Clear();
                    DB.pilots.AddRange(pilots);
                    ok = WriteDatabase();
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                ok = false;
            }
            finally {
                if (RecorderBase.UploadPilotsThreadNotify != null) {
                    RecorderBase.UploadPilotsThreadNotify(ok);
                }
            }
        }


        #endregion

        #region Private Functions
        private void SetDatabaudIdx(SerialPorts.LineSpeed baud) {
            dataBaud = baud;
            switch (dataBaud) {
            default:
            case SerialPorts.LineSpeed.Baud_9600:
                dataBaudIdx = 1;
                break;
            case SerialPorts.LineSpeed.Baud_19200:
                dataBaudIdx = 2;
                break;
            case SerialPorts.LineSpeed.Baud_38400:
                dataBaudIdx = 3;
                break;
            case SerialPorts.LineSpeed.Baud_57600:
                dataBaudIdx = 4;
                break;
            case SerialPorts.LineSpeed.Baud_115200:
                dataBaudIdx = 5;
                break;
            }
        }

        private byte SendCommand(byte cmd, byte param1, byte param2) {
            int i;
            byte []cmdarray = new byte[8];
            byte []b = new byte[1];
            ushort crc16 = 0;
            long t;
            // alte Zeichen verwerfen
            comPort.Flush();
            // Kommandointerpreter im VL zurcksetzen
            for (i = 0; i < 6; i++) {
                comPort.SendI((byte)VLFlow.CAN);
                Thread.Sleep(2);
            }
            // Kommandopaket aufbauen
            cmdarray[0] = cmd;
            cmdarray[1] = param1;
            cmdarray[2] = param2;
            // Kommando verschicken ( ENQ,Daten,CRC )
            comPort.SendI((byte)VLFlow.ENQ);
            Thread.Sleep(2);
            for (i = 0; i < cmdarray.Length; i++) {
                crc16 = UpdateCRC(cmdarray[i], crc16);
                comPort.SendI(cmdarray[i]);
                Thread.Sleep(2);
            }

            comPort.SendI((byte)(crc16 / 256));
            Thread.Sleep(2);
            comPort.SendI((byte)(crc16 % 256));
            Thread.Sleep(2);

            // Kommandobesttigung abwarten, aber hchstens timeout Sekunden
            t = DateTime.Now.Ticks;
            while (comPort.Recv(b, 1) < 1) {
                if (DateTime.Now.Ticks - t > 40000000) {
                    b[0] = 255;
                    break;
                }
            }

            return b[0];
            // Rckgabewert:     -1 : timeout
            //		 0..255 : Besttigungscode vom Logger
        }

        private ushort UpdateCRC(byte Octet, ushort CRC) {
            return (ushort)(0xffff & ( (CRC << 8) ^ Crc16Table[ (CRC >> 8) ^ Octet ] ));
        }

        private int  HdopToFxa(byte hdop) {
            return (int)((double)hdop * 100.01 / 3);
        }

        private int WriteBlock(byte [] buffer) {
            byte [] c = new byte[1];
            int i = 0;
            ushort crc16 = 0;

            while (comPort.Recv(c, 1) < 1) {
                if (IsInterrupted) {
                    break;
                }
            }

            if (c[0] == (byte)VLFlow.ACK) {
                comPort.Flush();
                for (i = 0; i < buffer.Length; i++) {
                    if (IsInterrupted) {
                        comPort.SendI((byte)VLFlow.CAN);
                        comPort.SendI((byte)VLFlow.CAN);
                        comPort.SendI((byte)VLFlow.CAN);
                        break;
                    }

                    comPort.Send(buffer[i]);
                    crc16 = UpdateCRC(buffer[i], crc16);
                }

                comPort.Send((byte)(crc16 / 256));
                comPort.Send((byte)(crc16 % 256));

            }

            while (comPort.Recv(c, 1) < 1) {
                if (IsInterrupted) {
                    i = 0;
                    break;
                }
            }

            if (c[0] != (byte)VLFlow.ACK) {
                i = 0;
            }
            comPort.Flush();
            return i;
        }

        private int ReadBlock(byte [] buffer) {
            return ReadBlock(buffer, 0);
        }

        private int ReadBlock(byte [] buffer, int startPos) {
            byte [] c = new byte[1];
            bool dle_r = false;
            ushort crc16 = 0;
            bool start = false;
            bool ende = false;
            int i;
            int idx;

            idx = startPos;
            for (i = idx; i < buffer.Length; i++) {
                buffer[i] = 0xff;
            }

            comPort.Flush();

            while (!ende) {
                // Zeichen anfordern und darauf warten
                comPort.SendI((byte)VLFlow.ACK);
                while (comPort.Recv(c, 1) < 1) {
                    if (IsInterrupted) {
                        break;
                    }
                    //Thread.Sleep(1);
                }

                if (IsInterrupted) {
                    comPort.SendI((byte)VLFlow.CAN);
                    comPort.SendI((byte)VLFlow.CAN);
                    comPort.SendI((byte)VLFlow.CAN);

                    break;
                }

                switch (c[0]) {
                case (byte)VLFlow.DLE:
                    if (!dle_r) {             //!DLE, DLE -> Achtung!
                        dle_r = true;
                    }
                    else { 	                 // DLE, DLE -> DLE-Zeichen
                        dle_r = false;
                        if (start && idx < buffer.Length) {
                            buffer[idx++] = c[0];
                            crc16 = UpdateCRC(c[0], crc16);
                        }
                    }
                    break;
                case (byte)VLFlow.ETX:
                    if (!dle_r) {             //!DLE, ETX -> Zeichen
                        if (start && idx < buffer.Length) {
                            buffer[idx++] = c[0];
                            crc16 = UpdateCRC(c[0], crc16);
                        }
                    }
                    else if (start) {
                        ende = true;                   // DLE, ETX -> Blockende
                        dle_r = false;
                    }
                    break;
                case (byte)VLFlow.STX:
                    if (!dle_r) {	         //!DLE, STX -> Zeichen
                        if (start && idx < buffer.Length) {
                            buffer[idx++] = c[0];
                            crc16 = UpdateCRC(c[0], crc16);
                        }
                    }
                    else {
                        start = true;           // DLE, STX -> Blockstart
                        dle_r = false;
                        crc16 = 0;
                    }
                    break;
                default: 
                    if (start) {
                        if (idx < buffer.Length) {
                            buffer[idx++] = c[0];
                        }
                        crc16 = UpdateCRC(c[0], crc16);
                    }
                    break;
                }
            }

            if (crc16 != 0) {
                throw new Exception("CRC Error at position " + idx.ToString());
            }
            else if (idx > 2) {              //CRC am Ende abschneiden
                idx -= 2;
                if (idx < buffer.Length) {
                    buffer[idx] = 0xff;
                }
                
                if (idx < buffer.Length - 1) {
                    buffer[idx + 1] = 0xff;
                }
            }
            else {
                idx = 0;
            }
            return idx;
        }

        private void ConvertToFlightDirectory(byte []buffer) {
            byte mainType, subType;
            int idx, dsLength, idx2;
            int binaryFileVersion = 0;
            bool end;
            char [] trimChars = {'\0', ' '};
            int flightOfDay, oldDay, oldMonth, oldYear;
            string pilot1 = "", pilot2 = "", pilot3 = "", pilot4 = "";
            DateTime tmpTime = DateTime.Now;
            int i;
            int h, m, s;
            DirectoryEntry de = null;

            flightDirectory.Clear();
            idx = 0;
            idx2 = 0;
            dsLength = 0;
            end = false;
            flightOfDay = 1;
            oldDay = oldMonth = oldYear = 0;

            while (!end) {
                mainType = (byte)(buffer[idx] & (byte)VLMainRecordType.rectyp_msk);
                switch (mainType) {
                case (byte)VLMainRecordType.rectyp_sep:
                    // new entry
                    de = new DirectoryEntry();
                    pilot1 = "";
                    pilot2 = "";
                    pilot3 = "";
                    pilot4 = "";
                    binaryFileVersion = buffer[idx] & ~(byte)VLMainRecordType.rectyp_msk;
                    if (binaryFileVersion > maxBfv) {
                        end = true;
                    }
                    dsLength = 1;
                    break;
                case (byte)VLMainRecordType.rectyp_vrt:
                case (byte)VLMainRecordType.rectyp_vrb:
                    dsLength = buffer[idx + 1];
                    if (mainType == (byte)VLMainRecordType.rectyp_vrb) {
                        idx2 = idx + 2;
                    }
                    else {
                        idx2 = idx + 3;
                    }
                    subType = buffer[idx2];
                    switch (subType) {
                    case (byte)VLSubRecordType.FLDCID:  // competition ID
                        de.competitionId = Encoding.ASCII.GetString(buffer, idx2 + 1, 3).Trim(trimChars);
                        break;
                    case (byte)VLSubRecordType.FLDGID :  // glider ID
                        de.gliderId = Encoding.ASCII.GetString(buffer, idx2 + 1, 7).Trim(trimChars);
                        break;
                    case (byte)VLSubRecordType.FLDPLT1 :  // Pilot
                        pilot1 = Encoding.ASCII.GetString(buffer, idx2 + 1, 16).Trim(trimChars);
                        break;
                    case (byte)VLSubRecordType.FLDPLT2 :  // Pilot
                        pilot2 = Encoding.ASCII.GetString(buffer, idx2 + 1, 16).Trim(trimChars);
                        break;
                    case (byte)VLSubRecordType.FLDPLT3 :  // Pilot
                        pilot3 = Encoding.ASCII.GetString(buffer, idx2 + 1, 16).Trim(trimChars);
                        break;
                    case (byte)VLSubRecordType.FLDPLT4 :  // Pilot
                        pilot4 = Encoding.ASCII.GetString(buffer, idx2 + 1, 16).Trim(trimChars);
                        break;
                    case (byte)VLSubRecordType.FLDHDR :  // Seriennummer einlesen
                    case (byte)VLSubRecordType.FLDETKF : // Takeoff-Flag setzen
                        // ignore these fields
                        break;
                    }
                    break;
                case (byte)VLMainRecordType.rectyp_fil:
                    dsLength = 1;
                    break;
                case (byte)VLMainRecordType.rectyp_pos:
                    dsLength = pos_ds_size[binaryFileVersion, 0];
                    break;
                case (byte)VLMainRecordType.rectyp_poc:
                    if((buffer[idx + 2] & 0x80) != 0) { // Endebedingung
                        end = true;
                    }
                    dsLength = pos_ds_size[binaryFileVersion, 1];
                    break;
                case (byte)VLMainRecordType.rectyp_tnd:
                    dsLength = 8;
                    i = (65536 * buffer[idx + 2]) + (256 * buffer[idx + 3]) + buffer[idx + 4];
                    h = i / 3600;
                    m = (i - (h * 3600)) / 60;
                    s = i - (h * 3600) - (m * 60);
                    tmpTime = new DateTime(
                        (10 * (buffer[idx + 5] >> 4)) + (buffer[idx + 5] & 0x0f) + 2000,
                        (10 * (buffer[idx + 6] >> 4)) + (buffer[idx + 6] & 0x0f), 
                        (10 * (buffer[idx + 7] >> 4)) + (buffer[idx + 7] & 0x0f), h, m, s);
                    break;
                case (byte)VLMainRecordType.rectyp_end:
                    de.pilot = pilot1 + pilot2 + pilot3 + pilot4;
                    i = (65536 * buffer[idx + 4]) + (256 * buffer[idx + 5]) + buffer[idx + 6];
                    tmpTime -= TimeSpan.FromSeconds(i);
                    if (oldDay == tmpTime.Day && oldMonth == tmpTime.Month && oldYear == tmpTime.Year) {
                        flightOfDay++;
                    }
                    else {
                        flightOfDay = 1;
                        oldDay = tmpTime.Day;
                        oldMonth = tmpTime.Month;
                        oldYear = tmpTime.Year;
                    }

                    de.firstFix = tmpTime;

                    i = (65536 * buffer[idx + 1]) + (256 * buffer[idx + 2]) + buffer[idx + 3];
                    de.lastFix = de.firstFix + TimeSpan.FromSeconds(i);
                    de.fileName = string.Format("{0}{1}{2}{3}{4}{5}.IGC", 
                        c36[de.firstFix.Year % 10],
                        c36[de.firstFix.Month],
                        c36[de.firstFix.Day],
                        ManufacurerLetter, SerialNo, flightOfDay < 36 ? c36[flightOfDay] : '_');
                    flightDirectory.Add(de);
                    dsLength = 7;
                    break;
                default:
                    // hmm something went wrong
                    flightDirectory.Clear();
                    end = true;
                    break;
                }

                idx += dsLength;
            }
        }

        private IGCFile ConvertToIGC(byte []buffer) {
            IGCFile igcFile = new IGCFile();
            byte mainType, subType;
            bool end = false;
            bool tzSet = false;
            bool validFix = true;
            char [] trimChars = {'\0'};
            int idx = 0, dsLength = 0, idx2;
            int signaturePos = 0;
            int binaryFileVersion = 0;
            string pilot1 = "", pilot2 = "", pilot3 = "", pilot4 = "";
            long timeRelative = 0;
            long declarationTime = 0;
            DateTime firstFix = DateTime.Now;
            DateTime realTime = DateTime.Now;
            long lat = 0, lon = 0, tmpLon = 0;
            long diffLat, diffLon;
            double flightTimeZone = 0;
            int timeZone = 9999; //check to see if set
            int i, pos;
            int h, m, s;
            int pressure = 0;
            int gpsAlt = 0;
            int fxa = 0;
            int enl = 0;
            int pAlt = 0;
            string tmpS;
            ArrayList CRecordTmp = new ArrayList(12);
            string declTime = "";
            int taskId = 0;
            int nrTurnpoints = 0;
            WayPoint wp;
            int latDeg, lonDeg;
            double latMin, lonMin;

            // nr of turnpoints + 4 (start, takeoff, finish, landing)
            for (i = 0; i < 16; i++) {
                CRecordTmp.Add(new WayPoint());
            }

            while (!end) {
                mainType = (byte)(buffer[idx] & (byte)VLMainRecordType.rectyp_msk);
                switch (mainType) {
                case (byte)VLMainRecordType.rectyp_sep:
                    // new entry
                    timeRelative = 0;
                    binaryFileVersion = buffer[idx] & ~(byte)VLMainRecordType.rectyp_msk;
                    if (binaryFileVersion > maxBfv) {
                        // unsupported binary file version
                        end = true;
                    }
                    dsLength = 1;
                    break;
                case (byte)VLMainRecordType.rectyp_end:                  
                    dsLength = 41;
                    pilot1 = pilot1 + pilot2 + pilot3 + pilot4;
                    igcFile.PLT = string.Format("H{0}PLTPILOT:{1}", pilot1 != "" ? "F" : "O", pilot1);

                    end = true;
                    break;
                case (byte)VLMainRecordType.rectyp_fil:
                    dsLength = 1;
                    break;
                case (byte)VLMainRecordType.rectyp_tnd:
                    // errechnet rckwrts die Zeit des 1. Fixes
                    timeRelative += buffer[idx + 1];
                    i = (65536 * buffer[idx + 2]) + (256 * buffer[idx + 3]) + buffer[idx + 4];
                    h = i / 3600;
                    m = (i % 3600) / 60;
                    s = i % 60;
                    firstFix = new DateTime(
                        (10 * (buffer[idx + 5] >> 4)) + (buffer[idx + 5] & 0x0f) + 2000,
                        (10 * (buffer[idx + 6] >> 4)) + (buffer[idx + 6] & 0x0f), 
                        (10 * (buffer[idx + 7] >> 4)) + (buffer[idx + 7] & 0x0f), h, m, s);

                    firstFix -= TimeSpan.FromSeconds(timeRelative);

                    dsLength = 8;
                    break;
                case (byte)VLMainRecordType.rectyp_pos:
                case (byte)VLMainRecordType.rectyp_poc:
                    if((buffer[idx + 2] & 0x80) != 0) { // Endebedingung
                        end = true;
                    }
                    else {
                        timeRelative += buffer[idx + 2];
                        validFix = ((buffer[idx] & 0x10) >> 4) != 0 ? true : false;
                        if (mainType == (byte)VLMainRecordType.rectyp_pos) {
                            dsLength = pos_ds_size[binaryFileVersion, 0];
                            lon = (buffer[idx + 6] << 16) | (buffer[idx + 7] << 8) | buffer[idx + 8];
                            if ((buffer[idx + 9] & 0x80) != 0) {
                                lon = -lon;
                            }
                        }
                        else {
                            dsLength = pos_ds_size[binaryFileVersion, 1];
                            tmpLon = ((buffer[idx + 3] & 0x78) << 5) | buffer[idx + 5];
                            if ((buffer[idx + 6] & 0x80) != 0) {
                                tmpLon = -tmpLon;
                            }
                            lon += tmpLon;
                        }

                        // ftz mit Lngengrad fllen
			            // der erste gltige ist der letzte,
			            // der in ftz gespeichert wird
			            if (!tzSet) {
			                flightTimeZone = (double)lon;
			                if (validFix)
			                    tzSet = true;
			            }
                    }
                    break;
                case (byte)VLMainRecordType.rectyp_vrt:
                case (byte)VLMainRecordType.rectyp_vrb:
                    dsLength = buffer[idx + 1];
                    if (mainType == (byte)VLMainRecordType.rectyp_vrb) {
                        idx2 = idx + 2;
                    }
                    else {
                        timeRelative += buffer[idx + 2];
                        idx2 = idx + 3;
                    }
                    subType = buffer[idx2];
                    switch (subType) {
                    case (byte)VLSubRecordType.FLDNTP:
                        // we ignore number of turnpoints and read it on demand
                        nrTurnpoints = buffer[idx2 + 1];
                        declarationTime = timeRelative;
                        break;
                    case (byte)VLSubRecordType.FLDTID:
                        taskId = (buffer[idx2 + 1] * 256) + buffer[idx2 +2];
                        declarationTime = timeRelative;
                        break;
                    case (byte)VLSubRecordType.FLDFDT:
                        // we ignore this because conv version is >= 422
                        break;
                    case (byte)VLSubRecordType.FLDTZN:  // Zeitzonenoffset einlesen
                        if (buffer[idx2 + 1] < 128) {
                            timeZone = 15 * buffer[idx2 + 1];
                        }
                        else {
                            timeZone = (buffer[idx2 + 1] - 256) * 15;
                        }
                        break;
                    case (byte)VLSubRecordType.FLDTKF:
                        Unpack(buffer, idx2 + 1, (WayPoint)CRecordTmp[0]);
                        break;
                    case (byte)VLSubRecordType.FLDSTA:
                        Unpack(buffer, idx2 + 1, (WayPoint)CRecordTmp[1]);
                        break;
                    case (byte)VLSubRecordType.FLDFIN:
                        Unpack(buffer, idx2 + 1, (WayPoint)CRecordTmp[CRecordTmp.Count - 2]);
                        break;
                    case (byte)VLSubRecordType.FLDLDG:
                        Unpack(buffer, idx2 + 1, (WayPoint)CRecordTmp[CRecordTmp.Count - 1]);
                        break;
                    case (byte)VLSubRecordType.FLDTP1:
                        Unpack(buffer, idx2 + 1, (WayPoint)CRecordTmp[2]);
                        break;
                    case (byte)VLSubRecordType.FLDTP2:
                        Unpack(buffer, idx2 + 1, (WayPoint)CRecordTmp[3]);
                        break;
                    case (byte)VLSubRecordType.FLDTP3:
                        Unpack(buffer, idx2 + 1, (WayPoint)CRecordTmp[4]);
                        break;
                    case (byte)VLSubRecordType.FLDTP4:
                        Unpack(buffer, idx2 + 1, (WayPoint)CRecordTmp[5]);
                        break;
                    case (byte)VLSubRecordType.FLDTP5:
                        Unpack(buffer, idx2 + 1, (WayPoint)CRecordTmp[6]);
                        break;
                    case (byte)VLSubRecordType.FLDTP6:
                        Unpack(buffer, idx2 + 1, (WayPoint)CRecordTmp[7]);
                        break;
                    case (byte)VLSubRecordType.FLDTP7:
                        Unpack(buffer, idx2 + 1, (WayPoint)CRecordTmp[8]);
                        break;
                    case (byte)VLSubRecordType.FLDTP8:
                        Unpack(buffer, idx2 + 1, (WayPoint)CRecordTmp[9]);
                        break;
                    case (byte)VLSubRecordType.FLDTP9:
                        Unpack(buffer, idx2 + 1, (WayPoint)CRecordTmp[10]);
                        break;
                    case (byte)VLSubRecordType.FLDTP10:
                        Unpack(buffer, idx2 + 1, (WayPoint)CRecordTmp[11]);
                        break;
                    case (byte)VLSubRecordType.FLDTP11:
                        Unpack(buffer, idx2 + 1, (WayPoint)CRecordTmp[12]);
                        break;
                    case (byte)VLSubRecordType.FLDTP12:
                        Unpack(buffer, idx2 + 1, (WayPoint)CRecordTmp[13]);
                        break;
                    case (byte)VLSubRecordType.FLDPLT1:  // Pilotenname einlesen
                        pilot1 = Encoding.ASCII.GetString(buffer, idx2 + 1, 64);
                        if ((pos = pilot1.IndexOfAny(trimChars)) != -1) {
                            pilot1 = pilot1.Substring(0, pos);
                        }
                        pilot1.Trim();
                        break;
                    case (byte)VLSubRecordType.FLDPLT2:  // Pilotenname einlesen
                        pilot2 = Encoding.ASCII.GetString(buffer, idx2 + 1, 64);
                        if ((pos = pilot2.IndexOfAny(trimChars)) != -1) {
                            pilot2 = pilot2.Substring(0, pos);
                        }
                        pilot2.Trim();
                        break;
                    case (byte)VLSubRecordType.FLDPLT3:  // Pilotenname einlesen
                        pilot3 = Encoding.ASCII.GetString(buffer, idx2 + 1, 64);
                        if ((pos = pilot3.IndexOfAny(trimChars)) != -1) {
                            pilot3 = pilot3.Substring(0, pos);
                        }
                        pilot3.Trim();
                        break;
                    case (byte)VLSubRecordType.FLDPLT4:  // Pilotenname einlesen
                        pilot4 = Encoding.ASCII.GetString(buffer, idx2 + 1, 64);
                        if ((pos = pilot4.IndexOfAny(trimChars)) != -1) {
                            pilot4 = pilot4.Substring(0, pos);
                        }
                        pilot4.Trim();
                        break;
                    case (byte)VLSubRecordType.FLDGTY:  // Flugzeugtyp einlesen
                        tmpS = Encoding.ASCII.GetString(buffer, idx2 + 1, 50);
                        if ((pos = tmpS.IndexOfAny(trimChars)) != -1) {
                            tmpS = tmpS.Substring(0, pos);
                        }
                        igcFile.GTY = string.Format("H{0}GTYGLIDERTYPE:{1}", tmpS != "" ? "F" : "O", tmpS.Trim());
                        break;
                    case (byte)VLSubRecordType.FLDGID: // Flugzeugkennzeichen einlesen
                        tmpS = Encoding.ASCII.GetString(buffer, idx2 + 1, 50);
                        if ((pos = tmpS.IndexOfAny(trimChars)) != -1) {
                            tmpS = tmpS.Substring(0, pos);
                        }
                        tmpS.Trim();
                        igcFile.GID = string.Format("H{0}GIDGLIDERID:{1}", tmpS != "" ? "F" : "O", tmpS);
                        break;
                    case (byte)VLSubRecordType.FLDCCL:  // Wettbewerbsklasse einlesen
                        tmpS = Encoding.ASCII.GetString(buffer, idx2 + 1, 50);
                        if ((pos = tmpS.IndexOfAny(trimChars)) != -1) {
                            tmpS = tmpS.Substring(0, pos);
                        }
                        igcFile.CCL = string.Format("HFCCLCOMPETITIONCLASS:{0}", tmpS.Trim());
                        break;
                    case (byte)VLSubRecordType.FLDCID:  // Wettbewerbskennzeichen einlesen
                        tmpS = Encoding.ASCII.GetString(buffer, idx2 + 1, 3);
                        if ((pos = tmpS.IndexOfAny(trimChars)) != -1) {
                            tmpS = tmpS.Substring(0, pos);
                        }
                        igcFile.CID = string.Format("HFCIDCOMPETITIONID:{0}", tmpS.Trim());
                        break;
                    case (byte)VLSubRecordType.FLDHDR:  // Seriennummer und anderes einlesen
                        // Public-Key erst mal lschen
                        // 19.10.99 weggemacht, weil schon in main vorhanden
                        //dsa_y_b[0] = 2; dsa_y_b[1] = 0;
                        //memset(&dsa_y_b[2],0,(sizeof dsa_y_b)-2);

                        //	*serno = (256L*p2[1]+p2[2]);

                        // sonstiges einlesen
                        //strcpy(igcheader.A,wordtoserno(*serno));
                        
                        igcFile.A = "A" + ManufacurerCode + SerialNo;
                        igcFile.DTM = string.Format("HFDTM{0:000}GPSDATUM:WGS84", (uint)buffer[idx2 + 3]);
                        //sprintf(igcheader.DTM,"%03u",p2[3]);
                        igcFile.RHW = string.Format("HFRHWHARDWAREVERSION:{0:X}.{1:X}", buffer[idx2 + 4] >> 4, buffer[idx2 + 4] & 0xf);
                        //sprintf(igcheader.RHW,"%0X.%0X",p2[4]>>4,(p2[4]&0xf));
                        tmpS = string.Format("{0:X}.{1:X}", buffer[idx2 + 5] >> 4, buffer[idx2 + 5] & 0xf);
                        if (tmpS.CompareTo("3.3") < 0) {
                            igcFile.I = "I013638FXA";
                        }
                        else {
                            igcFile.I = "I023638FXA3941ENL";
                        }
                        igcFile.RFW = string.Format("HFRFWFIRMWAREVERSION:{0}", tmpS);
                        //sprintf(igcheader.RFW,"%0X.%0X",p2[5]>>4,(p2[5]&0xf));
                        igcFile.FXA = string.Format("HFFXA{0:000}", (uint)buffer[idx2 + 7]);
                        //sprintf(igcheader.FXA,"%03u",p2[7]);
                        igcFile.FTY = "HFFTYFR TYPE:GARRECHT INGENIEURGESELLSCHAFT,VOLKSLOGGER 1.0";

                        igcFile.CONV = "LCONV-VER:4.24";

                        break;
                    }
                    break;
                default:
                    dsLength = 1;
                    end = true;
                    break;
                }
                idx += dsLength;
            }

            signaturePos = idx;

            igcFile.DTE = string.Format("HFDTE{0}", firstFix.ToString("ddMMyy"));
            declTime = (firstFix + TimeSpan.FromSeconds(declarationTime)).ToString("ddMMyyHHmmss");

            realTime = firstFix;
            end = false;
            idx = 0;

            while (!end) {
                mainType = (byte)(buffer[idx] & (byte)VLMainRecordType.rectyp_msk);
                switch (mainType) {
                case (byte)VLMainRecordType.rectyp_sep:
                    if (binaryFileVersion > maxBfv) {
                        // unsupported binary file version
                        end = true;
                    }
                    dsLength = 1;
                    break;
                case (byte)VLMainRecordType.rectyp_fil:
                    dsLength = 1;
                    break;
                case (byte)VLMainRecordType.rectyp_pos:
                case (byte)VLMainRecordType.rectyp_poc:
                    if((buffer[idx + 2] & 0x80) != 0) { // Endebedingung
                        end = true;
                    }
                    else {
                        realTime += TimeSpan.FromSeconds(buffer[idx + 2]);
                        validFix = ((buffer[idx] & 0x10) >> 4) != 0 ? true : false;
                        pressure = ((buffer[idx] & 0x0f) << 8) | buffer[idx + 1];
                        if (mainType == (byte)VLMainRecordType.rectyp_pos) {
                            dsLength = pos_ds_size[binaryFileVersion, 0];
                            lat = (buffer[idx + 3] & 0x7f) << 16 | (buffer[idx + 4] << 8) | buffer[idx + 5];
                            if ((buffer[idx + 3] & 0x80) == 0) {
                                lat = -lat;
                            }
                            lon = (buffer[idx + 6] << 16) | (buffer[idx + 7] << 8) | buffer[idx + 8];
                            if ((buffer[idx + 9] & 0x80) != 0) {
                                lon = -lon;
                            }
                            gpsAlt = ((buffer[idx + 9] & 0x70) << 4) | buffer[idx + 10];
                            fxa = HdopToFxa((byte)(buffer[idx + 9] & 0x0f));
                            enl = 4 * buffer[idx + 11];
                        }
                        else {
                            dsLength = pos_ds_size[binaryFileVersion, 1];
                            diffLat = ((buffer[idx + 3] & 0x07) << 8) | buffer[idx + 4];
                            if ((buffer[idx + 3] & 0x80) == 0) {
                                diffLat = -diffLat;
                            }

                            diffLon = ((buffer[idx + 3] & 0x78) << 5) | buffer[idx + 5];
                            if ((buffer[idx + 6] & 0x80) != 0) {
                                diffLon = -diffLon;
                            }

                            lat += diffLat;
                            lon += diffLon;
                            gpsAlt = ((buffer[idx + 6] & 0x70) << 4) | buffer[idx + 7];
                            fxa = HdopToFxa((byte)(buffer[idx + 6] & 0x0f));
                            enl = 4 * buffer[idx + 8];
                        }

                        gpsAlt = 10 * gpsAlt - 1000;
                        enl = EnlFlt(enl);
                        pAlt = PressureToAltitude(pressure);

                        igcFile.BRecord.Add(string.Format("B{0}{1:00}{2:00000}{3}{4:000}{5:00000}{6}{7}{8:00000}{9:00000}{10:000}{11:000}", 
                            realTime.ToString("HHmmss"),
                            Math.Abs(lat / 60000),
                            Math.Abs(lat % 60000),
                            lat <= 0 ? "N" : "S",
                            Math.Abs(lon / 60000),
                            Math.Abs(lon % 60000),
                            lon < 0 ? "W" : "E",
                            validFix == true ? "A" : "V",
                            pAlt,
                            gpsAlt,
                            fxa,
                            enl));
                    }
                    break;
                case (byte)VLMainRecordType.rectyp_vrt:
                case (byte)VLMainRecordType.rectyp_vrb:
                    dsLength = buffer[idx + 1];
                    if (mainType == (byte)VLMainRecordType.rectyp_vrb) {
                        idx2 = idx + 2;
                    }
                    else {
                        realTime += TimeSpan.FromSeconds(buffer[idx + 2]);
                        idx2 = idx + 3;
                    }
                    subType = buffer[idx2];
                    switch (subType) {
                    case (byte)VLSubRecordType.FLDEPEV:
                        igcFile.BRecord.Add(string.Format("E{0}PEVEVENTBUTTON PRESSED", realTime.ToString("HHmmss")));
                        break;
                    case (byte)VLSubRecordType.FLDETKF:
                        igcFile.BRecord.Add(string.Format("LGCSTKF{0}TAKEOFF DETECTED", realTime.ToString("HHmmss")));
                        break;
                    }
                    break;
                case (byte)VLMainRecordType.rectyp_tnd:
                    dsLength = 8;
                    realTime += TimeSpan.FromSeconds(buffer[idx + 1]);
                    break;
                default:
                    end = true;
                    break;
                }
                idx += dsLength;
            }

            // create g-record
            byte [] tmp = new byte[3];
            int triCnt = 0;
            int blockCnt = 0;
            string gRecord = "";
            for (idx = 0; idx < signaturePos; idx++) {
                tmp[triCnt++] = buffer[idx];
                if (triCnt == 3) {
                    triCnt = 0;
                    gRecord += new string(ToBase64(tmp));
                    tmp[0] = 0xff;
                    tmp[1] = 0xff;
                    tmp[2] = 0xff;
                    blockCnt++;
                    if (blockCnt == 18) {
                        blockCnt = 0;
                        igcFile.GRecord.Add(string.Format("G{0}", gRecord));
                        gRecord = "";
                    }
                }
            }

            if (triCnt > 0 || blockCnt > 0) {
                gRecord += new string(ToBase64(tmp));
                igcFile.GRecord.Add(string.Format("G{0}", gRecord));
            }

            igcFile.CRecord.Add(string.Format("C{0}000000{1:0000}{2:00}", declTime, taskId, nrTurnpoints));
            for (i = 0; i < nrTurnpoints + 2; i++) {
                wp = (WayPoint)CRecordTmp[i];
                latDeg = Math.Abs(wp.Latitude) / 360000;
                latMin = (Math.Abs(wp.Latitude) % 360000) / 6.0;
                lonDeg = Math.Abs(wp.Longitude) / 360000;
                lonMin = (Math.Abs(wp.Longitude) % 360000) / 6.0;
                igcFile.CRecord.Add(string.Format("C{0:00}{1:00000}{2}{3:000}{4:00000}{5}{6}", latDeg, latMin, wp.Latitude <= 0 ? "N" : "S", lonDeg, lonMin, wp.Longitude < 0 ? "W" : "E", wp.shortName));
            }
            for (i = CRecordTmp.Count - 2; i < CRecordTmp.Count; i++) {
                wp = (WayPoint)CRecordTmp[i];
                latDeg = Math.Abs(wp.Latitude) / 360000;
                latMin = (Math.Abs(wp.Latitude) % 360000) / 6.0;
                lonDeg = Math.Abs(wp.Longitude) / 360000;
                lonMin = (Math.Abs(wp.Longitude) % 360000) / 6.0;
                igcFile.CRecord.Add(string.Format("C{0:00}{1:00000}{2}{3:000}{4:00000}{5}{6}", latDeg, latMin, wp.Latitude <= 0 ? "N" : "S", lonDeg, lonMin, wp.Longitude < 0 ? "W" : "E", wp.shortName));
            }

            return igcFile;
        }

        private char [] ToBase64(byte [] tmp) {
            char [] bas64ar = new char[4];
            bas64ar[0] = base64tab[tmp[0] >> 2];
            bas64ar[1] = base64tab[((tmp[0] & 0x03) << 4) | (tmp[1] >> 4)];
            bas64ar[2] = base64tab[((tmp[1] & 0x0f) << 2) | (tmp[2] >> 6)];
            bas64ar[3] = base64tab[tmp[2] & 0x3f];
            return bas64ar;
        }

        private void SetComSpeed(SerialPorts.LineSpeed speed) {
            comPort.Close();
            Thread.Sleep(300);
            base.Open(this.port, speed);
        }

        private void Unpack(byte [] buffer, int idx, WayPoint p) {
            char [] trimChars = {'\0'};
            long lat, lon;

            p.shortName = Encoding.ASCII.GetString(buffer, idx, 6).Trim(trimChars);
            
            //typ = packed[6] &0x7F;
            lat = ((buffer[idx + 7] & 0x7F) * 65536) + (buffer[idx + 8] * 256) + buffer[idx + 9];
            if (lat >= 5400000) {
                lat = 5400000;
            }
            lat = ((lat / 60000) * 360000) + ((lat % 60000) * 6);
            p.Latitude = (int)((buffer[idx + 7] & 0x80) != 0 ? lat : -lat);

            lon = (buffer[idx + 10] * 65536) + (buffer[idx + 11] * 256) + buffer[idx + 12];
            if (lon >= 10800000) {
                lon = 10800000;
            }
            lon = ((lon / 60000) * 360000) + ((lon % 60000) * 6);
            p.Longitude = (int)((buffer[idx + 6] & 0x80) != 0 ? -lon : lon);
        }

        private int EnlFlt(int enl) {
            if (enl < 500) {
                enl /= 2;
            }
            else if (enl < 750) {
                enl = 250 + 2 * (enl - 500);
            }
            else {
                enl = (int)(750 + (enl - 750) * 1.5);
            }
            return enl;
        }

        private int PressureToAltitude(int druck) {
            /*
            genaue Umrechnung von Druckwert nach Hhe.
            Druckwert ist ein innerhalb des Loggers temperaturkompensierter Wert, der
            proportional zum gemessenen Umgebungsdruck am Logger ist.
            1100 mbar entsprechen einem Druckwert von 4096;
            */            
            double GMR = 9.80665 * 28.9644 / 8314.32;
            double tgrad = -6.5E-3;
            double p0 = 1013.25;
            double p11 = 0.2233611050922 * p0;
            double t0 = 288.15;
            double t11 = 216.65;
            double p;
            double hoehe;
            // Umrechnung von normierten ADC-Werten in hPa
            p = 1100.0 * druck / 4096;
            // Berechnen der Druckhhe anhand des Druckes
            if (p > p11)
                hoehe = (t0 * (Math.Exp((tgrad / GMR) * Math.Log(p0 / p)) - 1) / tgrad);
            else
                hoehe = (t11 * Math.Log(p11 / p) / GMR + 11000);
            return (int)hoehe;
        }

        private bool ReadDatabase() {
            byte [] buffer = new byte[VLAPI_DBB_MEMSIZE];
            bool ok = false;
            IsInterrupted = false;
            if (SendCommand((byte)VLCmd.cmd_RDB, 0, (byte)dataBaudIdx) == 0) {
                SetComSpeed(dataBaud);
                try {
                    if (ok = (ReadBlock(buffer) > 0)) {
                        haveDatabase = true;
                        DB = new VolksloggerDatabase(buffer);
                        flightDeclaration = DB.flightDeclaration;
                    }
                }
                catch (Exception e) {
                    if (MessageBox.Show(string.Format("{0}\n\nRecreate database from scratch?", e.ToString()), "Transfer error", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes) {
                        DB = new VolksloggerDatabase();
                        haveDatabase = true;
                        ok = true;
                        flightDeclaration = DB.flightDeclaration;
                    }
                }
                SetComSpeed(commandBaud);
            }
            return ok;
        }

        private bool WriteDatabase() {
            byte [] buffer = DB.BuildDB();
            bool ok = false;
            IsInterrupted = false;
            if (SendCommand((byte)VLCmd.cmd_PDB, 0, 0) == 0) {
                ok = (WriteBlock(buffer) == buffer.Length);
            }
            return ok;
        }

        #endregion
    
    
        public override string Description {
            get {
                // TODO:  Add Volkslogger.Description getter implementation
                return string.Format(@"Adapter for Volkslogger GPS Recorder

This is a serial connection adapter. The prefered line speed is {0} baud. If you ran into trouble with incomplete transfers, then try at a lower speed.

Connect to the recorder within the first 10 secs. after power up (while VL is counting down) or switch VL into transfer mode with menu
ACT.PC

Note that each turnpoint can be defined individual in type (cylinder/sector/line) and size (radius, width).
", preferedLineSpeed) + base.OptionsString;
            }
        }    
    }

    internal class VolksloggerDatabase {
        private enum DBOffsets {DBBBeg  = 0x0000, DBBEnd  = 0x3000, FrmBeg  = 0x3000, FrmEnd  = 0x4000};
        private byte [] dbBlock;
        private byte [] declBlock;
        private Header[] headers = new Header[8];

        struct Header {
            internal int dsCount;
            internal uint firstDs;
            internal uint lastDs;
            internal int dsLength;
            internal int keyLength;
        }

        internal WaypointCatalog waypoints;
        internal ArrayList tasks;
        internal ArrayList pilots;
        internal FlightDeclaration flightDeclaration;
        internal Task declarationTask;

        public VolksloggerDatabase() {
            waypoints = new WaypointCatalog();
            tasks = new ArrayList();
            pilots = new ArrayList();
            flightDeclaration = new FlightDeclaration();
            declarationTask = new Task();

            InitDB();
        }

        public VolksloggerDatabase(byte [] buffer) : this() {
            Buffer.BlockCopy(buffer, (int)DBOffsets.DBBBeg, dbBlock, 0, dbBlock.Length);
            Buffer.BlockCopy(buffer, (int)DBOffsets.FrmBeg, declBlock, 0, declBlock.Length);
            int i, pos, nullPos;
            string tmpS;

            // init headers from readout
            for(i = 0; i < headers.Length; i++) {
                if (dbBlock[6 * i] != 0xff || dbBlock[(6 * i) + 1] != 0xff) {
                    headers[i].firstDs = (uint)256 * dbBlock[6 * i] + dbBlock[(6 * i) + 1];
                    headers[i].lastDs = (uint)256 * dbBlock[(6 * i) + 2] + dbBlock[(6 * i) + 3];
                    headers[i].dsLength = dbBlock[(6 * i) + 4];
                    headers[i].keyLength = dbBlock[(6 * i) + 5];
                }
            }

            // convert and get list of waypoints
            if (headers[0].firstDs != 0xffff) {
                i = 1 + ((int)headers[0].lastDs - (int)headers[0].firstDs) / headers[0].dsLength;
                pos = (int)headers[0].firstDs;
                while (i-- > 0) {
                    waypoints.Add(GetWP(pos, dbBlock));
                    pos += headers[0].dsLength;
                }
            }

            // convert and get list of tasks
            if (headers[3].firstDs != 0xffff) {
                i = 1 + ((int)headers[3].lastDs - (int)headers[3].firstDs) / headers[3].dsLength;
                pos = (int)headers[3].firstDs;
                while (i-- > 0) {
                    tasks.Add(GetTask(pos));
                    pos += headers[3].dsLength;
                }
            }
            
            // convert and get list of pilots
            if (headers[1].firstDs != 0xffff) {
                i = 1 + (int)(headers[1].lastDs - (int)headers[1].firstDs) / headers[1].dsLength;
                pos = (int)headers[1].firstDs;
                while (i-- > 0) {
                    pilots.Add(GetPilot(pos));
                    pos += headers[1].dsLength;
                }
            }

            flightDeclaration = new FlightDeclaration();

            // convert and get flight declaration
            for (pos = 0; pos < 4; pos++) {
                if ((i = FindDeclarationField((int)Volkslogger.VLSubRecordType.FLDPLT1 + pos)) >= 0) {
                    tmpS = Encoding.ASCII.GetString(declBlock, i + 2, 16);
                    if ((nullPos = tmpS.IndexOf('\0')) != -1) {
                        tmpS = tmpS.Substring(0, nullPos);
                    }
                    flightDeclaration.pilot += tmpS.Trim() + ' ';
                }
            }
            flightDeclaration.pilot = flightDeclaration.pilot.TrimEnd(' ').ToUpper();

            if ((i = FindDeclarationField((int)Volkslogger.VLSubRecordType.FLDGID)) >= 0) {
                tmpS = Encoding.ASCII.GetString(declBlock, i + 2, 7);
                if ((nullPos = tmpS.IndexOf('\0')) != -1) {
                    tmpS = tmpS.Substring(0, nullPos);
                }
                flightDeclaration.gliderId = tmpS.Trim().ToUpper();
            }
            
            if ((i = FindDeclarationField((int)Volkslogger.VLSubRecordType.FLDGTY)) >= 0) {
                tmpS = Encoding.ASCII.GetString(declBlock, i + 2, 12);
                if ((nullPos = tmpS.IndexOf('\0')) != -1) {
                    tmpS = tmpS.Substring(0, nullPos);
                }
                flightDeclaration.gliderType = tmpS.Trim().ToUpper();
            }

            if ((i = FindDeclarationField((int)Volkslogger.VLSubRecordType.FLDCCL)) >= 0) {
                tmpS = Encoding.ASCII.GetString(declBlock, i + 2, 12);
                if ((nullPos = tmpS.IndexOf('\0')) != -1) {
                    tmpS = tmpS.Substring(0, nullPos);
                }
                flightDeclaration.competitionClass = tmpS.Trim().ToUpper();
            }

            if ((i = FindDeclarationField((int)Volkslogger.VLSubRecordType.FLDCID)) >= 0) {
                tmpS = Encoding.ASCII.GetString(declBlock, i + 2, 3).TrimEnd(' ').ToUpper();
                if ((nullPos = tmpS.IndexOf('\0')) != -1) {
                    tmpS = tmpS.Substring(0, nullPos);
                }
                flightDeclaration.competitionId = tmpS.Trim().ToUpper();
            }

            if ((i = FindDeclarationField((int)Volkslogger.VLSubRecordType.FLDTKF)) >= 0) {
                GetDeclWP(i + 2);
            }
            if ((i = FindDeclarationField((int)Volkslogger.VLSubRecordType.FLDSTA)) >= 0) {
                GetDeclWP(i + 2);
            }

            if ((i = FindDeclarationField((int)Volkslogger.VLSubRecordType.FLDNTP)) >= 0) {
                int N = declBlock[i + 2];
                for (pos = 0; pos < N; pos++) {
                    if ((i = FindDeclarationField((int)Volkslogger.VLSubRecordType.FLDTP1 + pos)) >= 0) {
                        GetDeclWP(i + 2);
                    }
                }
            }
            if ((i = FindDeclarationField((int)Volkslogger.VLSubRecordType.FLDFIN)) >= 0) {
                GetDeclWP(i + 2);
                GetDeclWP(i + 2);
            }
        }

        private void InitDB() {
            int i;
            // init database and headers
            dbBlock = new byte [DBOffsets.DBBEnd - DBOffsets.DBBBeg];
            declBlock = new byte [DBOffsets.FrmEnd - DBOffsets.FrmBeg];
            
            for (i = 0; i < dbBlock.Length; i++) {
                dbBlock[i] = 0xff;
            }
            
            for (i = 0; i < declBlock.Length; i++) {
                declBlock[i] = 0xff;
            }

            for (i = 0; i < headers.Length; i++) {
                headers[i].dsCount = 0;
                headers[i].firstDs = 0xffff;
                headers[i].lastDs = 0xffff;
            }
            headers[0].dsLength = 13;
            headers[0].keyLength = 6;
            headers[1].dsLength  = 16;
            headers[1].keyLength = 16;
            headers[2].dsLength  = 7;
            headers[2].keyLength = 7;
            headers[3].dsLength  = 144;
            headers[3].keyLength = 14;
        }

        internal byte [] BuildDB() {
            byte [] buffer = new byte[Volkslogger.VLAPI_DBB_MEMSIZE];
            int pos = 6 * 8; // 48 bytes for headers, start behind headers
            int i;
            InitDB();
            byte [] tmp;
            //pilots.Add("harald maier");

            // convert and write waypoints
            tmp = new byte[headers[0].dsLength];
            // set pos of first record
            headers[0].firstDs = (uint)pos;
            i = 0;
            foreach (WayPoint wp in waypoints) {
                if (i >= 500) {
                    break;
                }
                PutWP(wp, tmp);
                pos = AddDs(0, pos, tmp);
                i++;
            }
            CloseTable(0);

            // convert and write list of pilots
            tmp = new byte[headers[1].dsLength];
            // set pos of first record
            headers[1].firstDs = (uint)pos;
            i = 0;
            foreach (Pilot p in pilots) {
                if (i >= 25) {
                    break;
                }
                PutPilot(p, tmp);
                pos = AddDs(1, pos, tmp);
                i++;
            }
            CloseTable(1);

            // convert and write list of tasks
            tmp = new byte[headers[3].dsLength];
            // set pos of first record
            headers[3].firstDs = (uint)pos;
            i = 0;
            foreach (Task t in tasks) {
                if (i >= 25) {
                    break;
                }
                PutTask(t, tmp);
                pos = AddDs(3, pos, tmp);
                i++;
            }
            CloseTable(3);

            // convert and write flight declaration
            pos = 0;
            tmp = new byte[16];
            string tmpS = flightDeclaration.pilot.ToUpper().PadRight(64);
            for (i = 0; i < 4; i++) {
                Encoding.ASCII.GetBytes(tmpS, i * 16, 16, tmp, 0);
                pos = AddFDFDs((int)Volkslogger.VLSubRecordType.FLDPLT1 + i, pos, tmp);
            }

            tmp = new byte[7];
            Encoding.ASCII.GetBytes(flightDeclaration.gliderId.ToUpper().PadRight(7), 0, 7, tmp, 0);
            pos = AddFDFDs((int)Volkslogger.VLSubRecordType.FLDGID, pos, tmp);

            tmp = new byte[12];
            Encoding.ASCII.GetBytes(flightDeclaration.gliderType.ToUpper().PadRight(12), 0, 12, tmp, 0);
            pos = AddFDFDs((int)Volkslogger.VLSubRecordType.FLDGTY, pos, tmp);

            Encoding.ASCII.GetBytes(flightDeclaration.competitionClass.ToUpper().PadRight(12), 0, 12, tmp, 0);
            pos = AddFDFDs((int)Volkslogger.VLSubRecordType.FLDCCL, pos, tmp);

            tmp = new byte[3];
            Encoding.ASCII.GetBytes(flightDeclaration.competitionId.ToUpper().PadRight(3), 0, 3, tmp, 0);
            pos = AddFDFDs((int)Volkslogger.VLSubRecordType.FLDCID, pos, tmp);

            tmp = new byte[16];
            if (declarationTask.Count >= 4) {
                // takeoff
                PutDeclWP((WayPoint)declarationTask[0], declarationTask.GetSector(0), tmp);
                pos = AddFDFDs((int)Volkslogger.VLSubRecordType.FLDTKF, pos, tmp);
                // start
                PutDeclWP((WayPoint)declarationTask[1], declarationTask.GetSector(1), tmp);
                pos = AddFDFDs((int)Volkslogger.VLSubRecordType.FLDSTA, pos, tmp);

                for (i = 2; i < Math.Min(declarationTask.Count - 2, 10); i++) {
                    // rout points
                    PutDeclWP((WayPoint)declarationTask[i], declarationTask.GetSector(i), tmp);
                    pos = AddFDFDs((int)Volkslogger.VLSubRecordType.FLDTP1 + (i - 2), pos, tmp);
                }

                PutDeclWP((WayPoint)declarationTask[i], declarationTask.GetSector(i), tmp);
                pos = AddFDFDs((int)Volkslogger.VLSubRecordType.FLDFIN, pos, tmp);

                tmp = new byte [1];
                tmp[0] = (byte)(declarationTask.Count - 4);
                pos = AddFDFDs((int)Volkslogger.VLSubRecordType.FLDNTP, pos, tmp);

            }

            Buffer.BlockCopy(dbBlock, 0, buffer, (int)DBOffsets.DBBBeg, dbBlock.Length);
            Buffer.BlockCopy(declBlock, 0, buffer, (int)DBOffsets.FrmBeg, declBlock.Length);

            return buffer;
        }
 
        private void PutTask(Task t, byte [] buffer) {
            int i;
            int pos = 0;
            // convert and write waypoints
            byte [] tmp = new byte[headers[0].dsLength];

            Encoding.ASCII.GetBytes(t.Name.ToUpper().PadRight(14), 0, 14, buffer, 0);
            pos += 14;

            // ignore takeoff and landing
            for(i = 1; i < Math.Min(t.Count - 1, 11); i++) {
                PutWP(t[i], tmp);
                Buffer.BlockCopy(tmp, 0, buffer, pos, tmp.Length);
                pos += tmp.Length;
            }

            //fill record with blank route points
            tmp = new byte[headers[0].dsLength];
            for (int ii = 0; ii < tmp.Length; ii++) {
                tmp[ii] = 0xff;
            }
            while (i <= 10) {
                Buffer.BlockCopy(tmp, 0, buffer, pos, tmp.Length);
                pos += tmp.Length;
                i++;
            }
        }

        private void PutPilot(Pilot p, byte [] buffer) {
            Encoding.ASCII.GetBytes(string.Format("{0} {1}", p.FirstName, p.LastName).ToUpper().PadRight(16), 0, 16, buffer, 0);
        }

        private void PutDeclWP(WayPoint wp, SectorDefinition sector, byte [] buffer) {
            PutWP(wp, buffer);

            buffer[13] = (byte)(sector.directionFrom / 2);

            switch (sector.sectorType) {
            case SectorTypes.Line:
                buffer[15] = (byte)Volkslogger.VLSectorTypes.LINE;
                // find two integer numbers between 1 and 15 the product of which
                // is just lower or equal the linewidth
                int w1 = 0;
                int w2 = 0;
                int r = sector.radius1 / 1000 * 2; 
                for(int i = 1; i <= 15; i++) {
                    if(r % i == 0 && r / i <= 15) {
                        w1 = i;
                        w2 = r / i;
                        break;
                    }
                }
                buffer[14] = (byte)((w1 << 4) + w2);
                break;
            case SectorTypes.Sector:
                buffer[15] = (byte)Volkslogger.VLSectorTypes.SECTOR;
                buffer[14] = (byte)(Math.Min(sector.radius2 / 100, 15) + (Math.Min(sector.radius1 / 1000, 15) << 4 ));
                break;
            default:
                // we can't handle areas here, reset to std sector
                // automatic calculation
                buffer[13] = 180;
                // no cylinder, sector endless (FAI sector)
                buffer[14] = (byte)(0 + (15 << 4 ));
                buffer[15] = (byte)Volkslogger.VLSectorTypes.SECTOR;
                break;
            }
        }

        private void PutWP(WayPoint wp, byte [] buffer) {
            byte type = 0;
            int lat, lon;
            Encoding.ASCII.GetBytes(wp.shortName.ToUpper().PadRight(6), 0, 6, buffer, 0);
            lat = Math.Abs(wp.Latitude / 6); // see getwp
            lon = Math.Abs(wp.Longitude / 6); // see getwp
            if (wp.type == WayPointTypeId.Airport || wp.type == WayPointTypeId.Glidersite) {
                type = (byte)Volkslogger.VolksloggerFlags.Airport;
            }
            if (wp.surface == SurfaceTypeId.Hard) {
                type |= (byte)Volkslogger.VolksloggerFlags.HardSurface;
            }
            if (wp.landable) {
                type |= (byte)Volkslogger.VolksloggerFlags.Landable;
            }
            buffer[6] = (byte)((type & 0x7f) | (wp.Longitude < 0 ? 0x80 : 0));
            buffer[7] = (byte)((lat >> 16) | ((wp.Latitude > 0) ? 0x80 : 0));
            lat &= 0x0000ffff;
            buffer[8] = (byte)(lat >> 8);
            buffer[9] = (byte)(lat & 0x000000ff);
            buffer[10] = (byte)(lon >> 16);
            lon &= 0x0000ffff;
            buffer[11] = (byte)(lon >> 8);
            buffer[12] = (byte)(lon & 0x000000ff);
        }

        private void GetDeclWP(int pos) {
            // just read the waypoint, ignore sectors and cylinders (so far)
            SectorDefinition sector;
            declarationTask.Add(GetWP(pos, declBlock));
            sector = declarationTask.GetSector(declarationTask.Count - 1);

            Volkslogger.VLSectorTypes ozTyp = (Volkslogger.VLSectorTypes)declBlock[pos + 15];
            sector.directionFrom = declBlock[pos + 13] * 2;
            if(ozTyp == Volkslogger.VLSectorTypes.LINE) {
                sector.sectorType = SectorTypes.Line;
                sector.radius1 = ((declBlock[pos + 14] & 0x0f) * ((declBlock[pos + 14] & 0xf0) >> 4) * 1000) / 2;
            }
            else {
                sector.sectorType = SectorTypes.Sector;
                sector.radius1 = ((declBlock[pos + 14] & 0xf0) >> 4) * 1000;
                sector.radius2 = (declBlock[pos + 14] & 0x0f) * 100;
            }
        }

        private WayPoint GetWP(int pos, byte [] buffer) {
            WayPoint wp;
            string name;
            int ll;
            int lat, lon, type;
            name = Encoding.ASCII.GetString(buffer, pos, 6).TrimEnd(' ').ToUpper();
            type = (int)(buffer[pos + 6] & 0x7f);

            ll = 65536 * (buffer[pos + 7] & 0x7f) + 256 * buffer[pos + 8] + buffer[pos + 9];
            lat = ll * 6; // / 60000.0 * 360000.0
            if ((buffer[pos + 7] & 0x80) == 0) {
                lat = -lat;
            }
            ll = 65536 * buffer[pos + 10] + 256 * buffer[pos + 11] + buffer[pos + 12];
            lon = ll * 6; // / 60000.0 * 360000.0
            if ((buffer[pos + 6] & 0x80) != 0) {
                lon = -lon;
            }
            //ll = (int32) (65536.0 * p[10] + 256 * p[11] + p[12]);
            //lon = ll / 60000.0;
            //if (p[6] & 0x80)
            //    lon = -lon;
            wp = new WayPoint(lat, lon, name);
            wp.shortName = name;
            if ((type & (int)Volkslogger.VolksloggerFlags.Airport) > 0) {
                wp.type = WayPointTypeId.Airport;
            }
            wp.landable = ((type & (int)Volkslogger.VolksloggerFlags.Landable) > 0);
            if (wp.landable) {
                if ((type & (int)Volkslogger.VolksloggerFlags.HardSurface) > 0) {
                    wp.surface = SurfaceTypeId.Hard;
                }
                else {
                    wp.surface = SurfaceTypeId.Grass;
                }
            }

            return wp;
        }

        private Task GetTask(int pos) {
            WayPoint wp;
            Task t;
            string name;
            int i;

            name = Encoding.ASCII.GetString(dbBlock, pos, 14).TrimEnd(' ').ToUpper();
            pos += 14;
            t = new Task(name);

            for(i = 0; i < 10; i++) {
                wp = GetWP(pos, dbBlock);
                if (char.IsLetterOrDigit(wp.longName[0])) {
                    if (t.Count == 0) {
                        // prepend takeoff
                        t.Add(wp);
                    }
                    t.Add(wp);
                    pos += 13;
                }
                else {
                    // append landing
                    t.Add(t[t.Count - 1]);
                    break;
                }
            }

            return t;
        }

        private Pilot GetPilot(int pos) {
            char [] delim = {' '};
            string [] tokens;
            Pilot p = new Pilot();
            tokens = Encoding.ASCII.GetString(dbBlock, pos, 16).TrimEnd(' ').ToUpper().Split(delim, 2);
            p.FirstName = tokens.Length > 0 ? tokens[0] : "UNKNOWN";
            p.LastName = tokens.Length > 1 ? tokens[1] : "UNKNOWN";
            return p;
        }

        private int FindDeclarationField(int id) {
            int i;
            int ii = -1;
            try {
                for(i = 0; i < declBlock.Length - 1; ) {
                    if (declBlock[i + 1] == id) {
                        // Feld gefunden
                        ii = i;
                        break;
                    }
                    if (declBlock[i] == 0) {
                        // Zyklus verhindern
                        ii = -1;
                        break;
                    }
                    i += declBlock[i];
                }
            }
            catch {
                ii = -1;
            }
            return ii;
        }

        private int AddDs(int table, int pos, byte [] ds) {
            int dsLength = headers[table].dsLength;
            if (pos + dsLength < (int)DBOffsets.DBBEnd) {
                // only open tables
                if (headers[table].lastDs == 0xffff) {
                    Buffer.BlockCopy(ds, 0, dbBlock, pos, dsLength);
                    headers[table].dsCount++;
                    pos += dsLength;
                }
            }
            return pos;
        }

        private int AddFDFDs(int id, int pos, byte [] buffer) {
            int dsLength = buffer.Length;
            if (pos + dsLength + 2  < (int)DBOffsets.FrmEnd) {
                declBlock[pos] = (byte)(dsLength + 2);
                declBlock[pos + 1] = (byte)id;
                Buffer.BlockCopy(buffer, 0, declBlock, pos + 2, dsLength);
                pos += dsLength + 2;
            }
            return pos;
        }

        private void CloseTable(int table) {
            int header = table * 6;
            // calculate position of last record
            headers[table].lastDs = headers[table].firstDs + (uint)((headers[table].dsCount - 1) * headers[table].dsLength);
            // build database header
            dbBlock[header] = (byte)(headers[table].firstDs / 256);
            dbBlock[header + 1] = (byte)(headers[table].firstDs % 256);
            dbBlock[header + 2] = (byte)(headers[table].lastDs / 256);
            dbBlock[header + 3] = (byte)(headers[table].lastDs % 256);
            dbBlock[header + 4] = (byte)(headers[table].dsLength);
            dbBlock[header + 5] = (byte)(headers[table].keyLength);
        }
    }
}
