// name   : Map.cs
// author : Harald Maier
// date   : 03.08.2003
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;

using SoaringDotNET.GUI;
using SoaringDotNET.Data;

namespace SoaringDotNET.Controls
{
    /// <summary>
    /// Summary description for Map.
    /// </summary>
    public class Map : System.Windows.Forms.UserControl
    {
        private SoaringDotNet app;
        private WgsArea viewArea;

        private Font font;
        private Pen dashPen;
        
        private Bitmap mapBmp = null;
        private TextureBrush mapBrush = null;
        private Pen mapPen = null;

        private Bitmap totalBmp = null;
        private TextureBrush totalBrush = null;
        private Pen totalPen = null;

        private Pen zoomPen = null;
        private int oldX = 0, oldY = 0;
        private int oldWidth = 0, oldHeight = 0;
        private int startX = 0, startY = 0;
        private Point popupWindowPos = new Point(0, 0);
        private int draggingTaskPointIdx = -1;
        private bool draggingArea;
        private bool draggingTask;
        private bool moving;

        private WaypointCatalog currentCatalog;
        private Flight currentFlight;
        private System.Windows.Forms.ContextMenu contextMenuMap;
        private System.Windows.Forms.MenuItem menuItemCenterMap;
        private System.Windows.Forms.MenuItem menuItemCreateWaypoint;
        private Task currentTask;
        private WayPoint dummyPoint;
        private AirspcaePopupInfo toolTip;
        private System.Windows.Forms.MenuItem menuItemCenterMapToHomepoint;
        private System.Windows.Forms.MenuItem menuItemEditWaypoint;
        private System.Windows.Forms.MenuItem menuItem1;
        private System.Windows.Forms.MenuItem menuItemAirspaceInfo;
        private System.Windows.Forms.MenuItem menuItem2;
        private System.Windows.Forms.MenuItem menuItemCreateTask;
        private System.Windows.Forms.MenuItem menuItem3;

        private enum TaskActions {MovePoint, AddPoint, RemovePoint};
        private TaskActions taskAction;

        public Map() {
            font = new Font("Arial", 8.0f);
            dashPen = new Pen(Color.Black);
            dashPen.DashStyle = DashStyle.Dash;
            zoomPen = new Pen(Color.Black);
            zoomPen.DashStyle = DashStyle.Dot;
            //
            // Required for Windows Form Designer support
            //
            InitializeComponent();

            //
            // TODO: Add any constructor code after InitializeComponent call
            //
            viewArea = new WgsArea(-49*360000, 8*360000, -47*360000, 10*360000);

            currentCatalog = null;
            currentFlight = null;
            currentTask = null;

            draggingArea = false;
            draggingTask = false;
            moving = false;

            dummyPoint = new WayPoint();
            taskAction = TaskActions.MovePoint;
            toolTip = new AirspcaePopupInfo(this);
        }

        #region Windows Form Designer generated code
        /// <summary>
        /// Required method for Designer support - do not modify
        /// the contents of this method with the code editor.
        /// </summary>
        private void InitializeComponent() {
            this.contextMenuMap = new System.Windows.Forms.ContextMenu();
            this.menuItemCenterMap = new System.Windows.Forms.MenuItem();
            this.menuItemCenterMapToHomepoint = new System.Windows.Forms.MenuItem();
            this.menuItemCreateWaypoint = new System.Windows.Forms.MenuItem();
            this.menuItemEditWaypoint = new System.Windows.Forms.MenuItem();
            this.menuItem1 = new System.Windows.Forms.MenuItem();
            this.menuItemAirspaceInfo = new System.Windows.Forms.MenuItem();
            this.menuItem2 = new System.Windows.Forms.MenuItem();
            this.menuItemCreateTask = new System.Windows.Forms.MenuItem();
            this.menuItem3 = new System.Windows.Forms.MenuItem();
            // 
            // contextMenuMap
            // 
            this.contextMenuMap.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
                                                                                           this.menuItemCenterMap,
                                                                                           this.menuItemCenterMapToHomepoint,
                                                                                           this.menuItem2,
                                                                                           this.menuItemCreateTask,
                                                                                           this.menuItem3,
                                                                                           this.menuItemCreateWaypoint,
                                                                                           this.menuItemEditWaypoint,
                                                                                           this.menuItem1,
                                                                                           this.menuItemAirspaceInfo});
            this.contextMenuMap.Popup += new System.EventHandler(this.OnPopupContextMenu);
            // 
            // menuItemCenterMap
            // 
            this.menuItemCenterMap.Index = 0;
            this.menuItemCenterMap.Text = "&Center map";
            this.menuItemCenterMap.Click += new System.EventHandler(this.OnCenterMap);
            // 
            // menuItemCenterMapToHomepoint
            // 
            this.menuItemCenterMapToHomepoint.Index = 1;
            this.menuItemCenterMapToHomepoint.Text = "Center map to &homepoint";
            this.menuItemCenterMapToHomepoint.Click += new System.EventHandler(this.OnCenterHome);
            // 
            // menuItemCreateWaypoint
            // 
            this.menuItemCreateWaypoint.Index = 5;
            this.menuItemCreateWaypoint.Text = "Create &waypoint";
            this.menuItemCreateWaypoint.Click += new System.EventHandler(this.OnCreateWaypoint);
            // 
            // menuItemEditWaypoint
            // 
            this.menuItemEditWaypoint.Index = 6;
            this.menuItemEditWaypoint.Text = "&Edit waypoint";
            this.menuItemEditWaypoint.Click += new System.EventHandler(this.OnEditWaypoint);
            // 
            // menuItem1
            // 
            this.menuItem1.Index = 7;
            this.menuItem1.Text = "-";
            // 
            // menuItemAirspaceInfo
            // 
            this.menuItemAirspaceInfo.Index = 8;
            this.menuItemAirspaceInfo.Text = "&Airspace info";
            this.menuItemAirspaceInfo.Click += new System.EventHandler(this.OnAirspaceInfo);
            // 
            // menuItem2
            // 
            this.menuItem2.Index = 2;
            this.menuItem2.Text = "-";
            // 
            // menuItemCreateTask
            // 
            this.menuItemCreateTask.Index = 3;
            this.menuItemCreateTask.Text = "Create &task";
            this.menuItemCreateTask.Click += new System.EventHandler(this.OnCreateTask);
            // 
            // menuItem3
            // 
            this.menuItem3.Index = 4;
            this.menuItem3.Text = "-";
            // 
            // Map
            // 
            this.BackColor = System.Drawing.Color.AliceBlue;
            this.ContextMenu = this.contextMenuMap;
            this.Cursor = System.Windows.Forms.Cursors.Default;
            this.Name = "Map";
            this.Size = new System.Drawing.Size(292, 266);

        }
        #endregion

        #region Protected Functions
        protected override void OnPaint(PaintEventArgs e) {
            // TODO:  Add Map.OnPaint implementation
            Graphics g = e.Graphics;
            Graphics bmpMap, bmpTotal;
            long text;
            long lat, lon;
            long minLat, latRange;
            long minLon, lonRange;
            long step, startPos, diff;
			//int width = Width - 1;
			//int height = Height - 1;
            string txt;
            Pen drawingPen;

            // create a new map Bitmap
            // this is the "background" drawing
            if (oldWidth != Width || oldHeight != Height) {
                mapBmp = new Bitmap(Width, Height, CreateGraphics());
                totalBmp = new Bitmap(Width, Height, CreateGraphics());
                oldWidth = Width;
                oldHeight = Height;
            }

            bmpMap = Graphics.FromImage(mapBmp);
            bmpMap.Clear(BackColor);
            bmpTotal = Graphics.FromImage(totalBmp);

            SizeArea();

            minLat = viewArea.TopLat;
            latRange = viewArea.Height;

            minLon = viewArea.LeftLon;
            lonRange = viewArea.Width;

            // draw map raster
            step = 90000;
            while(latRange / step > 10) {
                step += 9000;
            }

            diff = (Math.Abs(minLat) % step);
            if (diff > 0) {
                if (minLat < 0) {
                    startPos = diff;
                }
                else {
                    startPos = step - diff;
                }
            }
            else {
                startPos = 0;
            }

            //text = minLat + diff;
            text = minLat + startPos;

            while (startPos <= latRange) {
                lat = startPos * Height / latRange;
                if (text % 360000 == 0) {
                    txt = string.Format("{0:00}{1}", Math.Abs(text / 360000), text <= 0 ? "N" : "S");
                    bmpMap.DrawString(txt, font, Brushes.Black, 0, lat - 12);
                    drawingPen = Pens.Black;
                }
                else if (text % 180000 == 0) {
                    txt = string.Format("{0:00}30{1}", Math.Abs(text / 360000), text <= 0 ? "N" : "S");
                    bmpMap.DrawString(txt, font, Brushes.Black, 0, lat - 12);
                    drawingPen = dashPen;
                }
                else {
                    drawingPen = dashPen;
                }

                startPos += step;
                text += step;

                bmpMap.DrawLine(drawingPen, 0, lat, Width, lat);
            }
            
            step = 90000;
            while(lonRange / step > 10) {
                step += 90000;
            }

            diff = (Math.Abs(minLon) % step);
            if (diff > 0) {
                if (minLon < 0) {
                    startPos = diff;
                }
                else {
                    startPos = step - diff;
                }
            }
            else {
                startPos = 0;
            }

            text = minLon + startPos;
            //text = minLon + (minLon >= 0 ? startPos : -startPos);            
            while (startPos <= lonRange) {
                lon = startPos * Width / lonRange;

                if (text <= -64800000) {
                    text += 129600000;
                }
                else if (text > 64800000) {
                    text -= 129600000;
                }

                if (text % 360000 == 0) {
                    txt = string.Format("{0:000}{1}", Math.Abs(text / 360000), text >= 0 ? "E" : "W");
                    bmpMap.DrawString(txt, font, Brushes.Black, lon, 0);
                    drawingPen = Pens.Black;
                }
                else if (text % 180000 == 0) {
                    txt = string.Format("{0:000}30{1}", Math.Abs(text / 360000), text >= 0 ? "E" : "W");
                    bmpMap.DrawString(txt, font, Brushes.Black, lon, 0);
                    drawingPen = dashPen;
                }
                else {
                    drawingPen = dashPen;
                }

                startPos += step;
                text += step;

                bmpMap.DrawLine(drawingPen, lon, 0, lon, Height);
            }

            // draw airspaces on map
            foreach (AirspaceElement space in AppContents.airspaces) {
                space.Draw(bmpMap, viewArea, Width, Height);
            }

            // draw waypoints on map
            if (currentCatalog != null) {
                currentCatalog.Draw(bmpMap, viewArea, Width, Height);
            }

            // draw flight on map, task "over" map
            if (currentFlight != null) {
                currentFlight.Draw(bmpMap, viewArea, Width, Height);
            }
            // show map
            
            g.DrawImage(mapBmp, 0, 0);
            bmpTotal.DrawImage(mapBmp, 0, 0);

            if (currentTask != null) {
                currentTask.Draw(g, viewArea, Width, Height);
                currentTask.Draw(bmpTotal, viewArea, Width, Height);
            }
            
            // create new brush an pen objects for layered drawing (for floating objects)
            mapBrush = new TextureBrush(mapBmp);
            mapPen = new Pen(mapBrush, 3);

            // create new brush an pen objects for zooming
            totalBrush = new TextureBrush(totalBmp);
            totalPen = new Pen(totalBrush, 3);

            System.GC.Collect();
        }
        
        protected override void OnMouseWheel(MouseEventArgs e) {
            // TODO:  Add Map.OnMouseWheel implementation
            if (e.Delta < 0) {
                this.OnZoomIn();
            }
            else {
                this.OnZoomOut();
            }
        }
    
        protected override void OnMouseMove(MouseEventArgs e) {
            // TODO:  Add Map.OnMouseMove implementation
            base.OnMouseMove (e);

            int lat = (int)((long)e.Y  * (long)viewArea.Height / (long)(Height)) + viewArea.TopLat;
            int lon = (int)((long)e.X * (long)viewArea.Width / (long)(Width)) + viewArea.LeftLon;
            if (lon <= -64800000) {
                lon += 129600000;
            }
            else if (lon > 64800000) {
                lon -= 129600000;
            }
            app.UpdatePosInfo(lat, lon);

            if (e.X < 5) {
                if (e.Y < 5) {
                    Cursor = Cursors.PanNW;
                }
                else if (e.Y >= Height - 5) {
                    Cursor = Cursors.PanSW;
                }
                else {
                    Cursor = Cursors.PanWest;
                }
            }
            else if (e.X >= Width - 5) {
                if (e.Y < 5) {
                    Cursor = Cursors.PanNE;
                }
                else if (e.Y >= Height - 5) {
                    Cursor = Cursors.PanSE;
                }
                else {
                    Cursor = Cursors.PanEast;
                }
            }
            else if (e.Y < 5) {
                Cursor = Cursors.PanNorth;
            }
            else if (e.Y >= Height - 5) {
                Cursor = Cursors.PanSouth;
            }
            else {
                if (currentTask != null) {
                    if (currentTask.FindElementNearPos(e.X, e.Y, 3) != -1 ||
                        (currentCatalog != null && currentCatalog.FindElementNearPos(e.X, e.Y, 3) != -1)) {
                        Cursor = Cursors.Hand;
                    }
                    else {
                        Cursor = Cursors.Default;
                    }
                }
                else {
                    Cursor = Cursors.Default;
                }
            }

            if (draggingArea) {
                Graphics g = CreateGraphics();
                if (moving) {
                    // erase old carret
                    g.DrawRectangle(totalPen, oldX > startX ? startX : oldX, oldY > startY ? startY : oldY,
                        Math.Abs(oldX - startX), Math.Abs(oldY - startY));
                }
                else {
                    moving = true;
                }

                g.DrawRectangle(zoomPen, e.X > startX ? startX : e.X, e.Y > startY ? startY : e.Y,
                    Math.Abs(e.X - startX), Math.Abs(e.Y - startY));
                oldX = e.X;
                oldY = e.Y;
            }
            else if (draggingTask) {
                Graphics g = CreateGraphics();
                // erase old task
                int idx;
                currentTask.Draw(g, mapPen, mapBrush, viewArea, Width, Height);
                if ((idx = currentTask.FindElementNearPos(e.X, e.Y, 3)) != -1) {
                    // look for task points
                    currentTask[draggingTaskPointIdx] = currentTask[idx];
                }
                else if (currentCatalog != null && (idx = currentCatalog.FindElementNearPos(e.X, e.Y, 3)) != -1) {
                    // look for waypoints
                    currentTask[draggingTaskPointIdx] = (WayPoint)currentCatalog.VisibleWaypoints[idx];
                }
                else {
                    // insert temp point
                    dummyPoint.Latitude = lat;
                    dummyPoint.Longitude = lon;
                    currentTask[draggingTaskPointIdx] = dummyPoint;
                }
                currentTask.Draw(g, viewArea, Width, Height);
                moving = true;
            }
        }

        protected override void OnMouseDown(MouseEventArgs e) {
            base.OnMouseDown(e);
            startX = e.X;
            startY = e.Y;
            if (e.Button == MouseButtons.Left) {
                if (e.X < 5) {
                    if (e.Y < 5) {
                        OnMoveNW();
                    }
                    else if (e.Y >= Height - 5) {
                        OnMoveSW();
                    }
                    else {
                        OnMoveWest();
                    }
                }
                else if (e.X >= Width - 5) {
                    if (e.Y < 5) {
                        OnMoveNE();
                    }
                    else if (e.Y >= Height - 5) {
                        OnMoveSE();
                    }
                    else {
                        OnMoveEast();
                    }
                }
                else if (e.Y < 5) {
                    OnMoveNorth();
                }
                else if (e.Y >= Height - 5) {
                    OnMoveSouth();
                }
                else if (currentTask != null) {
                    draggingTaskPointIdx = currentTask.FindElementNearPos(e.X, e.Y, 3);
                    if (taskAction == TaskActions.AddPoint) {
                        Graphics g = CreateGraphics();
                        WayPoint w;

                        int lat = (e.Y  * viewArea.Height / (Height)) + viewArea.TopLat;
                        int lon = (e.X * viewArea.Width / (Width)) + viewArea.LeftLon;
                        int idx;
                        if (lon <= -64800000) {
                            lon += 129600000;
                        }
                        else if (lon > 64800000) {
                            lon -= 129600000;
                        }

                        // erase old task
                        currentTask.Draw(g, mapPen, mapBrush, viewArea, Width, Height);

                        if ((idx = currentTask.FindElementNearPos(e.X, e.Y, 3)) != -1) {
                            // look for task points
                            w = currentTask[idx];
                        }
                        else if (currentCatalog != null && (idx = currentCatalog.FindElementNearPos(e.X, e.Y, 3)) != -1) {
                            // look for waypoints
                            w = (WayPoint)currentCatalog.VisibleWaypoints[idx];
                        }
                        else {
                            // insert temp point
                            dummyPoint.Latitude = lat;
                            dummyPoint.Longitude = lon;
                            w = dummyPoint;
                        }

                        if (currentTask.Count == 0) {
                            // replace dummy point , create new one
                            currentTask.Insert(0, w != dummyPoint ? w : new WayPoint(w.Latitude, w.Longitude, "WP 1")/*, false*/);
                            draggingTaskPointIdx++;
                        }
                        currentTask.Insert(++draggingTaskPointIdx, w/*, false*/);
                        currentTask.Draw(g, viewArea, Width, Height);
                        app.Update(currentTask);
                    }
                    draggingTask = draggingTaskPointIdx != -1;
                    draggingArea = !draggingTask;

                    moving = false;
                }
                else {
                    draggingArea = true;
                    moving = false;
                }
            }
        }

        protected override void OnMouseUp(MouseEventArgs e) {
            // TODO:  Add Map.OnMouseUp implementation
            base.OnMouseUp(e);
            if (draggingArea && moving) {
                // erase old carret
                Graphics g = CreateGraphics();

                g.DrawRectangle(totalPen, oldX > startX ? startX : oldX, oldY > startY ? startY : oldY,
                    Math.Abs(oldX - startX), Math.Abs(oldY - startY));

                if (Math.Abs(startX - oldX) > 5 && Math.Abs(startY - oldY) > 5) {
                    long startLat = viewArea.TopLat;
                    long startLon = viewArea.LeftLon;
                    long startWidth = viewArea.Width;
                    long startHeight = viewArea.Height;

                    viewArea.TopLat = (int)(((oldY > startY ? startY : oldY) * startHeight / (Height)) + startLat);
                    viewArea.LeftLon = (int)(((oldX > startX ? startX : oldX) * startWidth / (Width)) + startLon);
                    viewArea.BottomLat = (int)(((oldY > startY ? oldY : startY) * startHeight / (Height)) + startLat);
                    viewArea.RightLon = (int)(((oldX > startX ? oldX : startX) * startWidth / (Width)) + startLon);

                    RedrawMap();
                }
            }
            else if (draggingTask && draggingTaskPointIdx != -1) {
                if (taskAction == TaskActions.RemovePoint) {
                    Graphics g = CreateGraphics();
                    if (draggingTaskPointIdx != -1) {
                        currentTask.Draw(g, mapPen, mapBrush, viewArea, Width, Height);
                        currentTask.Delete(draggingTaskPointIdx);
                        currentTask.Draw(g, viewArea, Width, Height);
                        app.Update(currentTask);
                    }
                }
                else if (currentTask[draggingTaskPointIdx] == dummyPoint) {
                    if (moving) {
                        // replace dummy point , create new one
                        currentTask[draggingTaskPointIdx] = new WayPoint(dummyPoint.Latitude, dummyPoint.Longitude, string.Format("WP {0}", draggingTaskPointIdx + 1));
                    }
                    else {
                        //remove dummy point
                        currentTask.Delete(draggingTaskPointIdx);
                    }
                }
                app.Update(currentTask);
            }
            draggingTask = false;
            draggingArea = false;
            moving = false;
        }

            
        protected override void OnKeyDown(KeyEventArgs e) {
            // TODO:  Add Map.OnKeyDown implementation
            //base.OnKeyDown (e);
            switch (e.KeyCode) {
            case Keys.ControlKey:
                taskAction = TaskActions.RemovePoint;
                break;
            case Keys.ShiftKey:
                taskAction = TaskActions.AddPoint;
                break;
            case Keys.PageUp:
                OnZoomOut();
                break;
            case Keys.PageDown:
                OnZoomIn();
                break;
            }
        }
    
        protected override void OnKeyUp(KeyEventArgs e) {
            // TODO:  Add Map.OnKeyUp implementation
            //base.OnKeyUp (e);
            if (e.KeyCode == Keys.ControlKey || e.KeyCode == Keys.ShiftKey) {
                taskAction = TaskActions.MovePoint;
            }
        }

        protected override void OnMouseEnter(EventArgs e) {
            // TODO:  Add Map.OnMouseEnter implementation
            base.OnMouseEnter (e);
            Focus();
        }
        #endregion

        #region Public Functions
        public void RedrawMap() {
            PaintEventArgs e = new PaintEventArgs(CreateGraphics(), new Rectangle());
            OnPaint(e);
        }

        public void OnZoomIn() {
            int widthExt = viewArea.Width / 16;
            int heightExt = viewArea.Height / 16;

            viewArea.LeftLon += widthExt;
            viewArea.RightLon -= widthExt;
            
            viewArea.TopLat += heightExt;
            viewArea.BottomLat -= heightExt;
            RedrawMap();
        }

        public void OnZoomOut() {
            int widthExt = viewArea.Width / 8;
            int heightExt = viewArea.Height / 8;
            if (viewArea.Width < 32400000 && viewArea.Height < 32400000 &&
               (viewArea.BottomLat + heightExt) < 32400000 && (viewArea.TopLat - heightExt) > -32400000) {
                    viewArea.LeftLon -= widthExt;
                    viewArea.RightLon += widthExt;
                    viewArea.TopLat -= heightExt;
                    viewArea.BottomLat += heightExt;

                RedrawMap();
            }
        }

        public void OnMoveWest() {
            int step = viewArea.Width / 2;
            viewArea.LeftLon -= step;
            viewArea.RightLon -= step;
            RedrawMap();
        }

        public void OnMoveEast() {
            int step = viewArea.Width / 2;
            viewArea.LeftLon += step;
            viewArea.RightLon += step;
            RedrawMap();
        }

        public void OnMoveNorth() {
            int step = viewArea.Height / 2;
            if (viewArea.TopLat - step > -32400000) {
                viewArea.TopLat -= step;
                viewArea.BottomLat -= step;
                RedrawMap();
            }
        }

        public void OnMoveSouth() {
            int step = viewArea.Height / 2;
            if (viewArea.BottomLat + step < 32400000) {
                viewArea.TopLat += step;
                viewArea.BottomLat += step;
                RedrawMap();
            }
        }

        public void OnMoveNW() {
            int step = viewArea.Height / 2;
            if (viewArea.TopLat - step > -32400000) {
                viewArea.TopLat -= step;
                viewArea.BottomLat -= step;

                step = viewArea.Width / 2;
                viewArea.LeftLon -= step;
                viewArea.RightLon -= step;

                RedrawMap();
            }
        }

        public void OnMoveNE() {
            int step = viewArea.Height / 2;
            if (viewArea.TopLat - step > -32400000) {
                viewArea.TopLat -= step;
                viewArea.BottomLat -= step;

                step = viewArea.Width / 2;
                viewArea.LeftLon += step;
                viewArea.RightLon += step;

                RedrawMap();
            }
        }

        public void OnMoveSW() {
            int step = viewArea.Height / 2;
            if (viewArea.BottomLat + step < 32400000) {
                viewArea.TopLat += step;
                viewArea.BottomLat += step;

                step = viewArea.Width / 2;
                viewArea.LeftLon -= step;
                viewArea.RightLon -= step;

                RedrawMap();
            }
        }

        public void OnMoveSE() {
            int step = viewArea.Height / 2;
            if (viewArea.BottomLat + step < 32400000) {
                viewArea.TopLat += step;
                viewArea.BottomLat += step;

                step = viewArea.Width / 2;
                viewArea.LeftLon += step;
                viewArea.RightLon += step;

                RedrawMap();
            }
        }

        public void CenterTo(WayPoint wp) {
            CenterTo(wp.Latitude, wp.Longitude);
        }

        public void CenterTo(int lat, int lon) {
            int w = viewArea.Width;
            int h = viewArea.Height;
            viewArea.LeftLon = lon - (w / 2);
            viewArea.RightLon = lon + (w / 2);
            viewArea.TopLat = lat - (h / 2);
            viewArea.BottomLat = lat + (h / 2);

            if (viewArea.LeftLon <= -64800000) {
                viewArea.LeftLon += 129600000;
            }
            else if (viewArea.LeftLon > 64800000) {
                viewArea.LeftLon -= 129600000;
            }

            if (viewArea.RightLon <= -64800000) {
                viewArea.RightLon += 129600000;
            }
            else if (viewArea.RightLon > 64800000) {
                viewArea.RightLon -= 129600000;
            }

            RedrawMap();
        }

        public void OnCenterHome() {
            int w = viewArea.Width;
            int h = viewArea.Height;
            int lat = app.appContents.homePoint.Latitude;
            int lon = app.appContents.homePoint.Longitude;
            viewArea.LeftLon = lon - (w / 2);
            viewArea.RightLon = lon + (w / 2);
            viewArea.TopLat = lat - (h / 2);
            viewArea.BottomLat = lat + (h / 2);
            RedrawMap();
        }

        public void OnCenterTask() {
            if (currentFlight != null) {
                currentFlight.CalcTaskArea();
                if (!currentFlight.currentTask.Area.IsEmpty) {
                    viewArea.TopLeft = currentFlight.currentTask.Area.TopLeft;
                    viewArea.BottomRight= currentFlight.currentTask.Area.BottomRight;
                    RedrawMap();
                }
                else {
                    OnCenterFlight();
                }
            }
            else if (currentTask != null) {
                currentTask.CalcArea();
                if (!currentTask.Area.IsEmpty) {
                    viewArea.TopLeft = currentTask.Area.TopLeft;
                    viewArea.BottomRight= currentTask.Area.BottomRight;
                    RedrawMap();
                }
            }
        }

        public void OnCenterFlight() {
            if (currentFlight != null) {
                currentFlight.CalcFlightArea();
                if (!currentFlight.Area.IsEmpty) {
                    viewArea.TopLeft = currentFlight.Area.TopLeft;
                    viewArea.BottomRight= currentFlight.Area.BottomRight;
                    RedrawMap();
                }
            }
        }

        public void Switch(WaypointCatalog c) {
            if (currentCatalog != c) {
                currentCatalog = c;
                RedrawMap();
            }
        }

        public void Update(WaypointCatalog c) {
            if (c == currentCatalog) {
                RedrawMap();
            }
        }

        public void Switch(Flight f) {
            if (currentFlight != f) {
                currentFlight = f;
                currentTask = null;
                if (currentFlight != null) {
                    viewArea.TopLeft = currentFlight.Area.TopLeft;
                    viewArea.BottomRight= currentFlight.Area.BottomRight;
                }
                RedrawMap();
            }
        }

        public void Switch(Task k) {
            if (currentTask != k) {
                currentTask = k;
                currentFlight = null;
                if (currentTask != null && !currentTask.Area.IsEmpty) {
                    viewArea.TopLeft = currentTask.Area.TopLeft;
                    viewArea.BottomRight= currentTask.Area.BottomRight;
                }
                RedrawMap();
            }
        }

        public void HideToolTip() {
            if (toolTip.Visible) {
                toolTip.Hide();
            }
        }

        #endregion
        
        #region Private Functions
        private void SizeArea() {
            // calculate correct aspect ratio
            double mapFactor = Math.Cos(Math.Abs(WgsPoint.InternalToRad(viewArea.Center.Latitude)));
            double screenFactor = (double)Width / (double)Height;

            int reqWidth = (int)(viewArea.Height / mapFactor * screenFactor);
            WgsPoint center = viewArea.Center;

            if (viewArea.Width < reqWidth) {
                reqWidth /= 2;
                // screen has sapce for more lon 
                if (viewArea.LeftLon < viewArea.RightLon) {
                    viewArea.LeftLon = center.Longitude - reqWidth;
                    viewArea.RightLon = center.Longitude + reqWidth;
                }
                else {
                    viewArea.LeftLon = viewArea.Center.Longitude + reqWidth;
                    viewArea.RightLon = viewArea.Center.Longitude - reqWidth;
                }
            }
            else if (viewArea.Width > reqWidth) {
                // we need more space for lat
                int reqHeight = (int)((viewArea.Width * mapFactor / screenFactor) / 2);
                if (viewArea.Center.Latitude - reqHeight > -32400000 && viewArea.Center.Latitude  + reqHeight < 32400000) {
                    viewArea.TopLat = center.Latitude - reqHeight;
                    viewArea.BottomLat = center.Latitude + reqHeight;
                }
            }
        }

        private void OnCreateWaypoint(object sender, System.EventArgs e) {
            int lat = (startY  * viewArea.Height / this.Height) + viewArea.TopLat;
            int lon = (startX * viewArea.Width / this.Width) + viewArea.LeftLon;
            if (lon <= -64800000) {
                lon += 129600000;
            }
            else if (lon > 64800000) {
                lon -= 129600000;
            }
        
            app.AddWaypoint(lat, lon);
        }

        private void OnCreateTask(object sender, System.EventArgs e) {
            app.NewTask();        
        }

        private void OnCenterMap(object sender, System.EventArgs e) {
            long w = viewArea.Width;
            long h = viewArea.Height;
            long lat = ((long)startY  * h / this.Height) + viewArea.TopLat;
            long lon = ((long)startX * w / this.Width) + viewArea.LeftLon;
            
            CenterTo((int)lat, (int)lon);
        }

        private void OnCenterHome(object sender, System.EventArgs e) {
            OnCenterHome();
        }

        private void OnPopupContextMenu(object sender, System.EventArgs e) {
            menuItemCreateWaypoint.Enabled = currentCatalog != null;
            popupWindowPos = PointToClient(MousePosition);
            menuItemEditWaypoint.Enabled = (currentCatalog != null && currentCatalog.FindElementNearPos(popupWindowPos, 3) != -1);
        }

        private void OnEditWaypoint(object sender, System.EventArgs e) {
            int pos;
            if (currentCatalog != null && !popupWindowPos.IsEmpty && (pos = currentCatalog.FindElementNearPos(popupWindowPos, 3)) != -1) {
                app.EditWaypoint((WayPoint)currentCatalog.VisibleWaypoints[pos]);
            }        
        }

        private void OnAirspaceInfo(object sender, System.EventArgs e) {
            Point p = PointToScreen(popupWindowPos);
            if (AppContents.airspaces != null) {
                foreach (AirspaceElement space in AppContents.airspaces) {
                    if (space.region != null && space.region.IsVisible(popupWindowPos)) {
                        toolTip.Add(space.ToString());
                    }
                }

                if (popupWindowPos.X + toolTip.Width > Width) {
                    toolTip.Left = p.X - toolTip.Width;
                }
                else {
                    toolTip.Left = p.X;
                }

                if (popupWindowPos.Y + toolTip.Height > Height) {
                    toolTip.Top = p.Y - toolTip.Height;
                }
                else {
                    toolTip.Top = p.Y;
                }
                toolTip.Show();
            }
        }

        #endregion

        #region Attributes
        public SoaringDotNet App {
            get {
                return app;
            }
            set {
                app = value;
            }
        }

        public WgsPoint Center {
            get {
                return viewArea.Center;
            }
        }
        #endregion
    }
}
