// name   : SeeYouFileHandler.cs
// author : Harald Maier
// date   : 17.10.2003
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.IO;
using System.Collections;
using System.Windows.Forms;

using SoaringDotNET.Data;
using SoaringDotNET.GUI;

namespace SoaringDotNET.FileFormats
{
	/// <summary>
	/// 
	/// </summary>
	public class SeeYouFileHandler : SoaringDotNET.FileFormats.FileHandlerBase
	{
        public enum SeeYouFlags {Unknown = 0, Waypoint = 1, AirportGrass = 2, Outlanding = 3, Glidersite = 4, AirportHard = 5, Pass = 6, Mountain = 7};
        private Hashtable waypointsByName;

        public SeeYouFileHandler()
		{
            // 
			// TODO: Add constructor logic here
			//
            waypointsByName = new Hashtable();
		}
	
        public override void Load(StreamReader f, WaypointCatalog catalogue) {
            this.Load(f, catalogue, null);
        }
    
        public override void Load(StreamReader f, WaypointCatalog catalogue, ArrayList tasks) {
            // TODO:  Add SeeYouFileHandler.Load implementation
            string line, t;
            WayPoint wp;
            string [] tokens;
            int i;

            waypointsByName.Clear();

            try {
                while ((line = f.ReadLine()) != null) {
                    // read task declarations
                    if (line.IndexOf("Related Tasks") != -1) {
                        if (tasks == null) {
                            break;
                        }

                        while ((line = f.ReadLine()) != null) {
                            tokens = line.Trim().Replace("\"", "").Split(',');
                            //,"LAUPHEIM MIL","LAUPHEIM MIL","BIBERACH RISS","LAUPHEIM MIL","LAUPHEIM MIL",
                            Task task = new Task(string.Format("Task{0}", AppContents.tasks.Count + tasks.Count + 1));
                            for (i = 1; i < tokens.Length; i++) {
                                t = tokens[i];
                                if (waypointsByName.ContainsKey(t)) {
                                    task.Add((WayPoint)waypointsByName[t]);
                                }
                            }
                            tasks.Add(task);
                        }
                    }
                    else if ((wp = ParseLine(line)) != null) {
                        waypointsByName[wp.longName] = wp;
                        catalogue.Add(wp);
                    }
                }
            }
            catch (Exception e) {
                if (e.Message != "Abort") {
                    throw e;
                }
            }
        }
    
        public override void Save(StreamWriter f, WaypointCatalog catalogue) {
            // TODO:  Add SeeYouFileHandler.Save implementation
            this.Save(f, catalogue, null);
        }
    
        public override void Save(StreamWriter f, WaypointCatalog catalogue, ArrayList tasks) {
            // TODO:  Add SeeYouFileHandler.Save implementation
            string line;
            waypointsByName.Clear();

            f.WriteLine("name,code,country,lat,lon,elev,style,rwdir,rwlen,freq,desc");

            if (catalogue != null) {
                foreach (WayPoint w in catalogue) {
                    line = BuildLine(w);
                    f.WriteLine(line);
                    waypointsByName[w.longName] = null;
                }
            }

            // ensure that all waypoints are in cup file, even those from other catalogues
            if (tasks != null) {
                foreach (Task t in tasks) {
                    foreach (WayPoint w in t) {
                        if (!waypointsByName.ContainsKey(w.longName)) {
                            line = BuildLine(w);
                            f.WriteLine(line);
                            waypointsByName[w.longName] = null;
                        }
                    }
                }

                f.WriteLine("-----Related Tasks-----");

                foreach (Task t in tasks) {
                    foreach (WayPoint w in t) {
                        f.Write(string.Format(",\"{0}\"", w.longName));
                    }
                    f.WriteLine();
                }
            }
        }
    
        public override void Save(StreamWriter f, ArrayList tasks) {
            // TODO:  Add SeeYouFileHandler.Save implementation
            this.Save(f, null, tasks);
        }
    
        public override WayPoint ParseLine(string line) {
            WayPoint wp = null;
            string [] tokens;
            string t;
            int tmp, lat, lon, i;
            char []sep = {','};
            // skip header line
            // name,code,country,lat,lon,elev,style,rwdir,rwlen,freq,desc
            try {
                tokens = line.Trim().Replace("\"", "").Split(sep, 11);
                t = tokens[0].ToLower();
                if (tokens.Length >= 11 && t != "name" && t != "title") {
                    t = tokens[3].Trim();
                    tmp = (int)(double.Parse(t.Substring(0, t.Length - 1), AppContents.ni) * 1000.0);
                    lat = (tmp / 100000) * 360000;
                    lat += (tmp % 100000) * 6;
                    if (t[t.Length - 1] == 'N') {
                        lat = -lat;
                    }

                    t = tokens[4].Trim();
                    tmp = (int)(double.Parse(t.Substring(0, t.Length - 1), AppContents.ni) * 1000.0);
                    lon = (tmp / 100000) * 360000;
                    lon += (tmp % 100000) * 6;
                    if (t[t.Length - 1] == 'W') {
                        lon = -lon;
                    }

                    t = tokens[5].Trim();
                    i = -1;
                    if (t != "") {
                        try {
                            switch (t[t.Length - 1]) {
                            case 'm':
                            case 'M':
                                i = (int)double.Parse(t.Substring(0, t.Length - 1), AppContents.ni);
                                break;
                            case 't':
                            case 'T':
                                // ft feet
                                i = (int)(double.Parse(t.Substring(0, t.Length - 2), AppContents.ni) * ConvertionFactors.FEET2METER);
                                break;
                            default:
                                i = -1;
                                break;
                            }                    
                        }
                        catch (Exception) {
                            i = -1;
                        }
                    }

                    wp = new WayPoint(lat, lon, tokens[0]);
                    wp.Elevation = i;

                    switch ((SeeYouFlags)int.Parse(tokens[6].Trim())) {
                    case SeeYouFlags.Unknown:
                        wp.type = WayPointTypeId.Unknown;
                        break;
                    case SeeYouFlags.Waypoint:
                        wp.type = WayPointTypeId.Unknown;
                        break;
                    case SeeYouFlags.Pass:
                    case SeeYouFlags.Mountain:
                        wp.type = WayPointTypeId.Landmark;
                        break;
                    case SeeYouFlags.Outlanding:
                        wp.type = WayPointTypeId.Outlanding;
                        wp.landable = true;
                        break;
                    case SeeYouFlags.Glidersite:
                        wp.type = WayPointTypeId.Glidersite;
                        // assume grass
                        wp.surface = SurfaceTypeId.Grass;
                        wp.landable = true;
                        break;
                    case SeeYouFlags.AirportHard:
                        wp.type = WayPointTypeId.Airport;
                        wp.landable = true;
                        wp.surface = SurfaceTypeId.Hard;
                        break;
                    case SeeYouFlags.AirportGrass:
                        wp.type = WayPointTypeId.Airport;
                        wp.landable = true;
                        wp.surface = SurfaceTypeId.Grass;
                        break;
                    }

                    t = tokens[7].Trim();
                    if (t != "") {
                        try{
                            wp.runway = string.Format("{0:00}", int.Parse(t, AppContents.ni) / 10);
                        }
                        catch (Exception) {
                            wp.runway = "";
                        }
                    }

                    t = tokens[8].Trim();
                    if (t != "") {
                        try {
                            switch (t[t.Length - 1]) {
                            case 'M':
                            case 'm':
                                if (t[t.Length - 2] == 'n' || t[t.Length - 2] == 'N') {
                                    // nautic miles
                                    i = (int)(double.Parse(t.Substring(0, t.Length - 2), AppContents.ni) * ConvertionFactors.NAUTICMILES2METERS);
                                }
                                else {
                                    // meters
                                    i = (int)double.Parse(t.Substring(0, t.Length - 1), AppContents.ni);
                                }
                                break;
                            case 'L':
                            case 'l':
                                // nl statute miles
                                i = (int)(double.Parse(t.Substring(0, t.Length - 2), AppContents.ni) * ConvertionFactors.STATUTEMILES2METERS);
                                break;
                            default:
                                i = -1;
                                break;
                            }
                        }
                        catch (Exception) {
                            i = -1;
                        }
                        if (i == 0) {
                            i = -1;
                        }
                        wp.length = i;
                    }

                    if (wp.length == -1) {
                        wp.runway = "";
                    }

                    wp.frequency = tokens[9].Trim();
                    wp.description = tokens[10].Trim();

                    t = tokens[2].Trim();
                    if (t != "") {
                        wp.country = (CountryTypeId) (AppContents.countryTypesByCode.ContainsKey(t) ? AppContents.countryTypesByCode[t] : CountryTypeId.Unknown);
                    }
                }
            }
            catch (Exception e) {
                if (MessageBox.Show("An exception occured while reading file (SeeYou)\n\nThe fault line is:\n\n" +
                    line + "\n\n" + e.StackTrace + "\n\n" + e.Message, "Error", MessageBoxButtons.OKCancel, MessageBoxIcon.Error) == DialogResult.Cancel) {
                    throw new Exception("Abort");
                }
                wp = null;
            }
            return wp;
        }
    
        public override string BuildLine(WayPoint wp) {
            bool isNeg;
            double tLat, tLon;
            long tmp;
            string latChar, lonChar;

            isNeg = wp.Latitude < 0;
            tmp = Math.Abs(wp.Latitude);
            tLat = (tmp / 360000) * 100;
            tLat += (tmp % 360000.0) / 6000.0;

            if (isNeg) {
                latChar = "N";
            }
            else  {
                latChar = "S";
            }

            isNeg = wp.Longitude < 0;
            tmp = Math.Abs(wp.Longitude);
            tLon = (tmp / 360000) * 100;
            tLon += (tmp % 360000.0) / 6000.0;

            if (isNeg) {
                lonChar = "W";
            }
            else  {
                lonChar = "E";
            }

            switch (wp.type) {
            case WayPointTypeId.Airport:
                if (wp.surface == SurfaceTypeId.Hard) {
                    tmp = (int)SeeYouFlags.AirportHard;
                }
                else {
                    tmp = (int)SeeYouFlags.AirportGrass;
                }
                break;
            case WayPointTypeId.Glidersite:
                tmp = (int)SeeYouFlags.Glidersite;
                break;
            case WayPointTypeId.Outlanding:
                tmp = (int)SeeYouFlags.Outlanding;
                break;
            case WayPointTypeId.Landmark:
                tmp = (int)SeeYouFlags.Mountain;
                break;
            default:
                tmp = (int)SeeYouFlags.Unknown;
                break;
            }

            // name,code,country,lat,lon,elev,style,rwdir,rwlen,freq,desc
            return string.Format(AppContents.ni, "\"{0}\",{1},{2},{3:0000.000}{4},{5:00000.000}{6},{7},{8},{9},{10},{11},\"{12}\"", wp.longName, wp.shortName, AppContents.countryTypesById[wp.country],
                tLat, latChar, tLon, lonChar, wp.Elevation != -1 ? wp.Elevation.ToString() + ".0m" : "", tmp, wp.runway != "" ? (int.Parse(wp.runway) * 10).ToString() : "", wp.length != -1 ? wp.length.ToString() + ".0m" : "", wp.frequency, wp.description);
        }
    }
}
