// name   : SoaringPilot.cs
// author : Harald Maier
// date   : 25.11.2003
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.Collections;
using System.Text;
using System.Threading;
using System.Windows.Forms;
using System.IO;

using SoaringDotNET.Data;
using SoaringDotNET.GUI;
using SoaringDotNET.FileFormats;
using SerialPorts;

namespace SoaringDotNET.Recorders
{
	/// <summary>
	/// 
	/// </summary>
	public enum SoaringPilotFlags {Airport = 1, Turnpoint = 2, Outlanding = 4, Start = 8, Finish = 16, Landmark = 32,
        Landable = 128, Area = 256, Thermal = 512}

	public class SoaringPilot : SoaringDotNET.Recorders.SerialConnectionRecorder
	{
		public SoaringPilot()
		{
			// 
			// TODO: Add constructor logic here
			//
            options = RecorderOptions.DownloadFlights |
                RecorderOptions.DownloadTasks |
                RecorderOptions.DownloadWaypoints |
                RecorderOptions.UploadTask |
                RecorderOptions.UploadWaypoints |
                RecorderOptions.UploadSUA |
                RecorderOptions.ExportToFile;
            
            capacity.maxNrTasks = (int)CapacityNumbers.Unlimited;
            capacity.maxNrWaypoints = (int)CapacityNumbers.Unlimited;
            capacity.maxNrWaypointsPerTask = (int)CapacityNumbers.Unlimited;

            name = "Soaring Pilot";
            ManufacurerCode = "XSP";
            ManufacurerLetter = "X";
            supportedLineSpeeds = new object[] {2400, 4800, 9600, 19200, 38400};
            preferedLineSpeed = 19200;
		}

        private ArrayList ReadFile(string fileName) {
            return RedirectToFile ? ReadFileFromDisk(fileName) : ReadFileFromPort();
        }

        private void WriteFile(ArrayList file, string fileName) {
            if (RedirectToFile) {
                WriteFileToDisk(file, fileName);
            }
            else {
                WriteFileToPort(file);
            }
        }

        protected void WriteFileToPort(ArrayList file) {
            comPort.Flush();
            foreach (string s in file) {
                if (IsInterrupted) {
                    break;
                }
                comPort.Send(s + "\r\n");
            }
        }

        protected ArrayList ReadFileFromPort() {
            bool start = false;
            string s = "";
            byte []b = new byte[1];
            ArrayList file = new ArrayList();
            comPort.Flush();
            long t = DateTime.Now.Ticks;

            while (!IsInterrupted) {                
                if (comPort.Recv(b, 1) > 0) {
                    start = true;
                    t = DateTime.Now.Ticks;
                    switch((char)b[0]) {
                    case '\n':
                        file.Add(string.Copy(s));
                        s = "";
                        break;
                    case '\r':
                        break;
                    default:
                        s += Encoding.ASCII.GetString(b);
                        break;
                    }
                }
                else if (start && DateTime.Now.Ticks - t > 50000000) {
                    // assume no more data
                    if (s.Length > 0) {
                        file.Add(string.Copy(s));
                    }
                    break;
                }
                else if (DateTime.Now.Ticks - t > 50000000) {
                    throw new Exception("No response from recorder within 5 secs.!");
                }
            }

            if (IsInterrupted) {
                file.Clear();
            }

            return file;
        }

        protected ArrayList ReadFileFromDisk(string fileName) {
            // TODO:  Add SoaringPilotFile.ReadFile implementation
            string line;
            ArrayList file = new ArrayList();
            OpenFileDialog fd = new OpenFileDialog();
            fd.CheckFileExists = true;
            fd.Filter = "All files (*.*)|*.*|Waypoints (*.dat)|*.dat|Tasks (*.spt|*.spt|SUA (*.sua)|*.sua";
            fd.FilterIndex = 0;
            fd.FileName = fileName;

            if (fd.ShowDialog() == DialogResult.OK) {
                try {
                    using (StreamReader inStream = new StreamReader(fd.FileName)) {
                        while ((line = inStream.ReadLine()) != null) {
                            file.Add(line);
                        }
                    }
                }
                catch (Exception e) {
                    MessageBox.Show(e.ToString(), "Read Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
            return file;
        }
    
        protected void WriteFileToDisk(ArrayList file, string fileName) {
            // TODO:  Add SoaringPilotFile.WriteFile implementation
            SaveFileDialog fd = new SaveFileDialog();
            fd.CheckPathExists = true;
            fd.OverwritePrompt = false;
            fd.ValidateNames = true;
            fd.FileName = fileName;
            fd.Filter = "All files (*.*)|*.*|Waypoints (*.dat)|*.dat|Tasks (*.spt|*.spt|SUA (*.sua)|*.sua";
            fd.FilterIndex = 0;
            StreamWriter outStream = null;
            if (fd.ShowDialog() == DialogResult.OK) {
                try {
                    outStream = new StreamWriter(fd.FileName, false);
                    foreach (string line in file) {
                        outStream.WriteLine(line);
                    }
                }
                catch (Exception e) {
                    Console.WriteLine(e.ToString());
                    MessageBox.Show(e.ToString(), "Write Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
                finally {
                    if (outStream != null) {
                        outStream.Close();
                    }
                }
            }
        }
    
        protected override void GetSerialNo() {
            // TODO:  Add SoaringPilot.GetSerialNoThread implementation
            serialNo = 0;
            if (RecorderBase.SerialNoThreadNotify != null) {
                RecorderBase.SerialNoThreadNotify(true);
            }
        }
    
        protected override void DownloadWaypoints() {
            // TODO:  Add SoaringPilot.DownloadWaypoints implementation
            currentCatalogue = new WaypointCatalog("SoaringPilot" + SerialNo);
            FileHandlerBase fh = null;
            bool success = false;
            string [] tokens;
            WayPoint wp;

            IsInterrupted = false;
            try {
                ArrayList file = ReadFile("waypoints.cup");
                success = file.Count > 1;
                //twiddle out the file structure, cai or seeyou
                foreach (string line in file) {
                    tokens = line.Split(',');
                    if (tokens.Length >= 11) {
                        fh = new SeeYouFileHandler();
                        break;
                    }
                    else if (tokens.Length >= 6) {
                        fh = new CAIAndWinPilotFileHandler();
                        break;
                    }
                }
                if (fh == null) {
                    throw new Exception("Unknown file structure");
                }

                foreach (string line in file) {
                    if ((wp = fh.ParseLine(line)) != null) {
                        currentCatalogue.Add(wp);
                    }
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.Message, "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                success = false;
            }
            finally {
                if (RecorderBase.DownloadWaypointsThreadNotify != null) {
                    RecorderBase.DownloadWaypointsThreadNotify(success);
                }
            }
        }
    
        protected override void UploadWaypoints() {
            // TODO:  Add SoaringPilot.UploadWaypoints implementation
            SeeYouFileHandler fh = new SeeYouFileHandler();
            bool success = true;

            IsInterrupted = false;
            try {
                if (currentCatalogue != null) {
                    ArrayList file = new ArrayList(currentCatalogue.Count);

                    foreach (WayPoint w in currentCatalogue.VisibleWaypoints) {
                        file.Add(fh.BuildLine(w));
                    }

                    WriteFile(file, "waypoints.cup");
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                success = false;
            }
            finally {
                if (RecorderBase.UploadWaypointsThreadNotify != null) {
                    RecorderBase.UploadWaypointsThreadNotify(success);
                }
            }
        }
    
        protected override void DownloadTasks() {
            // TODO:  Add SoaringPilot.DownloadTasks implementation
            bool success = true;
            bool takeoff = false, landing = false;
            int nrWayointsInTask = 0, nrWaypointsRead = 0;
            int lat = 0, lon = 0;
            int elev = 0;
            SoaringPilotFlags type;

            string [] tokens;
            string [] coordItems;
            Task task = null;
            WayPoint wp;
            string t = "";
            char c;

            IsInterrupted = false;
            try {
                ArrayList file = ReadFile("tasks.spt");
                foreach (string line in file) {
                    tokens = line.Split(',');
                    if (tokens.Length > 0) {
                        switch (tokens[0]) {
                        case "TS":
                            nrWayointsInTask = int.Parse(tokens[2]);
                            nrWaypointsRead = 0;
                            if (tokens.Length > 3) {
                                takeoff = (tokens[3].IndexOf('T') != -1);
                                landing = (tokens[3].IndexOf('L') != -1);
                            }
                            else {
                                takeoff = landing = false;
                            }
                            task = new Task(tokens[1]);
                            break;
                        case "TW":
                            // lat
                            coordItems = tokens[1].Split(':');
                            switch(coordItems.Length) {
                            case 2:
                                // 48:23.083N format
                                t = coordItems[1];
                                lat = (int.Parse(coordItems[0]) * 360000) + (int)(double.Parse(t.Substring(0, t.Length - 1), AppContents.ni) * 6000.0);
                                break;
                            case 3:
                                // 48:00:42.00N format
                                t = coordItems[2];
                                lat = (int.Parse(coordItems[0]) * 360000) + (int.Parse(coordItems[1]) * 6000) + (int)(double.Parse(t.Substring(0, t.Length - 1), AppContents.ni) * 100);
                                break;
                            }

                            c = t[t.Length - 1];
                            if (c == 'N' || c == 'n') {
                                lat = -lat;
                            }

                            // lon
                            coordItems = tokens[2].Split(':');
                            switch(coordItems.Length) {
                            case 2:
                                // 48:23.083N format
                                t = coordItems[1];
                                lon = (int.Parse(coordItems[0]) * 360000) + (int)(double.Parse(t.Substring(0, t.Length - 1),  AppContents.ni) * 6000.0);
                                break;
                            case 3:
                                // 48:00:42.00N format
                                t = coordItems[2];
                                lon = (int.Parse(coordItems[0]) * 360000) + (int.Parse(coordItems[1]) * 6000) + (int)(double.Parse(t.Substring(0, t.Length - 1), AppContents.ni) * 100);
                                break;
                            }
                            
                            c = t[t.Length - 1];
                            if (c == 'W' || c == 'w') {
                                lon = -lon;
                            }

                            // type
                            try {
                                type = (SoaringPilotFlags)int.Parse(tokens[3]);
                            }
                            catch {
                                type = SoaringPilotFlags.Turnpoint;
                            }

                            // elevation
                            try {
                                t = tokens[4];
                                c = t[t.Length - 1];
                                elev = int.Parse(t.Substring(0, t.Length - 1));
                            }
                            catch {
                                elev = 0;
                                c = 'm';
                            }

                            if (c != 'M' && c != 'm') {
                                if (c == 'F' || c == 'f') {
                                    elev = (int)((double)elev * ConvertionFactors.FEET2METER);
                                }
                                else {
                                    throw new Exception("Unknown elevation unit " + c + " in line\n" + line);
                                }
                            }

                            if (tokens.Length > 6) {
                                wp = new WayPoint(lat, lon, tokens[6].Trim());
                                wp.shortName = tokens[5].Trim();
                            }
                            else {
                                wp = new WayPoint(lat, lon, tokens[5].Trim());
                            }
                            wp.Elevation = elev;
                            task.Add(wp);
                            nrWaypointsRead++;

                            if ((type & SoaringPilotFlags.Area) == SoaringPilotFlags.Area && tokens.Length > 7) {
                                // read area info
                                // 005998005329101202
                                // |     |     |  |
                                t = tokens[7];
                                SectorDefinition sector = task.GetSector(task.Count - 1);
                                sector.sectorType = SectorTypes.Area;
                                try {
                                    // round to 100m
                                    sector.radius1 = (int)Math.Round(double.Parse(t.Substring(0, 6)) * ConvertionFactors.NAUTICMILES2METERS / 1000 / 100) * 100;
                                    sector.radius2 = (int)Math.Round(double.Parse(t.Substring(6, 6)) * ConvertionFactors.NAUTICMILES2METERS / 1000 / 100) * 100;
                                    sector.directionFrom = int.Parse(t.Substring(12, 3));
                                    sector.directionTo = int.Parse(t.Substring(15, 3));
                                }
                                catch {
                                    task.SetSector(task.Count - 1, new SectorDefinition(AppContents.GetDefaultSector(TurnpointTypes.Turnpoint)));
                                }
                            }
                            break;
                        case "TE":
                            if (nrWayointsInTask != nrWaypointsRead) {
                                throw new Exception("Invalid task definition!\n" + nrWayointsInTask.ToString() + " waypoints defined, " +
                                    nrWaypointsRead.ToString() + " found!");
                            }
                            else {
                                // append takeoff and landing, if not in task
                                if (!takeoff) {
                                    task.Insert(1, task[0]);
                                }

                                if (!landing) {
                                    task.Add(task[task.Count - 1]);
                                }
                                tasks.Add(task);
                            }
                            break;
                        }
                    }
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                success = false;
            }
            finally {
                if (RecorderBase.DownloadTasksThreadNotify!= null) {
                    RecorderBase.DownloadTasksThreadNotify(success);
                }
            }
        }
    
        protected override void UploadTasks() {
            // TODO:  Add SoaringPilot.UploadTasks implementation
            //    ** -------------------------------------------------------------
            //    **      SOARINGPILOT Version 1.9.7 Tasks
            //    **      Date: 13 Feb 2004
            //    ** -------------------------------------------------------------
            //    TS,TASK002,6,TL
            //    TW,48:00:42.00N,010:06:07.00E,2,1903F,Tannhm,Tannhm
            //    TW,48:00:42.00N,010:06:07.00E,2,1903F,Tannhm,Tannhm
            //    TW,48:06:43.00N,009:45:49.00E,2,1903F,Bibrch,Bibrch
            //    TW,48:14:10.00N,010:08:19.00E,2,1680F,Illrts,Illrts,Area declaration
            //    TW,48:00:42.00N,010:06:07.00E,2,1903F,Tannhm,Tannhm
            //    TW,48:00:42.00N,010:06:07.00E,2,1903F,Tannhm,Tannhm
            //    TE
            bool success = false;
            string lat, lon, area;
            int tmp, i;
            SoaringPilotFlags type;
            WayPoint w;
            SectorDefinition sector;
            ArrayList file = new ArrayList();
            IsInterrupted = false;
            try {
                foreach (Task t in tasks) {
                    file.Add(string.Format("TS,{0},{1},{2}", t.Name, t.Count, t.Count > 3 ? "TL" : ""));
                    for (i = 0; i < t.Count; i++){
                        w = t[i];
                        sector = t.GetSector(i);
                        tmp = Math.Abs(w.Latitude);
                        lat = string.Format("{0:00}:{1:00}:{2:00}{3}", tmp / 360000, (tmp % 360000) / 6000, (tmp % 6000) / 100, w.Latitude <= 0 ? "N" : "S");
                        tmp = Math.Abs(w.Longitude);
                        lon = string.Format("{0:000}:{1:00}:{2:00}{3}", tmp / 360000, (tmp % 360000) / 6000, (tmp % 6000) / 100, w.Longitude < 0 ? "W" : "E");

                        if (sector.sectorType == SectorTypes.Area) {
                            // write area info
                            // 005998005329101202
                            // |     |     |  |
                            type = SoaringPilotFlags.Area;
                            area = string.Format(",{0:000000}{1:000000}{2:000}{3:000}", (int)(sector.radius1 / ConvertionFactors.NAUTICMILES2METERS * 1000),
                                (int)(sector.radius2 / ConvertionFactors.NAUTICMILES2METERS * 1000), sector.directionFrom, sector.directionTo);
                        }
                        else {
                            area = "";
                            switch (w.type) {
                            case WayPointTypeId.Airport:
                            case WayPointTypeId.Glidersite:
                                type = SoaringPilotFlags.Airport;
                                break;
                            case WayPointTypeId.Landmark:
                                type = SoaringPilotFlags.Landmark;
                                break;
                            case WayPointTypeId.Outlanding:
                                type = SoaringPilotFlags.Outlanding;
                                break;
                            default:
                                type = SoaringPilotFlags.Turnpoint;
                                break;
                            }

                            if (w.landable) {
                                type |= SoaringPilotFlags.Landable;
                            }
                        }
                        file.Add(string.Format("TW,{0},{1},{2},{3},{4},{5}{6}", lat, lon, (int)type, w.Elevation != -1 ? w.Elevation.ToString() + "M" : "", w.shortName, w.longName, area));
                    }
                    file.Add("TE");
                }
                WriteFile(file, "tasks.spt");
                success = true;
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                success = false;
            }
            finally {
                if (RecorderBase.UploadTasksThreadNotify != null) {
                    RecorderBase.UploadTasksThreadNotify(success);
                }
            }
        }
    
        protected override void UploadDeclaration() {
            if (RecorderBase.UploadDeclarationThreadNotify != null) {
                RecorderBase.UploadDeclarationThreadNotify(true);
            }
        }
    
        protected override void UploadPilots() {
            // TODO:  Add Volkslogger.UploadSUAs implementation
            if (RecorderBase.UploadPilotsThreadNotify != null) {
                RecorderBase.UploadPilotsThreadNotify(true);
            }
        }

        protected override void DownloadPilots() {
            // TODO:  Add Volkslogger.UploadSUAs implementation
            if (RecorderBase.DownloadPilotsThreadNotify != null) {
                RecorderBase.DownloadPilotsThreadNotify(true);
            }
        }

        protected override void GetFlightDir() {
            // TODO:  Add SoaringPilot.GetFlightDir implementation
            bool success = true;
            IsInterrupted = false;
            DirectoryEntry de = null;
            bool firstFix = true;
            TimeSpan lastFix = new TimeSpan(0);

            try {
                ArrayList file = ReadFileFromPort();
                foreach (string line in file) {
                    switch (line[0]) {
                    case 'B':
                        if (firstFix) {
                            de.firstFix += new TimeSpan(int.Parse(line.Substring(1, 2)), int.Parse(line.Substring(3, 2)), int.Parse(line.Substring(5, 2)));
                            firstFix = false;
                        }
                        else {
                            lastFix = new TimeSpan(int.Parse(line.Substring(1, 2)), int.Parse(line.Substring(3, 2)), int.Parse(line.Substring(5, 2)));
                        }
                        goto case 'E';
                    case 'E':
                    case 'F':
                    case 'L':
                        de.igcFile.BRecord.Add(line);
                        break;
                    case 'A':
                        if (de != null && de.firstFix.Ticks != 0) {
                            de.lastFix += lastFix;
                            flightDirectory.Add(de);
                            firstFix = true;
                        }
                        de = new DirectoryEntry();
                        de.igcFile.A = line;
                        break;
                    case 'H':
                        switch (line.Substring(2, 3)){
                        case "DTE":
                            de.igcFile.DTE = line;
                            try {
                                de.firstFix = new DateTime(int.Parse(line.Substring(9, 2)) + 2000, int.Parse(line.Substring(7, 2)), int.Parse(line.Substring(5, 2)));
                                de.lastFix = new DateTime(int.Parse(line.Substring(9, 2)) + 2000, int.Parse(line.Substring(7, 2)), int.Parse(line.Substring(5, 2)));
                            }
                            catch (Exception) {
                                de.firstFix = new DateTime(0);
                                de.lastFix = new DateTime(0);
                            }
                            break;
                        case "FXA":
                            de.igcFile.FXA = line;
                            break;
                        case "PLT":
                            de.igcFile.PLT = line;
                            de.pilot = line.Substring(line.IndexOf(':') + 1).Trim();
                            break;
                        case "GTY":
                            de.igcFile.GTY = line;
                            break;
                        case "GID":
                            de.igcFile.GID = line;
                            de.gliderId = line.Substring(line.IndexOf(':') + 1).Trim();
                            break;
                        case "DTM":
                            de.igcFile.DTM = line;
                            break;
                        case "FTY":
                            de.igcFile.FTY = line;
                            break;
                        case "RFW":
                            de.igcFile.RFW = line;
                            break;
                        case "RHW":
                            de.igcFile.RHW = line;
                            break;
                        case "GPS":
                            de.igcFile.GPS = line;
                            break;
                        case "CID":
                            de.igcFile.CID = line;
                            de.competitionId = line.Substring(line.IndexOf(':') + 1).Trim();
                            break;
                        case "CCL":
                            de.igcFile.CCL = line;
                            break;
                        case "SIT":
                            de.igcFile.SIT = line;
                            break;
                        case "TZN":
                            de.igcFile.TZN = line;
                            break;
                        }
                        break;
                    case 'I':
                        de.igcFile.I = line;
                        break;
                    case 'C':
                        de.igcFile.CRecord.Add(line);
                        break;
                    }
                }
                if (de != null && de.firstFix.Ticks != 0) {
                    de.lastFix += lastFix;
                    flightDirectory.Add(de);
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                flightDirectory.Clear();
                success = false;
            }
            finally {
                int flightOfDay, oldDay, oldMonth, oldYear;
                string sNo;

                flightOfDay = 1;
                oldDay = oldMonth = oldYear = 0;
                foreach (DirectoryEntry de2 in flightDirectory) {
                    if (oldDay != de2.firstFix.Day || oldMonth != de2.firstFix.Month || oldYear != de2.firstFix.Year) {
                        flightOfDay = 1;
                        oldDay = de2.firstFix.Day;
                        oldMonth = de2.firstFix.Month;
                        oldYear = de2.firstFix.Year;
                    }
                    else {
                        flightOfDay++;
                    }
                    // serial no
                    sNo = de2.igcFile.A.Substring(4, 3);
                    de2.fileName = string.Format("{0}{1}{2}{3}{4}{5}.IGC", 
                        c36[de2.firstFix.Year % 10],
                        c36[de2.firstFix.Month],
                        c36[de2.firstFix.Day],
                        ManufacurerLetter, sNo, flightOfDay < 36 ? c36[flightOfDay] : '_');
                }
                if (RecorderBase.FlightDirectoryThreadNotify != null) {
                    RecorderBase.FlightDirectoryThreadNotify(success);
                }
            }
        }
    
        protected override void DownloadFlights() {
            // TODO:  Add SoaringPilot.DownloadFlights implementation
            if (RecorderBase.DownloadFlightsThreadNotify != null) {
                RecorderBase.DownloadFlightsThreadNotify(true);
            }
        }
    
        protected override void UploadSUAs() {
            // TODO:  Add SoaringPilot.UploadSUAs implementation
            bool success = false;
            ArrayList file;
            IsInterrupted = false;
            SUAFileHandlerTimNewport fileHandler = new SUAFileHandlerTimNewport();
            try {
                file = fileHandler.WriteFile(suas);
                WriteFile(file, "suadata.sua");
                success = true;
            }
            catch (Exception e) {
                MessageBox.Show(e.ToString(), "Transfer Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                success = false;
            }
            finally {
                if (RecorderBase.UploadSUAsThreadNotify != null) {
                    RecorderBase.UploadSUAsThreadNotify(success);
                }
            }
        }

        public override string Description {
            get {
                // TODO:  Add SoaringPilot.Description getter implementation
                return string.Format(@"Adapter for SoaringPilot PDA Software

This is a serial connection adapter. The prefered line speed is {0} baud. If you ran into trouble with incomplete transfers, then try at a lower speed.

If your device doesn't support serial connection then select export to file and transfer them via expansion card.

You can download the waypoints either in Cambridge or in SeeYou format. The latter may contain more details.
Upload is in general in SeeYou format.

Download procedure  : First start download in SoaringDotNet then press the transmit button in SoaringPilot.
Upload procedure    : First press receive button in SoaringPilot then start upload in SoaringDotNet
", preferedLineSpeed) + base.OptionsString;
            }
        }        
    
        public override bool Open(string port, LineSpeed speed) {
            // TODO:  Add SoaringPilot.Open implementation
            comPort.Cnfg.SetFlowControl();
            return base.Open(port, speed);
        }
    }
}
