// name   : AirspaceElement.cs
// author : Harald Maier
// date   : 23.01.2004
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.Collections;
using System.Drawing;
using System.Drawing.Drawing2D;

using SoaringDotNET.Data;

namespace SoaringDotNET.GUI
{
	/// <summary>
	/// 
	/// </summary>
	public abstract class AirspaceElement : SoaringDotNET.GUI.MapAreaElement
	{
        public Region region;
        public Airspace airspace;

        public AirspaceElement(Airspace space) :
            base(MapElementTypes.Airspace) {
			// 
			// TODO: Add constructor logic here
			//
            airspace = space;
            pen = new Pen(Brushes.Black, 1);
		}

        public new AirspaceSegment this[int pos] {
            get {
                return (AirspaceSegment)airspace.points[pos];
            }
        }
    
        public new IEnumerator GetEnumerator() {
            return airspace.points.GetEnumerator();
        }

        public new int Count {
            get {
                return airspace.points.Count;
            }
        }

        public AirspaceTypes AirspaceType {
            get {
                return airspace.airspaceType;
            }
        }

        public AirspaceClasses AirspaceClass {
            get {
                return airspace.airspaceClass;
            }
        }

        public string AirspaceTypeString {
            get {
                return AppContents.airspaceTypes.getIdType((int)airspace.airspaceType).text;
            }
        }

        public string AirspaceClassString {
            get {
                return AppContents.airspaceClasses.getIdType((int)airspace.airspaceClass).text;
            }
        }

        public override void Draw(Graphics g, WgsArea view, int width, int height) {
            // TODO:  Add AirspaceElement.Draw implementation
            long lat, lat2;
            long lon, lon2;
            long latRange, lonRange;
            long minLat, minLon;
            long diffLat, diffLon;
            long diff;
            float radius;
            float startAngle, sweepAngle;
            double dist;

            minLat = view.TopLat;
            latRange = view.Height;
            minLon = view.LeftLon;
            lonRange = view.Width;

            try {
                if (view.AreaIntersect(Area) && !Area.IsEmpty) {
                    GraphicsPath gp = new GraphicsPath();
                    foreach (AirspaceSegment s in this) {
                        switch (s.type) {
                        case AirspaceSegmentTypes.Line:
                            lat = (s.latFrom - minLat) * height / latRange;
                            diff = s.lonFrom - minLon;
                            if (diff <= -64800000) {
                                diff += 129600000;
                            }
                            lon = diff * width / lonRange;

                            lat2 = (s.latTo - minLat) * height / latRange;
                            diff = s.lonTo - minLon;
                            if (diff <= -64800000) {
                                diff += 129600000;
                            }
                            lon2 = diff * width / lonRange;

                            gp.AddLine(lon, lat, lon2, lat2);
                            break;
                        case AirspaceSegmentTypes.Circle:
                            radius = (s.radius / 60.0f) * 360000.0f * height / latRange;
                            lat = (s.latCenter - minLat) * height / latRange;
                            diff = s.lonCenter - minLon;
                            if (diff <= -64800000) {
                                diff += 129600000;
                            }
                            lon = diff * width / lonRange;
                            gp.AddEllipse(lon - radius, lat - radius, 2.0f * radius, 2.0f * radius);
                            break;
                        case AirspaceSegmentTypes.ArcClock:
                        case AirspaceSegmentTypes.ArcCounterClock:
                            // calculate center and offset for upper left corner of arc rectangle
                            radius = (float)Math.Round(((s.radius / 60.0) * 360000.0) * height / latRange);
                            lat = (s.latCenter - minLat) * height / latRange;
                            diff = s.lonCenter - minLon;
                            if (diff <= -64800000) {
                                diff += 129600000;
                            }
                            lon = diff * width / lonRange;

                            // calc pos for start point and angle
                            lat2 = (s.latFrom - minLat) * height / latRange;
                            diff = s.lonFrom - minLon;
                            if (diff <= -64800000) {
                                diff += 129600000;
                            }
                            lon2 = diff * width / lonRange;

                            diffLat = lat2 - lat;
                            diffLon = lon2 - lon;
                            dist = Math.Sqrt(Math.Pow(diffLon, 2) + Math.Pow(diffLat, 2));
                            if (diffLat < 0) {
                                startAngle = (float)(Math.Asin(diffLon / dist) * 180.0 / Math.PI) + 270.0f;
                            }
                            else {
                                startAngle = 90.0f - (float)(Math.Asin(diffLon / dist) * 180.0 / Math.PI);
                            }

                            // calc pos for dest point and angle
                            lat2 = (s.latTo - minLat) * height / latRange;
                            diff = s.lonTo - minLon;
                            if (diff <= -64800000) {
                                diff += 129600000;
                            }
                            lon2 = diff * width / lonRange;

                            diffLat = lat2 - lat;
                            diffLon = lon2 - lon;
                            dist = Math.Sqrt(Math.Pow(diffLon, 2) + Math.Pow(diffLat, 2));
                            if (diffLat < 0) {
                                sweepAngle = (float)(Math.Asin(diffLon / dist) * 180.0 / Math.PI) + 270.0f;
                            }
                            else {
                                sweepAngle = 90.0f - (float)(Math.Asin(diffLon / dist) * 180.0 / Math.PI);
                            }

                            if (s.type == AirspaceSegmentTypes.ArcClock) {
                                if (sweepAngle < startAngle) {
                                    sweepAngle = 360.0f - startAngle + sweepAngle;
                                }
                                else {
                                    sweepAngle -= startAngle;
                                }
                            }
                            else if (s.type == AirspaceSegmentTypes.ArcCounterClock) {
                                if (sweepAngle > startAngle)
                                    sweepAngle = -(360.0f - sweepAngle + startAngle);
                                else {
                                    sweepAngle -= startAngle;
                                }
                            }

                            if (radius > 0.0) {
                                gp.AddArc(lon - radius, lat - radius, 2 * radius, 2 * radius, startAngle, sweepAngle);
                            }
                            break;
                        }
                    }

                    region = new Region(gp);
                    g.FillRegion(this.brush, region);
                    g.DrawPath(this.pen, gp);
                }
                else {
                    region = null;
                }
            }
            catch {
                region = null;
            }
        }
    
        public new void CalcArea() {
            // TODO:  Add AirspaceElement.CalcArea implementation
            area.Clear();
            int radius;
            WgsPoint center;
            int from;
            int to;
            int w, ww;

            foreach (AirspaceSegment s in airspace) {
                if (area.IsEmpty) {
                    switch (s.type) {
                    case AirspaceSegmentTypes.ArcClock:
                        radius = (int)(Math.Ceiling((s.radius / 60.0f) * 360000.0f));// / Math.Cos(WgsPoint.InternalToRad(s.latCenter)));
                        center = new WgsPoint(s.latCenter, s.lonCenter);
                        from = (int)Math.Floor(center.bearingDeg(new WgsPoint(s.latFrom, s.lonFrom)));
                        to = (int)Math.Ceiling(center.bearingDeg(new WgsPoint(s.latTo, s.lonTo)));
                        if (to < from) {
                            to += 360;
                        }
                        for (w = from; w < to; w += 90) {
                            ww = w % 360;
                            if (ww <= 90 && to >= 90) {
                                area.RightLon = s.lonCenter + radius;
                            }
                            else if (ww <= 180 && to >= 180) {
                                area.BottomLat = s.latCenter + radius;
                            }
                            else if (ww <= 270 && to >= 270) {
                                area.LeftLon = s.lonCenter - radius;
                            }
                            else if (ww <= 360 && to >= 360) {
                                area.TopLat = s.latCenter - radius;
                            }
                        }
                        if (area.BottomLat != 0) {
                            area.BottomLat = Math.Max(Math.Max(s.latFrom, s.latTo), area.BottomLat);
                        }
                        else {
                            area.BottomLat = Math.Max(s.latFrom, s.latTo);
                        }
                        if (area.TopLat != 0) {
                            area.TopLat = Math.Min(Math.Min(s.latFrom, s.latTo), area.TopLat);
                        }
                        else {
                            area.TopLat = Math.Min(s.latFrom, s.latTo);
                        }
                        if (area.LeftLon != 0) {
                            area.LeftLon = Math.Min(Math.Min(s.lonFrom, s.lonTo), area.LeftLon);
                        }
                        else {
                            area.LeftLon = Math.Min(s.lonFrom, s.lonTo);
                        }
                        if (area.RightLon != 0) {
                            area.RightLon = Math.Max(Math.Max(s.lonFrom, s.lonTo), area.RightLon);
                        }
                        else {
                            area.RightLon = Math.Max(s.lonFrom, s.lonTo);
                        }
                        break;
                    case AirspaceSegmentTypes.ArcCounterClock:
                        radius = (int)(Math.Ceiling((s.radius / 60.0f) * 360000.0f));// / Math.Cos(WgsPoint.InternalToRad(s.latCenter)));
                        center = new WgsPoint(s.latCenter, s.lonCenter);
                        from = (int)Math.Ceiling(center.bearingDeg(new WgsPoint(s.latFrom, s.lonFrom)));
                        to = (int)Math.Floor(center.bearingDeg(new WgsPoint(s.latTo, s.lonTo)));
                        if (from < to) {
                            from += 360;
                        }
                        for (w = from; w > to; w -= 90) {
                            ww = w % 360;
                            if (ww <= 90 && to >= 270) {
                                area.TopLat = s.latCenter - radius;
                            }
                            else if (ww <= 180 && to <= 90) {
                                area.RightLon = s.lonCenter + radius;
                            }
                            else if (ww <= 270 && to <= 180) {
                                area.BottomLat = s.latCenter + radius;
                            }
                            else if (ww <= 360 && to <= 270) {
                                area.LeftLon = s.lonCenter - radius;                                
                            }
                        }
                        if (area.BottomLat != 0) {
                            area.BottomLat = Math.Max(Math.Max(s.latFrom, s.latTo), area.BottomLat);
                        }
                        else {
                            area.BottomLat = Math.Max(s.latFrom, s.latTo);
                        }
                        if (area.TopLat != 0) {
                            area.TopLat = Math.Min(Math.Min(s.latFrom, s.latTo), area.TopLat);
                        }
                        else {
                            area.TopLat = Math.Min(s.latFrom, s.latTo);
                        }
                        if (area.LeftLon != 0) {
                            area.LeftLon = Math.Min(Math.Min(s.lonFrom, s.lonTo), area.LeftLon);
                        }
                        else {
                            area.LeftLon = Math.Min(s.lonFrom, s.lonTo);
                        }
                        if (area.RightLon != 0) {
                            area.RightLon = Math.Max(Math.Max(s.lonFrom, s.lonTo), area.RightLon);
                        }
                        else {
                            area.RightLon = Math.Max(s.lonFrom, s.lonTo);
                        }
                        break;
                    case AirspaceSegmentTypes.Circle:
                        radius = (int)(((s.radius / 60.0f) * 360000.0));// / Math.Cos(WgsPoint.InternalToRad(s.latCenter)));
                        area.LeftLon = s.lonCenter - radius;
                        area.RightLon = s.lonCenter + radius;
                        area.TopLat = s.latCenter - radius;
                        area.BottomLat = s.latCenter + radius;
                        break;
                    case AirspaceSegmentTypes.Line:
                        area.BottomLat = Math.Max(s.latFrom, s.latTo);
                        area.TopLat = Math.Min(s.latFrom, s.latTo);
                        area.LeftLon = Math.Min(s.lonFrom, s.lonTo);
                        area.RightLon = Math.Max(s.lonFrom, s.lonTo);
                        break;
                    }
                }
                else {
                    switch (s.type) {
                    case AirspaceSegmentTypes.ArcClock:
                        radius = (int)(Math.Ceiling((s.radius / 60.0f) * 360000.0f));// / Math.Cos(WgsPoint.InternalToRad(s.latCenter)));
                        center = new WgsPoint(s.latCenter, s.lonCenter);
                        from = (int)Math.Floor(center.bearingDeg(new WgsPoint(s.latFrom, s.lonFrom)));
                        to = (int)Math.Ceiling(center.bearingDeg(new WgsPoint(s.latTo, s.lonTo)));
                        if (to < from) {
                            to += 360;
                        }
                        for (w = from; w < to; w += 90) {
                            ww = w % 360;
                            if (ww <= 90 && to >= 90) {
                                area.RightLon = Math.Max(s.lonCenter + radius, area.RightLon);
                            }
                            else if (ww <= 180 && to >= 180) {
                                area.BottomLat = Math.Max(s.latCenter + radius, area.BottomLat);
                            }
                            else if (ww <= 270 && to >= 270) {
                                area.LeftLon = Math.Min(s.lonCenter - radius, area.LeftLon);
                            }
                            else if (ww <= 360 && to >= 360) {
                                area.TopLat = Math.Min(s.latCenter - radius, area.TopLat);
                            }
                        }
                        goto case AirspaceSegmentTypes.Line;
                    case AirspaceSegmentTypes.ArcCounterClock:
                        radius = (int)(Math.Ceiling((s.radius / 60.0f) * 360000.0f));// / Math.Cos(WgsPoint.InternalToRad(s.latCenter)));
                        center = new WgsPoint(s.latCenter, s.lonCenter);
                        from = (int)Math.Ceiling(center.bearingDeg(new WgsPoint(s.latFrom, s.lonFrom)));
                        to = (int)Math.Floor(center.bearingDeg(new WgsPoint(s.latTo, s.lonTo)));
                        if (from < to) {
                            from += 360;
                        }
                        for (w = from; w > to; w -= 90) {
                            ww = w % 360;
                            if (ww <= 90 && to >= 270) {
                                area.TopLat = Math.Min(s.latCenter - radius, area.TopLat);
                            }
                            else if (ww <= 180 && to <= 90) {
                                area.RightLon = Math.Max(s.lonCenter + radius, area.RightLon);
                            }
                            else if (ww <= 270 && to <= 180) {
                                area.BottomLat = Math.Max(s.latCenter + radius, area.BottomLat);
                            }
                            else if (ww <= 360 && to <= 270) {
                                area.LeftLon = Math.Min(s.lonCenter - radius, area.LeftLon);
                            }
                        }
                        goto case AirspaceSegmentTypes.Line;
                    case AirspaceSegmentTypes.Circle:
                        radius = (int)(((s.radius / 60.0f) * 360000.0f));// / Math.Cos(WgsPoint.InternalToRad(s.latCenter)));
                        area.LeftLon = Math.Min(s.lonCenter - radius, area.LeftLon);
                        area.RightLon = Math.Max(s.lonCenter + radius, area.RightLon);
                        area.TopLat = Math.Min(s.latCenter - radius, area.TopLat);
                        area.BottomLat = Math.Max(s.latCenter + radius, area.BottomLat);
                        break;
                    case AirspaceSegmentTypes.Line:
                        area.BottomLat = Math.Max(Math.Max(s.latFrom, s.latTo), area.BottomLat);
                        area.TopLat = Math.Min(Math.Min(s.latFrom, s.latTo), area.TopLat);
                        area.LeftLon = Math.Min(Math.Min(s.lonFrom, s.lonTo), area.LeftLon);
                        area.RightLon = Math.Max(Math.Max(s.lonFrom, s.lonTo), area.RightLon);
                        break;
                    }
                }
            }
        }
    
        public override string ToString() {
            // TODO:  Add AirspaceElement.ToString implementation
            string txt;

            txt = airspace.name;
            if (airspace.airspaceType != AirspaceTypes.ControlArea &&
                airspace.airspaceType != AirspaceTypes.ControlZone &&
                airspace.airspaceType != AirspaceTypes.TerminalControlArea) {
                txt += airspace.id != "" ? " (" + airspace.id + ")" : "";
            }
            txt += string.Format("\n{0} {1}\n{2}->{3}", AirspaceTypeString, AirspaceClass != AirspaceClasses.Unknown ? AirspaceClassString : "", 
                airspace.lowerBound, airspace.upperBound);
            if (airspace.radioFrequency.Length > 0 || airspace.commName.Length > 0) {
                txt += "\nCOM: " + airspace.commName + " " + airspace.radioFrequency;
            }
            if (airspace.remark.Length > 0) {
                txt += "\nREM:" + airspace.remark;
            }
            return txt;
        }

        public static AirspaceElement CreateNewAirspaceElement(Airspace space) {
            AirspaceElement element;

            switch (space.airspaceType) {
            case AirspaceTypes.ControlArea:
                switch (space.airspaceClass) {
                case AirspaceClasses.A:
                    element = new ClassA(space);
                    break;
                case AirspaceClasses.B:
                    element = new ClassB(space);
                    break;
                case AirspaceClasses.C:
                    element = new ClassC(space);
                    break;
                case AirspaceClasses.D:
                    element = new ClassD(space);
                    break;
                case AirspaceClasses.E:
                    if (space.lowerBound.IndexOf("1000") != -1) {
                        element = new ClassELow(space);
                    }
                    else {
                        element = new ClassEHigh(space);
                    }
                    break;
                case AirspaceClasses.F:
                    element = new ClassF(space);
                    break;
                case AirspaceClasses.G:
                    element = new ClassG(space);
                    break;
                default:
                    element = new ClassOther(space);
                    break;
                }
                break;
            case AirspaceTypes.ControlZone:
                element = new ClassCTLZ(space);
                break;
            case AirspaceTypes.TerminalControlArea:
                element = new ClassTCA(space);
                break;
            case AirspaceTypes.Danger:
                element = new ClassDanger(space);
                break;
            case AirspaceTypes.Prohibited:
                element = new ClassProhibited(space);
                break;
            case AirspaceTypes.Restricted:
                element = new ClassRestricted(space);
                break;
            case AirspaceTypes.Military:
                element = new ClassMilitary(space);
                break;
            case AirspaceTypes.MilitaryAerodromeTraffic:
                element = new ClassMATZ(space);
                break;
            case AirspaceTypes.Temporary:
                element = new ClassTempReserved(space);
                break;
            case AirspaceTypes.TMZ:
                element = new ClassTMZ(space);
                break;
            case AirspaceTypes.Warning:
                element = new ClassWarning(space);
                break;
            case AirspaceTypes.Alert:
                element = new ClassAlert(space);
                break;
            case AirspaceTypes.Soaring:
                element = new ClassSoaring(space);
                break;
            case AirspaceTypes.Other:
                element = new ClassOther(space);
                break;
            case AirspaceTypes.Airway:
                element = new ClassAirway(space);
                break;
            case AirspaceTypes.GliderProhibited:
                element = new ClassGliderProhibited(space);
                break;
            default:
                element = new ClassOther(space);
                break;
            }        
            return element;
        }
    }
}
