// name   : AirspaceGenerator.cs
// author : Harald Maier
// date   : 11.02.2004
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.Collections;
using System.IO;
using System.Globalization;
using System.Text.RegularExpressions;

using SoaringDotNET.Data;

namespace SoaringDotNET.Database
{
	/// <summary>
	/// 
	/// </summary>

    public enum DafifTypes {ControlArea = 6, ControlZone = 7, TerminalControlArea = 11};

    public class AirspaceGenerator
	{
        private Hashtable airspace;
        private NumberFormatInfo ni;
        public static char MAGIC_AIRSPACE_NUMBER_V1 = (char)0x22;
        public static char MAGIC_AIRSPACE_NUMBER = (char)0x23;

		public AirspaceGenerator()
		{
			// 
			// TODO: Add constructor logic here
			//
            airspace = new Hashtable(1000);
            ni = new NumberFormatInfo();
            ni.NumberDecimalSeparator = ".";
		}
    
        public void GenDatabase(string path, Country country, DateTime validTo, string destPath) {
            airspace.Clear();
            ReadAirspace(path, country);
            ReadSUA(path, country);

            Country intenal = AppContents.CountryFromDafif(country.code);

            WriteDatabase(intenal, validTo, destPath);
        }

        public static void WriteDatabase(Country country, ArrayList space, DateTime validTo, string destPath) {
            string fileName = string.Format(@"{0}\{1}.sdn", destPath, country.name);
            using(BinaryWriter writer = new BinaryWriter(new FileStream(fileName, FileMode.Create, FileAccess.Write)))  {
                writer.Write(MAGIC_AIRSPACE_NUMBER);
                writer.Write(string.Format("{0:00}{1:00}{2:0000}{3}", validTo.Day, validTo.Month, validTo.Year, country.code));
                writer.Write(space.Count);

                WriteList(writer, space);
            }
        }

        public static void AppendDatabase(Country country, ArrayList space, DateTime validTo, string destPath) {
            bool ex = false;
            int cnt;
            string fileName = string.Format(@"{0}\{1}.sdn", destPath, country.name);
            string valid;
            FileStream fs;

            if (File.Exists(fileName)) {
                ex = true;
            }

            fs = new FileStream(fileName, FileMode.OpenOrCreate, FileAccess.ReadWrite);

            using(BinaryWriter writer = new BinaryWriter(fs))  {
                cnt = space.Count;
                valid = string.Format("{0:00}{1:00}{2:0000}{3}", validTo.Day, validTo.Month, validTo.Year, country.code);
                if (ex) {
                    BinaryReader reader = new BinaryReader(fs);
                    reader.ReadByte(); //magic number
                    valid = reader.ReadString();//valid date
                    cnt += reader.ReadInt32(); // items
                }

                writer.Seek(0, SeekOrigin.Begin);
                writer.Write(MAGIC_AIRSPACE_NUMBER);
                writer.Write(valid);
                writer.Write(cnt);
                writer.Seek(0, SeekOrigin.End);

                WriteList(writer, space);
            }
            fs.Close();
        }

        private void ReadAirspace(string path, Country country) {
            string file = path + "\\bdry\\bdry_par.txt";
            string line;
            int dafifType;
            string [] tokens;
            Airspace space = null;
            AirspaceSegment segment;
            string name;
            char [] levels = {'L', 'B'};
            Regex split = new Regex("[ \t]+");

            // read airspace description
            using (StreamReader inStream = new StreamReader(file)) {
                while ((line = inStream.ReadLine()) != null) {
                    if (country.keys.ContainsKey(line.Substring(0, 2))) {
                        tokens = line.Split('\t');
                        dafifType = int.Parse(tokens[1]);
                        switch (dafifType) {
                        case (int)DafifTypes.ControlArea:
                        case (int)DafifTypes.ControlZone:
                        case (int)DafifTypes.TerminalControlArea:
                            if (tokens[13].IndexOfAny(levels) != -1 && tokens[2].Length > 2) {
                                switch (tokens[10]) {
                                case "A":
                                    space = new Airspace(AirspaceClasses.A);
                                    break;
                                case "B":
                                    space = new Airspace(AirspaceClasses.B);
                                    break;
                                case "C":
                                    space = new Airspace(AirspaceClasses.C);
                                    break;
                                case "D":
                                    space = new Airspace(AirspaceClasses.D);
                                    break;
                                case "E":
                                    space = new Airspace(AirspaceClasses.E);
                                    break;
                                case "F":
                                    space = new Airspace(AirspaceClasses.F);
                                    break;
                                case "G":
                                    space = new Airspace(AirspaceClasses.G);
                                    break;
                                default:
                                    continue;
                                }

                                space.id = tokens[0];
                                space.name = tokens[2];
                                space.lowerBound = tokens[15].TrimStart('0');
                                space.upperBound = tokens[14].TrimStart('0');
                                space.radioFrequency = split.Split(tokens[8])[0];
                                space.remark = tokens[12];
                                space.commName = tokens[7];

                                if (dafifType == (int)DafifTypes.ControlArea) {
                                    space.airspaceType = AirspaceTypes.ControlArea;
                                }
                                else if (dafifType == (int)DafifTypes.ControlZone) {
                                    if (country.code == "UK" && space.name.IndexOf("MATZ") != -1) {
                                        space.airspaceType = AirspaceTypes.MilitaryAerodromeTraffic;
                                    }
                                    else {
                                        space.airspaceType = AirspaceTypes.ControlZone;
                                    }
                                }
                                else if (dafifType == (int)DafifTypes.TerminalControlArea) {
                                    space.airspaceType = AirspaceTypes.TerminalControlArea;
                                }
                                
                                airspace.Add(tokens[0], space);
                            }
                            break;
                        }
                    }
                }
            }

            // read points
            file = path + "\\bdry\\bdry.txt";
            name = "";
            using (StreamReader inStream = new StreamReader(file)) {
                while ((line = inStream.ReadLine()) != null) {
                    if (country.keys.ContainsKey(line.Substring(0, 2))) {
                        tokens = line.Split('\t');
                        // new airspace
                        if (tokens[0] != name) {
                            name = tokens[0];
                            space = (Airspace)airspace[name];
                        }

                        // append points
                        if (space != null) {
                            segment = new AirspaceSegment();

                            switch (tokens[5]) {
                            case "H":
                            case "G":
                            case "B":
                                segment.type = AirspaceSegmentTypes.Line;
                                segment.latFrom = (int)(-double.Parse(tokens[8].Length > 0 ? tokens[8] : "0", ni) * 360000.0);
                                segment.lonFrom = (int)(double.Parse(tokens[10].Length > 0 ? tokens[10] : "0", ni) * 360000.0);
                                segment.latTo = (int)(-double.Parse(tokens[12].Length > 0 ? tokens[12] : "0", ni) * 360000.0);
                                segment.lonTo = (int)(double.Parse(tokens[14].Length > 0 ? tokens[14] : "0", ni) * 360000.0);
                                break;
                            case "C":
                                segment.type = AirspaceSegmentTypes.Circle;
                                segment.latCenter = (int)(-double.Parse(tokens[16].Length > 0 ? tokens[16] : "0", ni) * 360000.0);
                                segment.lonCenter = (int)(double.Parse(tokens[18].Length > 0 ? tokens[18] : "0", ni) * 360000.0);
                                segment.radius = float.Parse(tokens[19].Length > 0 ? tokens[19] : "0", ni);
                                break;
                            case "L":
                            case "R":
                                if (tokens[5] == "L") {
                                    segment.type = AirspaceSegmentTypes.ArcCounterClock;
                                }
                                else {
                                    segment.type = AirspaceSegmentTypes.ArcClock;
                                }
                                segment.latFrom = (int)(-double.Parse(tokens[8].Length > 0 ? tokens[8] : "0", ni) * 360000.0);
                                segment.lonFrom = (int)(double.Parse(tokens[10].Length > 0 ? tokens[10] : "0", ni) * 360000.0);
                                segment.latTo = (int)(-double.Parse(tokens[12].Length > 0 ? tokens[12] : "0", ni) * 360000.0);
                                segment.lonTo = (int)(double.Parse(tokens[14].Length > 0 ? tokens[14] : "0", ni) * 360000.0);
                                segment.latCenter = (int)(-double.Parse(tokens[16].Length > 0 ? tokens[16] : "0", ni) * 360000.0);
                                segment.lonCenter = (int)(double.Parse(tokens[18].Length > 0 ? tokens[18] : "0", ni) * 360000.0);
                                segment.radius = float.Parse(tokens[19].Length > 0 ? tokens[19] : "0", ni);
                                break;
                            }

                            space.points.Add(segment);
                        }
                    }
                }
            }
        }

        private void ReadSUA(string path, Country country) {
            string file = path + "\\suas\\suas_par.txt";
            string line;
            string [] tokens;
            Airspace space = null;
            AirspaceSegment segment;
            char [] levels = {'L', 'B'};
            string name, oldName;
            Regex split = new Regex("[ \t]+");
            // read airspace description
            using (StreamReader inStream = new StreamReader(file)) {
                while ((line = inStream.ReadLine()) != null) {
                    tokens = line.Split('\t');
                    name = tokens[0].Replace("(", "");
                    name = name.Replace(")", "");
                    if (country.keys.ContainsKey(name.Substring(0, 2))) {
                        if (tokens[11].IndexOfAny(levels) != -1) {
                            switch (tokens[2]) {
                            case "D":
                                space = new Airspace();
                                space.airspaceType = AirspaceTypes.Danger;
                                break;
                            case "R":
                                space = new Airspace();
                                space.airspaceType = AirspaceTypes.Restricted;
                                break;
                            case "P":
                                space = new Airspace();
                                space.airspaceType = AirspaceTypes.Prohibited;
                                break;
                            case "M":
                                space = new Airspace();
                                space.airspaceType = AirspaceTypes.Military;
                                break;
                            case "T":
                                space = new Airspace();
                                space.airspaceType = AirspaceTypes.Temporary;
                                break;
                            case "W":
                                space = new Airspace();
                                space.airspaceType = AirspaceTypes.Warning;
                                break;
                            case "A":
                                space = new Airspace();
                                space.airspaceType = AirspaceTypes.Alert;
                                break;
                            default:
                                continue;
                            }

                            space.id = tokens[0] + tokens[1];
                            space.name = tokens[3];
                            space.lowerBound = tokens[13].TrimStart('0');
                            space.upperBound = tokens[12].TrimStart('0');
                            space.remark = tokens[14];
                            space.radioFrequency = split.Split(tokens[9])[0];
                            space.commName = tokens[8];
                            airspace.Add(space.id, space);
                        }
                    }
                }
            }

            // read points
            file = path + "\\suas\\suas.txt";
            oldName = "";

            using (StreamReader inStream = new StreamReader(file)) {
                while ((line = inStream.ReadLine()) != null) {
                    tokens = line.Split('\t');
                    name = tokens[0].Replace("(", "");
                    name = name.Replace(")", "");
                    if (country.keys.ContainsKey(name.Substring(0, 2))) {
                        // new airspace
                        name = tokens[0] + tokens[1];
                        if (name != oldName) {
                            oldName = name;
                            space = (Airspace)airspace[name];
                        }

                        // append points
                        if (space != null) {
                            segment = new AirspaceSegment();

                            switch (tokens[6]) {
                            case "H":
                            case "G":
                            case "B":
                                segment.type = AirspaceSegmentTypes.Line;
                                segment.latFrom = (int)(-double.Parse(tokens[9].Length > 0 ? tokens[9] : "0", ni) * 360000.0);
                                segment.lonFrom = (int)(double.Parse(tokens[11].Length > 0 ? tokens[11] : "0", ni) * 360000.0);
                                segment.latTo = (int)(-double.Parse(tokens[13].Length > 0 ? tokens[13] : "0", ni) * 360000.0);
                                segment.lonTo = (int)(double.Parse(tokens[15].Length > 0 ? tokens[15] : "0", ni) * 360000.0);
                                break;
                            case "C":
                                segment.type = AirspaceSegmentTypes.Circle;
                                segment.latCenter = (int)(-double.Parse(tokens[17].Length > 0 ? tokens[17] : "0", ni) * 360000.0);
                                segment.lonCenter = (int)(double.Parse(tokens[19].Length > 0 ? tokens[19] : "0", ni) * 360000.0);
                                segment.radius = float.Parse(tokens[20].Length > 0 ? tokens[20] : "0", ni);
                                break;
                            case "L":
                            case "R":
                                if (tokens[6] == "L") {
                                    segment.type = AirspaceSegmentTypes.ArcCounterClock;
                                }
                                else {
                                    segment.type = AirspaceSegmentTypes.ArcClock;
                                }
                                segment.latFrom = (int)(-double.Parse(tokens[9].Length > 0 ? tokens[9] : "0", ni) * 360000.0);
                                segment.lonFrom = (int)(double.Parse(tokens[11].Length > 0 ? tokens[11] : "0", ni) * 360000.0);
                                segment.latTo = (int)(-double.Parse(tokens[13].Length > 0 ? tokens[13] : "0", ni) * 360000.0);
                                segment.lonTo = (int)(double.Parse(tokens[15].Length > 0 ? tokens[15] : "0", ni) * 360000.0);
                                segment.latCenter = (int)(-double.Parse(tokens[17].Length > 0 ? tokens[17] : "0", ni) * 360000.0);
                                segment.lonCenter = (int)(double.Parse(tokens[19].Length > 0 ? tokens[19] : "0", ni) * 360000.0);
                                segment.radius = float.Parse(tokens[20].Length > 0 ? tokens[20] : "0", ni);
                                break;
                            }

                            space.points.Add(segment);
                        }
                    }
                }
            }
        }

        private void WriteDatabase(Country country, DateTime validTo, string destPath) {
            string fileName = string.Format(@"{0}\{1}.sdn", destPath, country.name);
            using(BinaryWriter writer = new BinaryWriter(new FileStream(fileName, FileMode.Create, FileAccess.Write)))  {
                writer.Write(MAGIC_AIRSPACE_NUMBER);
                writer.Write(string.Format("{0:00}{1:00}{2:0000}{3}", validTo.Day, validTo.Month, validTo.Year, country.code));
                writer.Write(airspace.Count);

                WriteList(writer, new ArrayList(airspace.Values));
            }
        }

        private static void WriteList(BinaryWriter writer, ArrayList space) {
            foreach (Airspace a in space) {
                writer.Write(a.id);
                writer.Write(a.name);
                writer.Write((char)a.airspaceClass);
                writer.Write((char)a.airspaceType);
                writer.Write(a.lowerBound.Replace("AMSL", "ALT").Replace("GND", "SFC").Replace("SURFACE", "SFC"));
                writer.Write(a.upperBound.Replace("AMSL", "ALT").Replace("GND", "SFC").Replace("SURFACE", "SFC"));
                writer.Write(a.remark);
                writer.Write(a.commName);
                writer.Write(a.radioFrequency);
                writer.Write(a.points.Count);
                foreach (AirspaceSegment s in a.points) {
                    writer.Write((char)s.type);
                    writer.Write(s.latFrom);
                    writer.Write(s.lonFrom);
                    writer.Write(s.latTo);
                    writer.Write(s.lonTo);
                    writer.Write(s.latCenter);
                    writer.Write(s.lonCenter);
                    writer.Write(s.radius);
                }
            }
        }
    }
}
