// name   : AppContents.cs
// author : Harald Maier
// date   : 17.08.2003
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.Collections;
using System.Xml;
using System.IO;
using System.Windows.Forms;
using System.Globalization;
using System.Drawing;
using Microsoft.Win32;

using SoaringDotNET.GUI;

namespace SoaringDotNET.Data
{
	/// <summary>
	/// 
	/// </summary>
    public enum WayPointTypeId {Unknown = -1, Airport, Glidersite, Outlanding, Landmark};
    public enum SurfaceTypeId {Unknown = -1, Grass, Hard};
    public enum CountryTypeId {Unknown = -1, Andorra, UnitedArabEmirates, Afghanistan, AntiguaBarbuda, Anguilla, 
        Albania, Armenia, NetherlandsAntilles, Angola, Antarctica, Argentina, AmericanSamoa, Austria, Australia, 
        Aruba, AlandIslands, Azerbaijan, BosniaHerzegovina, Barbados, Bangladesh, 
        Belgium, BurkinaFaso, Bulgaria, Bahrain, Burundi, 
        Benin, Bermuda, BruneiDarussalam, Bolivia, Brazil, Bahamas, Bhutan, BouvetIsland, Botswana, Belarus, Belize, Canada, 
        CocosKeelingIslands, DemocraticRepublicCongo, CentralAfricanRepublic, Congo, Switzerland, CoteDIvoire, 
        CookIslands, Chile, Cameroon, China, Colombia, CostaRica, SerbiaMontenegro, Cuba, CapeVerde, ChristmasIsland, 
        Cyprus, CzechRepublic, Germany, Djibouti, Denmark, Dominica, DominicanRepublic, Algeria, Ecuador, Estonia, 
        Egypt, WesternSahara, Eritrea, Spain, Ethiopia, Finland, Fiji, FalklandIslands, FederatedStatesMicronesia, 
        FaroeIslands, France, FranceMetropolitan, Gabon, GreatBritain, Grenada, Georgia, FrenchGuiana, Ghana, Gibraltar, Greenland, 
        Gambia, Guinea, Guadeloupe, EquatorialGuinea, Greece, SGeorgiaSSandwichIslands, Guatemala, Guam, GuineaBissau, 
        Guyana, HongKong, HeardIslandMcDonaldIslands, Honduras, Croatia, Haiti, Hungary, Indonesia, Ireland, Israel, 
        India, BritishIndianOceanTerritory, Iraq, Iran, Iceland, Italy, Jamaica, Jordan, Japan, Kenya, Kyrgyzstan, 
        Cambodia, Kiribati, Comoros, SaintKittsNevis, KoreaNorth, KoreaSouth, Kuwait, CaymanIslands, Kazakhstan, Laos, 
        Lebanon, SaintLucia, Liechtenstein, SriLanka, Liberia, Lesotho, Lithuania, Luxembourg, Latvia, Libya, Morocco, 
        Monaco, Moldova, Madagascar, MarshallIslands, Macedonia, Mali, Myanmar, Mongolia, Macao, NorthernMarianaIslands, 
        Martinique, Mauritania, Montserrat, Malta, Mauritius, Maldives, Malawi, Mexico, Malaysia, Mozambique, Namibia, 
        NewCaledonia, Niger, NorfolkIsland, Nigeria, Nicaragua, Netherlands, Norway, Nepal, Nauru, Niue, NewZealand, 
        Oman, Panama, Peru, FrenchPolynesia, PapuaNewGuinea, Philippines, Pakistan, Poland, SaintPierreMiquelon, Pitcairn, 
        PuertoRico, PalestinianTerritory, Portugal, Palau, Paraguay, Qatar, Reunion, Romania, RussianFederation, Rwanda, 
        SaudiArabia, SolomonIslands, Seychelles, Sudan, Sweden, Singapore, SaintHelena, Slovenia, SvalbardJanMayen, 
        Slovakia, SierraLeone, SanMarino, Senegal, Somalia, Suriname, SaoTomePrincipe, USSR, ElSalvador, Syria, Swaziland, 
        TurksCaicosIslands, Chad, FrenchSouthernTerritories, Togo, Thailand, Tajikistan, Tokelau, TimorLeste, Turkmenistan, 
        Tunisia, Tonga, EastTimor, Turkey, TrinidadTobago, Tuvalu, Taiwan, Tanzania, Ukraine, Uganda, UnitedKingdom, 
        UnitedStatesMinorOutlyingIslands, UnitedStates, Uruguay, Uzbekistan, VaticanCityState, SaintVincentGrenadines, 
        Venezuela, VirginIslandsBritish, VirginIslandsUS, Vietnam, Vanuatu, WallisFutuna, Samoa, Yemen, Mayotte, Yugoslavia, 
        SouthAfrica, Zambia, Zaire, Zimbabwe};

    public enum FileTypeId {All, Flight, Task, Waypoint};
    public enum TurnpointTypes {Unknown = -1, Start, Turnpoint, Finish};

    public class AppContents {
        public static TranslationList waypointTypes;
        public static TranslationList surfaceTypes;
        public static TranslationList countryTypes;
        public static TranslationList airspaceTypes;
        public static TranslationList airspaceClasses;
        public static TranslationList suaTypes;
        public static TranslationList segmentTypes;

        public WayPoint homePoint;
        public static ArrayList waypointCatalogs = new ArrayList();
        public static ArrayList flights = new ArrayList();
        public static ArrayList tasks = new ArrayList();
        public static ArrayList pilots = new ArrayList();
        public static ArrayList airspaces = new ArrayList();
        public static ArrayList compCountries = new ArrayList();
        public static ArrayList loadedAirspaceCountries = new ArrayList();
        public static ArrayList allColors = new ArrayList();

        public static Hashtable gliderIndexes = new Hashtable();
        public static Hashtable countryTypesByCode = new Hashtable();
        public static Hashtable airspaceColours = new Hashtable();
        public static Hashtable countryTypesById = new Hashtable();
        public static Hashtable airspacesByCountry = new Hashtable();
        public static Hashtable olcAirfields = new Hashtable(10000);
        public static Hashtable dafifToInternalCountries = new Hashtable();

        public static MapColor classAColor;
        public static MapColor classBColor;
        public static MapColor classCColor;
        public static MapColor classDColor;
        public static MapColor classEHighColor;
        public static MapColor classELowColor;
        public static MapColor classFColor;
        public static MapColor classGColor;
        public static MapColor classAlertColor;
        public static MapColor classDangerColor;
        public static MapColor classMilitaryColor;
        public static MapColor classMilitaryAeroTrafficColor;
        public static MapColor classProhibitedColor;
        public static MapColor classGliderProhibitedColor;
        public static MapColor classRestrictedColor;
        public static MapColor classTempColor;
        public static MapColor classTMZColor;
        public static MapColor classWarningColor;
        public static MapColor classOtherColor;
        public static MapColor classSoaringColor;
        public static MapColor classCTLZColor;
        public static MapColor classTCAColor;
        public static MapColor classAirwayColor;

        public static MapColor taskColor;
        public static MapColor taskColorFAI;
        public static MapColor flightColor;

        public static MapColor landableColor;
        public static MapColor notLandableColor;

        public static  object[] compClasses = new object[0];

        public static NumberFormatInfo ni = new NumberFormatInfo();

        private static Hashtable defaultSectors = new Hashtable();

        public SoaringDotNET.Dialogs.SplashScreenDialog splash;

        public AppContents() {
            // 
            // TODO: Add constructor logic here
            //
            splash = new SoaringDotNET.Dialogs.SplashScreenDialog();
            splash.SetProgress(13);
            splash.Show();

            waypointTypes = new TranslationList();
            waypointTypes.Add(new IdType(WayPointTypeId.Unknown, "Unknown"));
            waypointTypes.Add(new IdType(WayPointTypeId.Airport, "Airport"));
            waypointTypes.Add(new IdType(WayPointTypeId.Glidersite, "Glidersite"));
            waypointTypes.Add(new IdType(WayPointTypeId.Outlanding, "Outlanding field"));
            waypointTypes.Add(new IdType(WayPointTypeId.Landmark, "Landmark"));

            surfaceTypes = new TranslationList();
            surfaceTypes.Add(new IdType(SurfaceTypeId.Unknown, "Unknown"));
            surfaceTypes.Add(new IdType(SurfaceTypeId.Grass, "Grass"));
            surfaceTypes.Add(new IdType(SurfaceTypeId.Hard, "Asphalt/Concrete"));

            airspaceClasses = new TranslationList();
            airspaceClasses.Add(new IdType(AirspaceClasses.Unknown, "Unknown"));
            airspaceClasses.Add(new IdType(AirspaceClasses.A, "Class A"));
            airspaceClasses.Add(new IdType(AirspaceClasses.B, "Class B"));
            airspaceClasses.Add(new IdType(AirspaceClasses.C, "Class C"));
            airspaceClasses.Add(new IdType(AirspaceClasses.D, "Class D"));
            airspaceClasses.Add(new IdType(AirspaceClasses.E, "Class E"));
            airspaceClasses.Add(new IdType(AirspaceClasses.F, "Class F"));
            airspaceClasses.Add(new IdType(AirspaceClasses.G, "Class G"));

            airspaceTypes = new TranslationList();
            airspaceTypes.Add(new IdType(AirspaceTypes.Airway, "Airway"));
            airspaceTypes.Add(new IdType(AirspaceTypes.Alert, "Alert"));
            airspaceTypes.Add(new IdType(AirspaceTypes.ControlArea, "CTA/CTR"));
            airspaceTypes.Add(new IdType(AirspaceTypes.ControlZone, "CTLZ"));
            airspaceTypes.Add(new IdType(AirspaceTypes.Danger, "Danger"));
            airspaceTypes.Add(new IdType(AirspaceTypes.GliderProhibited, "Glider prohibited"));
            airspaceTypes.Add(new IdType(AirspaceTypes.Military, "Military operations"));
            airspaceTypes.Add(new IdType(AirspaceTypes.MilitaryAerodromeTraffic, "Military aerodrome traffic"));
            airspaceTypes.Add(new IdType(AirspaceTypes.Other, "Other"));
            airspaceTypes.Add(new IdType(AirspaceTypes.Prohibited, "Prohibited"));
            airspaceTypes.Add(new IdType(AirspaceTypes.Restricted, "Restricted"));
            airspaceTypes.Add(new IdType(AirspaceTypes.Soaring, "Soaring sector"));
            airspaceTypes.Add(new IdType(AirspaceTypes.Temporary, "Temp. reserved"));
            airspaceTypes.Add(new IdType(AirspaceTypes.TerminalControlArea, "TCA"));
            airspaceTypes.Add(new IdType(AirspaceTypes.TMZ, "TMZ"));
            airspaceTypes.Add(new IdType(AirspaceTypes.Unknown, "Unknown"));
            airspaceTypes.Add(new IdType(AirspaceTypes.Warning, "Warning"));
            
            suaTypes = new TranslationList();
            suaTypes.Add(new IdType(SUATypes.Airway, "Airway"));
            suaTypes.Add(new IdType(SUATypes.Alert, "Alert"));
            suaTypes.Add(new IdType(SUATypes.CTACTR, "CTA/CTR"));
            suaTypes.Add(new IdType(SUATypes.Danger, "Danger"));
            suaTypes.Add(new IdType(SUATypes.GliderProhibited, "Glider prohibited"));
            suaTypes.Add(new IdType(SUATypes.Military, "Military operations"));
            suaTypes.Add(new IdType(SUATypes.MilitaryAerodromeTraffic, "Military aerodrome traffic"));
            suaTypes.Add(new IdType(SUATypes.Other, "Other"));
            suaTypes.Add(new IdType(SUATypes.Prohibited, "Prohibited"));
            suaTypes.Add(new IdType(SUATypes.Restricted, "Restricted"));
            suaTypes.Add(new IdType(SUATypes.Soaring, "Soaring sector"));
            suaTypes.Add(new IdType(SUATypes.Temporary, "Temp. reserved"));
            suaTypes.Add(new IdType(SUATypes.TMZ, "TMZ"));
            suaTypes.Add(new IdType(SUATypes.Unknown, "Unknown"));
            suaTypes.Add(new IdType(SUATypes.Warning, "Warning"));

            segmentTypes = new TranslationList();
            segmentTypes.Add(new IdType(AirspaceSegmentTypes.ArcClock, "Arc clockwise"));
            segmentTypes.Add(new IdType(AirspaceSegmentTypes.ArcCounterClock, "Arc counter-clockwise"));
            segmentTypes.Add(new IdType(AirspaceSegmentTypes.Circle, "Circle"));
            segmentTypes.Add(new IdType(AirspaceSegmentTypes.Line, "Line"));

            countryTypes = new TranslationList();

            splash.SetMessage("Register Countries");
            RegisterCountries();
            foreach (string s in countryTypesByCode.Keys) {
                countryTypesById[countryTypesByCode[s]] = s;
            }
            splash.StepProgess();

            splash.SetMessage("Register Competition Classes");
            splash.StepProgess();

            compClasses = new object [] {new CompetitionClass("15m FAI", 5), 
                                            new CompetitionClass("18m", 6),
                                            new CompetitionClass("Club", 2),
                                            new CompetitionClass("Doubleseater", 7),
                                            new CompetitionClass("Open", 8),
                                            new CompetitionClass("Standard", 3),
                                            new CompetitionClass("World", 1)};
                                                 
            ni.NumberDecimalSeparator = ".";
            ni.NumberGroupSeparator = "";

            splash.SetMessage("Read Pilots");
            ReadPilots();
            splash.StepProgess();

            splash.SetMessage("Read Gliders");
            ReadGliders();
            splash.StepProgess();
            
            splash.SetMessage("Read Colours");
            ReadColours();
            splash.StepProgess();
            
            splash.SetMessage("Read Arispace");
            ReadAirspace();
            splash.StepProgess();
            
            splash.SetMessage("Read Sectors");
            ReadSectors();
            splash.StepProgess();
            
            splash.SetMessage("Read Competition Countries");
            ReadCompetitionCountries();
            splash.StepProgess();

            splash.SetMessage("Read OLC Airfields");
            ReadOLCAirfileds();
            splash.StepProgess();
        }

        #region Public Functions

        public void SaveColors() {
            foreach (MapColor c in allColors) {
                c.Save();
            }
        }

        public static void SaveAirspace() {
            RegistryKey currUser = Registry.CurrentUser.CreateSubKey("Software\\SoaringDotNet\\Path");
            string dirName = (string)currUser.GetValue("AirspacePath", Application.StartupPath + @"\airspace");
            
            foreach (CountryTypeId country in loadedAirspaceCountries) {
                string fileName = string.Format(@"{0}\{1}.sdn", dirName, country.ToString());
                using(BinaryWriter writer = new BinaryWriter(new FileStream(fileName, FileMode.Create, FileAccess.Write)))  {
                    writer.Write(Database.AirspaceGenerator.MAGIC_AIRSPACE_NUMBER);
                    writer.Write(string.Format("{0:00}{1:00}{2:0000}{3}", 0, 0, 0, AppContents.countryTypesById[country]));
                    writer.Write(((ArrayList)airspacesByCountry[country]).Count);
                    foreach (AirspaceElement e in ((ArrayList)airspacesByCountry[country])) {
                        Airspace a = e.airspace;
                        writer.Write(a.id);
                        writer.Write(a.name);
                        writer.Write((char)a.airspaceClass);
                        writer.Write((char)a.airspaceType);
                        writer.Write(a.lowerBound.Replace("AMSL", "ALT").Replace("GND", "SFC").Replace("SURFACE", "SFC"));
                        writer.Write(a.upperBound.Replace("AMSL", "ALT").Replace("GND", "SFC").Replace("SURFACE", "SFC"));
                        writer.Write(a.remark);
                        writer.Write(a.commName);
                        writer.Write(a.radioFrequency);
                        writer.Write(a.points.Count);
                        foreach (AirspaceSegment s in a.points) {
                            writer.Write((char)s.type);
                            writer.Write(s.latFrom);
                            writer.Write(s.lonFrom);
                            writer.Write(s.latTo);
                            writer.Write(s.lonTo);
                            writer.Write(s.latCenter);
                            writer.Write(s.lonCenter);
                            writer.Write(s.radius);
                        }
                    }
                }

                if (((ArrayList)airspacesByCountry[country]).Count == 0) {
                    try {
                        File.Delete(fileName);
                    }
                    catch {
                    }
                }
            }
        }

        public void SaveSectors() {
            RegistryKey currUser = Registry.CurrentUser.CreateSubKey("Software\\SoaringDotNet\\Sector");
            RegistryKey s;

            foreach (TurnpointTypes type in defaultSectors.Keys) {
                SectorDefinition sector = (SectorDefinition)defaultSectors[type];

                s = currUser.CreateSubKey(((int)type).ToString());
                s.SetValue("Type", (int)sector.sectorType);
                s.SetValue("Radius 1", sector.radius1);
                s.SetValue("Radius 2", sector.radius2);
            }
        }

        public void SavePilots() {
            string fileName = Application.StartupPath + @"\pilots.xml";
            XmlNode n;
            XmlElement root;
            XmlAttribute attr;
            try {
                XmlDocument doc = new XmlDocument();
                doc.AppendChild(doc.CreateDocumentType("SoaringDotNetPilots", null, null, null));
                root = doc.CreateElement("Pilots");
                foreach (Pilot p in pilots) {
                    n = doc.CreateElement("Pilot");

                    attr = doc.CreateAttribute("FirstName");
                    attr.Value = p.FirstName;
                    n.Attributes.Append(attr);
                    attr = doc.CreateAttribute("LastName");
                    attr.Value = p.LastName;
                    n.Attributes.Append(attr);
                    attr = doc.CreateAttribute("BirthDay");
                    attr.Value = p.BirthDayString;
                    n.Attributes.Append(attr);
                    attr = doc.CreateAttribute("Street");
                    attr.Value = p.Street;
                    n.Attributes.Append(attr);
                    attr = doc.CreateAttribute("Address");
                    attr.Value = p.Address;
                    n.Attributes.Append(attr);
                    attr = doc.CreateAttribute("Phone");
                    attr.Value = p.Phone;
                    n.Attributes.Append(attr);
                    attr = doc.CreateAttribute("eMail");
                    attr.Value = p.Email;
                    n.Attributes.Append(attr);

                    root.AppendChild(n);
                }
                doc.AppendChild(root);
                doc.Save(fileName);
            }
            catch (Exception e) {
                MessageBox.Show("Error occured while writing pilots file\n\n" + fileName + "\n\n" +
                    e.Message + "\n\n" + e.StackTrace, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        public WaypointCatalog NewCatalog() {
            int i = 1;
            bool found;

            do {
                found = false;
                foreach (WaypointCatalog c in waypointCatalogs) {
                    if (c.Name == "Waypoint" + i.ToString()) {
                        found = true;
                        i++;
                        break;
                    }
                }
            } while (found);

            WaypointCatalog cc = new WaypointCatalog("Waypoint" + i.ToString());
            waypointCatalogs.Add(cc);
            return cc;
        }

        public WaypointCatalog FindCatalog(string name) {
            WaypointCatalog cc = null;
            foreach (WaypointCatalog c in waypointCatalogs) {
                if (c.Name == name) {
                    cc = c;
                    break;
                }
            }
            return cc;
        }

        public void ReloadAirspace() {
            airspaces.Clear();
            airspacesByCountry.Clear();
            loadedAirspaceCountries.Clear();
            ReadColours();
            ReadAirspace();
        }

        public static SectorDefinition GetDefaultSector(TurnpointTypes type) {
            return new SectorDefinition((SectorDefinition)defaultSectors[type]);
        }

        public static Pilot FindPilot(string lastName, string firstName) {
            string tmpFirstName = firstName.ToUpper();
            string tmpLastName = lastName.ToUpper();
            foreach (Pilot p in pilots) {
                if ((p.LastName.ToUpper() == tmpLastName && p.FirstName.ToUpper() == tmpFirstName) ||
                    (p.LastName.ToUpper() == tmpFirstName && p.FirstName.ToUpper() == tmpLastName)) {
                    return p;
                }
            }

            Pilot p2 = new Pilot();
            p2.LastName = lastName;
            p2.FirstName = firstName;
            p2.BirthDay = DateTime.Today;
            return p2;
        }

        public static void AddAirspace(AirspaceElement element) {
            airspaces.Add(element);
            if (!loadedAirspaceCountries.Contains(element.airspace.country)) {
                loadedAirspaceCountries.Add(element.airspace.country);
            }

            if (!airspacesByCountry.Contains(element.airspace.country)) {
                airspacesByCountry.Add(element.airspace.country, new ArrayList());
            }
            ((ArrayList)airspacesByCountry[element.airspace.country]).Add(element);
            element.CalcArea();
        }

        public static void DeleteAirspace(AirspaceElement element) {
            airspaces.Remove(element);
            ((ArrayList)airspacesByCountry[element.airspace.country]).Remove(element);
        }

        public static Country CountryFromDafif(string dafifCode) {
            Country c = new Country();
            if (dafifToInternalCountries.Contains(dafifCode)) {
                c.code = (string)dafifToInternalCountries[dafifCode];
                c.name = countryTypesByCode[c.code].ToString();
            }
            else {
                c.code = "XX";
                c.name = "Unknown";
            }
            return c;
        }

        public static Country CreateCountry(CountryTypeId id) {
            Country c = new Country();
            c.code = (String)countryTypesById[id];
            c.name = countryTypes.getIdType((int) id).text;
            return c;
        }
        #endregion

        #region Private Functions

        private void ReadColours() {
            classAColor = new MapColor("Class A", Color.DarkGreen);
            classBColor = new MapColor("Class B", Color.DarkGreen);
            classCColor = new MapColor("Class C", Color.DarkGreen);
            classDColor = new MapColor("Class D", Color.DarkGreen);
            classEHighColor = new MapColor("Class E (High)", Color.Pink);
            classELowColor = new MapColor("Class E (Low)", Color.LightSkyBlue);
            classFColor = new MapColor("Class F", Color.LightBlue);
            classGColor = new MapColor("Class G", Color.LightCoral);
            classAlertColor = new MapColor("Alert", Color.Red);
            classDangerColor = new MapColor("Danger", Color.DarkBlue);
            classMilitaryColor = new MapColor("Military Operations", Color.Red);
            classMilitaryAeroTrafficColor = new MapColor("Military aerodrome traffic", Color.Red);
            classProhibitedColor = new MapColor("Prohibited", Color.DarkBlue);
            classGliderProhibitedColor = new MapColor("Glider Prohibited", Color.Orange);
            classRestrictedColor = new MapColor("Restricted", Color.DarkBlue);
            classTempColor = new MapColor("Temporary Reserved", Color.Wheat);
            classTMZColor = new MapColor("TMZ", Color.Red);
            classWarningColor = new MapColor("Warning", Color.Red);
            classOtherColor = new MapColor("Other", Color.Gray);
            classSoaringColor = new MapColor("Soaring Sector", Color.SteelBlue);
            classCTLZColor = new MapColor("CTLZ", Color.DarkRed);
            classTCAColor = new MapColor("TCA", Color.DarkGreen);
            classAirwayColor = new MapColor("Airway", Color.DarkGray);

            taskColor = new MapColor("Task", Color.BlueViolet);
            taskColorFAI = new MapColor("Task (FAI)", Color.Lime);
            flightColor = new MapColor("Flight Path", Color.Goldenrod);

            landableColor = new MapColor("Waypoint (landable)", Color.Green);
            notLandableColor = new MapColor("Waypoint (not landable)", Color.Firebrick);

            allColors.Clear();
            allColors.AddRange(new object [] {classAirwayColor, classAlertColor, classAColor, classBColor, classCColor, classDColor, classEHighColor, classELowColor, 
                                                 classFColor, classGColor, classCTLZColor, classTCAColor, classTMZColor, 
                                                 classWarningColor, classDangerColor, classRestrictedColor, classProhibitedColor, 
                                                 classMilitaryColor, classMilitaryAeroTrafficColor, classSoaringColor, classTempColor, classGliderProhibitedColor, classOtherColor,
                                                 taskColor, taskColorFAI, flightColor,
                                                 landableColor, notLandableColor});

            CheckCompatibilityVersion("Software\\SoaringDotNet\\Map\\Colors", 2);

            foreach (MapColor c in allColors) {
                c.Load();
            }
        }

        private void ReadSectors() {
            RegistryKey currUser = Registry.CurrentUser.CreateSubKey("Software\\SoaringDotNet\\Sector");
            RegistryKey s;
            SectorDefinition sector;

            s = currUser.CreateSubKey(((int)TurnpointTypes.Start).ToString());
            sector = new SectorDefinition(TurnpointTypes.Start);
            sector.sectorType = (SectorTypes)s.GetValue("Type", SectorTypes.Sector);
            sector.radius1 = (int)s.GetValue("Radius 1", 3000);
            sector.radius2 = (int)s.GetValue("Radius 2", 0);
            defaultSectors.Add(TurnpointTypes.Start, sector);

            s = currUser.CreateSubKey(((int)TurnpointTypes.Turnpoint).ToString());
            sector = new SectorDefinition(TurnpointTypes.Turnpoint);
            sector.sectorType = (SectorTypes)s.GetValue("Type", SectorTypes.Sector);
            sector.radius1 = (int)s.GetValue("Radius 1", 3000);
            sector.radius2 = (int)s.GetValue("Radius 2", 0);
            defaultSectors.Add(TurnpointTypes.Turnpoint, sector);
        
            s = currUser.CreateSubKey(((int)TurnpointTypes.Finish).ToString());
            sector = new SectorDefinition(TurnpointTypes.Finish);
            sector.sectorType = (SectorTypes)s.GetValue("Type", SectorTypes.Sector);
            sector.radius1 = (int)s.GetValue("Radius 1", 3000);
            sector.radius2 = (int)s.GetValue("Radius 2", 0);
            defaultSectors.Add(TurnpointTypes.Finish, sector);
        }

        private void ReadCompetitionCountries() {
            compCountries.Add(new CompetitionCountry("Germany", new string []{"DE"}, 
                                                     "http://www.segelflugszene.de/olc-cgi/olc-d/olc"));
            compCountries.Add(new CompetitionCountry("Austria", new string []{"AT"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-at/olc"));
            compCountries.Add(new CompetitionCountry("Switzerland", new string []{"CH"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-ch/olc"));
            compCountries.Add(new CompetitionCountry("Slovenia", new string []{"SI"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-si/olc"));
            compCountries.Add(new CompetitionCountry("Czech Republic/Slovakia", new string []{"CZ", "SK"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-cz/olc"));
            compCountries.Add(new CompetitionCountry("France", new string []{"FR"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-fr/olc"));
            compCountries.Add(new CompetitionCountry("Canada", new string []{"CA"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-ca/olc"));
            compCountries.Add(new CompetitionCountry("Africa", new string []{"AF", "BO", "DZ", "HS", "LI", "LK", "LY", "MA", "MW", "MM", "MZ", "NA", "OM", "ZM", "EG", "SY", "TN", "ZA", "Z"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-af/olc"));
            compCountries.Add(new CompetitionCountry("Australia", new string []{"AU"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-au/olc"));
            compCountries.Add(new CompetitionCountry("Belgium", new string []{"BE"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-be/olc"));
            compCountries.Add(new CompetitionCountry("Brasil", new string []{"BR"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-br/olc"));
            compCountries.Add(new CompetitionCountry("Denmark", new string []{"DK"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-dk/olc"));
            compCountries.Add(new CompetitionCountry("Finland", new string []{"FI"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-fi/olc"));
            compCountries.Add(new CompetitionCountry("Greece", new string []{"GR"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-gr/olc"));
            compCountries.Add(new CompetitionCountry("Hungary", new string []{"HU"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-hu/olc"));
            compCountries.Add(new CompetitionCountry("Italy", new string []{"IT"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-it/olc"));
            compCountries.Add(new CompetitionCountry("Japan", new string []{"JP"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-jp/olc"));
            compCountries.Add(new CompetitionCountry("The Netherlands", new string []{"NL"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-nl/olc"));
            compCountries.Add(new CompetitionCountry("New Zealand", new string []{"NZ"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-nz/olc"));
            compCountries.Add(new CompetitionCountry("Norway", new string []{"NO"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-no/olc"));
            compCountries.Add(new CompetitionCountry("Poland", new string []{"PL"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-pl/olc"));
            compCountries.Add(new CompetitionCountry("Spain", new string []{"ES"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-es/olc"));
            compCountries.Add(new CompetitionCountry("Sweden", new string []{"SE"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-se/olc"));
            compCountries.Add(new CompetitionCountry("United Kingdom", new string []{"UK"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-uk/olc"));
            compCountries.Add(new CompetitionCountry("USA", new string []{"US"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-usa/olc"));
            compCountries.Add(new CompetitionCountry("Argentinia", new string []{"AR"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-ar/olc"));
            compCountries.Add(new CompetitionCountry("Luxembourg", new string []{"LU"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-lu/olc"));
            compCountries.Add(new CompetitionCountry("Israel", new string []{"IL"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-il/olc"));
            compCountries.Add(new CompetitionCountry("Portugal", new string []{"PT"},
                                                     "http://www.segelflugszene.de/olc-cgi/olc-pt/olc"));
        }

        private void ReadGliders() {
            string fileName = Application.StartupPath + @"\gliders.txt";
            string line;
            string []tokens;
            char []splits = {',', '='};
            Glider g;
            try {
                FileInfo fi = new FileInfo(fileName);
                if (fi.Exists) {
                    using (StreamReader reader = new StreamReader(fileName)) {
                        while ((line = reader.ReadLine()) != null) {
                            line.Trim();
                            tokens = line.Split(splits);
                            if (tokens.Length == 3) {
                                g = new Glider(tokens[0], tokens[2], int.Parse(tokens[1]));
                                gliderIndexes.Add(tokens[0], g);
                            }
                        }
                    }
                }
            }
            catch (Exception e) {
                MessageBox.Show("Error occured while reading glider index file\n\n" + fileName + "\n\n" +
                    e.Message + "\n\n" + e.StackTrace, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void ReadPilots() {
            string fileName = Application.StartupPath + @"\pilots.xml";
            XmlAttributeCollection attrs;

            Pilot p;
            try {
                FileInfo fi = new FileInfo(fileName);
                if (fi.Exists) {
                    using (StreamReader reader = new StreamReader(fileName)) {
                        XmlDocument doc = new XmlDocument();
                        doc.Load(reader);
                        foreach (XmlNode n in doc.GetElementsByTagName("Pilot")) {
                            p = new Pilot();
                            attrs = n.Attributes;
                            p.FirstName = attrs["FirstName"].Value;
                            p.LastName = attrs["LastName"].Value;
                            p.BirthDay = DateTime.ParseExact(attrs["BirthDay"].Value, "dd.MM.yyyy", null);
                            p.Street = attrs["Street"].Value;
                            p.Address = attrs["Address"].Value;
                            p.Phone = attrs["Phone"].Value;
                            p.Email = attrs["eMail"].Value;

                            pilots.Add(p);
                        }
                    }
                }
            }
            catch (Exception e) {
                MessageBox.Show("Error occured while reading pilots file\n\n" + fileName + "\n\n" +
                    e.Message + "\n\n" + e.StackTrace, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void ReadOLCAirfileds() {
            string fileName = Application.StartupPath + @"\airfields.xml";
            OLCAirfield field;

            try {
                FileInfo fi = new FileInfo(fileName);
                if (fi.Exists) {
                    using (StreamReader reader = new StreamReader(fileName)) {
                        XmlDocument doc = new XmlDocument();
                        doc.Load(reader);
                        foreach (XmlNode n in doc.GetElementsByTagName("airfields")) {
                            field = new OLCAirfield();
                            foreach (XmlNode childNode in n.ChildNodes) {
                                switch (childNode.Name) {
                                case "id":
                                    field.ID = childNode.InnerText;
                                    break;
                                case "name":
                                    field.Name = childNode.InnerText;
                                    break;
                                case "icao_id":
                                    field.ICAO = childNode.InnerText;
                                    break;
                                case "country":
                                    field.Country = childNode.InnerText;
                                    break;
                                }
                            }
                            if (!olcAirfields.ContainsKey(field.Country)) {
                                olcAirfields.Add(field.Country, new ArrayList());
                            }
                            ((ArrayList)olcAirfields[field.Country]).Add(field);
                        }
                    }
                }
            }
            catch (Exception e) {
                MessageBox.Show("Error occured while reading OLC airfields file\n\n" + fileName + "\n\n" +
                    e.Message + "\n\n" + e.StackTrace, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void ReadAirspace() {
            string dirName = null;
            string t, country;
            int airspaceCount, segmentCount;
            int i, j;
            Airspace space;
            AirspaceElement element;
            AirspaceSegment segment;
            char magicNumber;

            RegistryKey currUser = Registry.CurrentUser.CreateSubKey("Software\\SoaringDotNet\\Path");
            dirName = (string)currUser.GetValue("AirspacePath", Application.StartupPath + @"\airspace");

            //DateTime validTo;
            if (Directory.Exists(dirName)) {
                string [] fileNames = Directory.GetFiles(dirName);
                foreach (string file in fileNames) {
                    try {
                        FileInfo fi = new FileInfo(file);
                        using (BinaryReader reader = new BinaryReader(fi.OpenRead())) {
                            magicNumber = (char)reader.ReadByte();
                            if (magicNumber == Database.AirspaceGenerator.MAGIC_AIRSPACE_NUMBER) {
                                t = reader.ReadString();
                                /*                                validTo = new DateTime(int.Parse(t.Substring(4, 4)), int.Parse(t.Substring(2, 2)), int.Parse(t.Substring(0, 2)));
                                                                if (validTo < DateTime.Today) {
                                                                    MessageBox.Show(string.Format("The airspace file {0} has been expired on {1}!\nPlease generate or download the latest version.", fi.Name, validTo), "Warning", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                                                                }*/
                                country = t.Substring(8);
                                airspaceCount = reader.ReadInt32();
                                
                                for (i = 0; i < airspaceCount; i++) {
                                    space = new Airspace();
                                    space.id = reader.ReadString();
                                    space.name = reader.ReadString();
                                    space.airspaceClass = (AirspaceClasses)reader.ReadChar();
                                    space.airspaceType = (AirspaceTypes)reader.ReadChar();
                                    space.lowerBound = reader.ReadString();
                                    space.upperBound = reader.ReadString();
                                    space.remark = reader.ReadString();
                                    space.commName = reader.ReadString();
                                    space.radioFrequency = reader.ReadString();
                                    space.country = (CountryTypeId)AppContents.countryTypesByCode[country];
                                    segmentCount = reader.ReadInt32();
                                    for (j = 0; j < segmentCount; j++) {
                                        segment = new AirspaceSegment();
                                        segment.type = (AirspaceSegmentTypes)reader.ReadChar();
                                        segment.latFrom = reader.ReadInt32();
                                        segment.lonFrom = reader.ReadInt32();
                                        segment.latTo = reader.ReadInt32();
                                        segment.lonTo = reader.ReadInt32();
                                        segment.latCenter = reader.ReadInt32();
                                        segment.lonCenter = reader.ReadInt32();
                                        segment.radius = reader.ReadSingle();

                                        space.points.Add(segment);
                                    }

                                    element = AirspaceElement.CreateNewAirspaceElement(space);
                                    AppContents.AddAirspace(element);
                                }
                            }
                            else if (magicNumber == Database.AirspaceGenerator.MAGIC_AIRSPACE_NUMBER_V1) {
                                MessageBox.Show("Airspace file has file version 1\n\nPlease regenerate your airspace database!\n\n" + file,
                                    "Warning", MessageBoxButtons.OK, MessageBoxIcon.Error);
                                break;
                            }
                        }
                    }
                    catch (Exception e) {
                        MessageBox.Show("Error occured while reading airspace file\n\nPlease regenerate or download the latest version!\n\n" + file + "\n\n" +
                            e.Message + "\n\n" + e.StackTrace, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);

                        airspaces.Clear();
                        break;
                    }

                    IComparer comp = new AirspaceSorter();
                    airspaces.Sort(comp);
                }
            }
            else {
                try {
                    Directory.CreateDirectory(dirName);
                    MessageBox.Show("The airspace direcory has been created in\n\n" + dirName + "\n\n" +
                        "Please generate or download your requested airspace databases\nand copy the files into this directory.", "Information", MessageBoxButtons.OK, MessageBoxIcon.Information);
                }
                catch (Exception e) {
                    MessageBox.Show("Error occured while creating airspace directory\n\n" + dirName + "\n\n" +
                        e.Message + "\n\n" + e.StackTrace, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
        }

        private void RegisterCountries() {
            countryTypes.Add(new IdType(CountryTypeId.Unknown, "Unknown"));
            countryTypes.Add(new IdType(CountryTypeId.Andorra, "Andorra"));
            countryTypes.Add(new IdType(CountryTypeId.UnitedArabEmirates, "United Arab Emirates"));
            countryTypes.Add(new IdType(CountryTypeId.Afghanistan, "Afghanistan"));
            countryTypes.Add(new IdType(CountryTypeId.AntiguaBarbuda, "Antigua and Barbuda"));
            countryTypes.Add(new IdType(CountryTypeId.Anguilla, "Anguilla"));
            countryTypes.Add(new IdType(CountryTypeId.Albania, "Albania"));
            countryTypes.Add(new IdType(CountryTypeId.Armenia, "Armenia"));
            countryTypes.Add(new IdType(CountryTypeId.NetherlandsAntilles, "Netherlands Antilles"));
            countryTypes.Add(new IdType(CountryTypeId.Angola, "Angola"));
            countryTypes.Add(new IdType(CountryTypeId.Antarctica, "Antarctica"));
            countryTypes.Add(new IdType(CountryTypeId.Argentina, "Argentina"));
            countryTypes.Add(new IdType(CountryTypeId.AmericanSamoa, "American Samoa"));
            countryTypes.Add(new IdType(CountryTypeId.Austria, "Austria"));
            countryTypes.Add(new IdType(CountryTypeId.Australia, "Australia"));
            countryTypes.Add(new IdType(CountryTypeId.Aruba, "Aruba"));
            countryTypes.Add(new IdType(CountryTypeId.AlandIslands, "Aland Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Azerbaijan, "Azerbaijan"));
            countryTypes.Add(new IdType(CountryTypeId.BosniaHerzegovina, "Bosnia and Herzegovina"));
            countryTypes.Add(new IdType(CountryTypeId.Barbados, "Barbados"));
            countryTypes.Add(new IdType(CountryTypeId.Bangladesh, "Bangladesh"));
            countryTypes.Add(new IdType(CountryTypeId.Belgium, "Belgium"));
            countryTypes.Add(new IdType(CountryTypeId.BurkinaFaso, "Burkina Faso"));
            countryTypes.Add(new IdType(CountryTypeId.Bulgaria, "Bulgaria"));
            countryTypes.Add(new IdType(CountryTypeId.Bahrain, "Bahrain"));
            countryTypes.Add(new IdType(CountryTypeId.Burundi, "Burundi"));
            countryTypes.Add(new IdType(CountryTypeId.Benin, "Benin"));
            countryTypes.Add(new IdType(CountryTypeId.Bermuda, "Bermuda"));
            countryTypes.Add(new IdType(CountryTypeId.BruneiDarussalam, "Brunei Darussalam"));
            countryTypes.Add(new IdType(CountryTypeId.Bolivia, "Bolivia"));
            countryTypes.Add(new IdType(CountryTypeId.Brazil, "Brazil"));
            countryTypes.Add(new IdType(CountryTypeId.Bahamas, "Bahamas"));
            countryTypes.Add(new IdType(CountryTypeId.Bhutan, "Bhutan"));
            countryTypes.Add(new IdType(CountryTypeId.BouvetIsland, "Bouvet Island"));
            countryTypes.Add(new IdType(CountryTypeId.Botswana, "Botswana"));
            countryTypes.Add(new IdType(CountryTypeId.Belarus, "Belarus"));
            countryTypes.Add(new IdType(CountryTypeId.Belize, "Belize"));
            countryTypes.Add(new IdType(CountryTypeId.Canada, "Canada"));
            countryTypes.Add(new IdType(CountryTypeId.CocosKeelingIslands, "Cocos (Keeling) Islands"));
            countryTypes.Add(new IdType(CountryTypeId.DemocraticRepublicCongo, "Democratic Republic of the Congo"));
            countryTypes.Add(new IdType(CountryTypeId.CentralAfricanRepublic, "Central African Republic"));
            countryTypes.Add(new IdType(CountryTypeId.Congo, "Congo"));
            countryTypes.Add(new IdType(CountryTypeId.Switzerland, "Switzerland"));
            countryTypes.Add(new IdType(CountryTypeId.CoteDIvoire, "Cote D'Ivoire (Ivory Coast)"));
            countryTypes.Add(new IdType(CountryTypeId.CookIslands, "Cook Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Chile, "Chile"));
            countryTypes.Add(new IdType(CountryTypeId.Cameroon, "Cameroon"));
            countryTypes.Add(new IdType(CountryTypeId.China, "China"));
            countryTypes.Add(new IdType(CountryTypeId.Colombia, "Colombia"));
            countryTypes.Add(new IdType(CountryTypeId.CostaRica, "Costa Rica"));
            countryTypes.Add(new IdType(CountryTypeId.SerbiaMontenegro, "Serbia and Montenegro"));
            countryTypes.Add(new IdType(CountryTypeId.Cuba, "Cuba"));
            countryTypes.Add(new IdType(CountryTypeId.CapeVerde, "Cape Verde"));
            countryTypes.Add(new IdType(CountryTypeId.ChristmasIsland, "Christmas Island"));
            countryTypes.Add(new IdType(CountryTypeId.Cyprus, "Cyprus"));
            countryTypes.Add(new IdType(CountryTypeId.CzechRepublic, "Czech Republic"));
            countryTypes.Add(new IdType(CountryTypeId.Germany, "Germany"));
            countryTypes.Add(new IdType(CountryTypeId.Djibouti, "Djibouti"));
            countryTypes.Add(new IdType(CountryTypeId.Denmark, "Denmark"));
            countryTypes.Add(new IdType(CountryTypeId.Dominica, "Dominica"));
            countryTypes.Add(new IdType(CountryTypeId.DominicanRepublic, "Dominican Republic"));
            countryTypes.Add(new IdType(CountryTypeId.Algeria, "Algeria"));
            countryTypes.Add(new IdType(CountryTypeId.Ecuador, "Ecuador"));
            countryTypes.Add(new IdType(CountryTypeId.Estonia, "Estonia"));
            countryTypes.Add(new IdType(CountryTypeId.Egypt, "Egypt"));
            countryTypes.Add(new IdType(CountryTypeId.WesternSahara, "Western Sahara"));
            countryTypes.Add(new IdType(CountryTypeId.Eritrea, "Eritrea"));
            countryTypes.Add(new IdType(CountryTypeId.Spain, "Spain"));
            countryTypes.Add(new IdType(CountryTypeId.Ethiopia, "Ethiopia"));
            countryTypes.Add(new IdType(CountryTypeId.Finland, "Finland"));
            countryTypes.Add(new IdType(CountryTypeId.Fiji, "Fiji"));
            countryTypes.Add(new IdType(CountryTypeId.FalklandIslands, "Falkland Islands (Malvinas)"));
            countryTypes.Add(new IdType(CountryTypeId.FederatedStatesMicronesia, "Federated States of Micronesia"));
            countryTypes.Add(new IdType(CountryTypeId.FaroeIslands, "Faroe Islands"));
            countryTypes.Add(new IdType(CountryTypeId.France, "France"));
            countryTypes.Add(new IdType(CountryTypeId.FranceMetropolitan, "France, Metropolitan"));
            countryTypes.Add(new IdType(CountryTypeId.Gabon, "Gabon"));
            countryTypes.Add(new IdType(CountryTypeId.GreatBritain, "Great Britain (UK)"));
            countryTypes.Add(new IdType(CountryTypeId.Grenada, "Grenada"));
            countryTypes.Add(new IdType(CountryTypeId.Georgia, "Georgia"));
            countryTypes.Add(new IdType(CountryTypeId.FrenchGuiana, "French Guiana"));
            countryTypes.Add(new IdType(CountryTypeId.Ghana, "Ghana"));
            countryTypes.Add(new IdType(CountryTypeId.Gibraltar, "Gibraltar"));
            countryTypes.Add(new IdType(CountryTypeId.Greenland, "Greenland"));
            countryTypes.Add(new IdType(CountryTypeId.Gambia, "Gambia"));
            countryTypes.Add(new IdType(CountryTypeId.Guinea, "Guinea"));
            countryTypes.Add(new IdType(CountryTypeId.Guadeloupe, "Guadeloupe"));
            countryTypes.Add(new IdType(CountryTypeId.EquatorialGuinea, "Equatorial Guinea"));
            countryTypes.Add(new IdType(CountryTypeId.Greece, "Greece"));
            countryTypes.Add(new IdType(CountryTypeId.SGeorgiaSSandwichIslands, "S. Georgia and S. Sandwich Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Guatemala, "Guatemala"));
            countryTypes.Add(new IdType(CountryTypeId.Guam, "Guam"));
            countryTypes.Add(new IdType(CountryTypeId.GuineaBissau, "Guinea-Bissau"));
            countryTypes.Add(new IdType(CountryTypeId.Guyana, "Guyana"));
            countryTypes.Add(new IdType(CountryTypeId.HongKong, "Hong Kong"));
            countryTypes.Add(new IdType(CountryTypeId.HeardIslandMcDonaldIslands, "Heard Island and McDonald Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Honduras, "Honduras"));
            countryTypes.Add(new IdType(CountryTypeId.Croatia, "Croatia (Hrvatska)"));
            countryTypes.Add(new IdType(CountryTypeId.Haiti, "Haiti"));
            countryTypes.Add(new IdType(CountryTypeId.Hungary, "Hungary"));
            countryTypes.Add(new IdType(CountryTypeId.Indonesia, "Indonesia"));
            countryTypes.Add(new IdType(CountryTypeId.Ireland, "Ireland"));
            countryTypes.Add(new IdType(CountryTypeId.Israel, "Israel"));
            countryTypes.Add(new IdType(CountryTypeId.India, "India"));
            countryTypes.Add(new IdType(CountryTypeId.BritishIndianOceanTerritory, "British Indian Ocean Territory"));
            countryTypes.Add(new IdType(CountryTypeId.Iraq, "Iraq"));
            countryTypes.Add(new IdType(CountryTypeId.Iran, "Iran"));
            countryTypes.Add(new IdType(CountryTypeId.Iceland, "Iceland"));
            countryTypes.Add(new IdType(CountryTypeId.Italy, "Italy"));
            countryTypes.Add(new IdType(CountryTypeId.Jamaica, "Jamaica"));
            countryTypes.Add(new IdType(CountryTypeId.Jordan, "Jordan"));
            countryTypes.Add(new IdType(CountryTypeId.Japan, "Japan"));
            countryTypes.Add(new IdType(CountryTypeId.Kenya, "Kenya"));
            countryTypes.Add(new IdType(CountryTypeId.Kyrgyzstan, "Kyrgyzstan"));
            countryTypes.Add(new IdType(CountryTypeId.Cambodia, "Cambodia"));
            countryTypes.Add(new IdType(CountryTypeId.Kiribati, "Kiribati"));
            countryTypes.Add(new IdType(CountryTypeId.Comoros, "Comoros"));
            countryTypes.Add(new IdType(CountryTypeId.SaintKittsNevis, "Saint Kitts and Nevis"));
            countryTypes.Add(new IdType(CountryTypeId.KoreaNorth, "KP Korea (North)"));
            countryTypes.Add(new IdType(CountryTypeId.KoreaSouth, "KR Korea (South)"));
            countryTypes.Add(new IdType(CountryTypeId.Kuwait, "Kuwait"));
            countryTypes.Add(new IdType(CountryTypeId.CaymanIslands, "Cayman Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Kazakhstan, "Kazakhstan"));
            countryTypes.Add(new IdType(CountryTypeId.Laos, "Laos"));
            countryTypes.Add(new IdType(CountryTypeId.Lebanon, "Lebanon"));
            countryTypes.Add(new IdType(CountryTypeId.SaintLucia, "Saint Lucia"));
            countryTypes.Add(new IdType(CountryTypeId.Liechtenstein, "Liechtenstein"));
            countryTypes.Add(new IdType(CountryTypeId.SriLanka, "Sri Lanka"));
            countryTypes.Add(new IdType(CountryTypeId.Liberia, "Liberia"));
            countryTypes.Add(new IdType(CountryTypeId.Lesotho, "Lesotho"));
            countryTypes.Add(new IdType(CountryTypeId.Lithuania, "Lithuania"));
            countryTypes.Add(new IdType(CountryTypeId.Luxembourg, "Luxembourg"));
            countryTypes.Add(new IdType(CountryTypeId.Latvia, "Latvia"));
            countryTypes.Add(new IdType(CountryTypeId.Libya, "Libya"));
            countryTypes.Add(new IdType(CountryTypeId.Morocco, "Morocco"));
            countryTypes.Add(new IdType(CountryTypeId.Monaco, "Monaco"));
            countryTypes.Add(new IdType(CountryTypeId.Moldova, "Moldova"));
            countryTypes.Add(new IdType(CountryTypeId.Madagascar, "Madagascar"));
            countryTypes.Add(new IdType(CountryTypeId.MarshallIslands, "Marshall Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Macedonia, "Macedonia"));
            countryTypes.Add(new IdType(CountryTypeId.Mali, "Mali"));
            countryTypes.Add(new IdType(CountryTypeId.Myanmar, "Myanmar"));
            countryTypes.Add(new IdType(CountryTypeId.Mongolia, "Mongolia"));
            countryTypes.Add(new IdType(CountryTypeId.Macao, "Macao"));
            countryTypes.Add(new IdType(CountryTypeId.NorthernMarianaIslands, "Northern Mariana Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Martinique, "Martinique"));
            countryTypes.Add(new IdType(CountryTypeId.Mauritania, "Mauritania"));
            countryTypes.Add(new IdType(CountryTypeId.Montserrat, "Montserrat"));
            countryTypes.Add(new IdType(CountryTypeId.Malta, "Malta"));
            countryTypes.Add(new IdType(CountryTypeId.Mauritius, "Mauritius"));
            countryTypes.Add(new IdType(CountryTypeId.Maldives, "Maldives"));
            countryTypes.Add(new IdType(CountryTypeId.Malawi, "Malawi"));
            countryTypes.Add(new IdType(CountryTypeId.Mexico, "Mexico"));
            countryTypes.Add(new IdType(CountryTypeId.Malaysia, "Malaysia"));
            countryTypes.Add(new IdType(CountryTypeId.Mozambique, "Mozambique"));
            countryTypes.Add(new IdType(CountryTypeId.Namibia, "Namibia"));
            countryTypes.Add(new IdType(CountryTypeId.NewCaledonia, "New Caledonia"));
            countryTypes.Add(new IdType(CountryTypeId.Niger, "Niger"));
            countryTypes.Add(new IdType(CountryTypeId.NorfolkIsland, "Norfolk Island"));
            countryTypes.Add(new IdType(CountryTypeId.Nigeria, "Nigeria"));
            countryTypes.Add(new IdType(CountryTypeId.Nicaragua, "Nicaragua"));
            countryTypes.Add(new IdType(CountryTypeId.Netherlands, "Netherlands"));
            countryTypes.Add(new IdType(CountryTypeId.Norway, "Norway"));
            countryTypes.Add(new IdType(CountryTypeId.Nepal, "Nepal"));
            countryTypes.Add(new IdType(CountryTypeId.Nauru, "Nauru"));
            countryTypes.Add(new IdType(CountryTypeId.Niue, "Niue"));
            countryTypes.Add(new IdType(CountryTypeId.NewZealand, "New Zealand (Aotearoa)"));
            countryTypes.Add(new IdType(CountryTypeId.Oman, "Oman"));
            countryTypes.Add(new IdType(CountryTypeId.Panama, "Panama"));
            countryTypes.Add(new IdType(CountryTypeId.Peru, "Peru"));
            countryTypes.Add(new IdType(CountryTypeId.FrenchPolynesia, "French Polynesia"));
            countryTypes.Add(new IdType(CountryTypeId.PapuaNewGuinea, "Papua New Guinea"));
            countryTypes.Add(new IdType(CountryTypeId.Philippines, "Philippines"));
            countryTypes.Add(new IdType(CountryTypeId.Pakistan, "Pakistan"));
            countryTypes.Add(new IdType(CountryTypeId.Poland, "Poland"));
            countryTypes.Add(new IdType(CountryTypeId.SaintPierreMiquelon, "Saint Pierre and Miquelon"));
            countryTypes.Add(new IdType(CountryTypeId.Pitcairn, "Pitcairn"));
            countryTypes.Add(new IdType(CountryTypeId.PuertoRico, "Puerto Rico"));
            countryTypes.Add(new IdType(CountryTypeId.PalestinianTerritory, "Palestinian Territory"));
            countryTypes.Add(new IdType(CountryTypeId.Portugal, "Portugal"));
            countryTypes.Add(new IdType(CountryTypeId.Palau, "Palau"));
            countryTypes.Add(new IdType(CountryTypeId.Paraguay, "Paraguay"));
            countryTypes.Add(new IdType(CountryTypeId.Qatar, "Qatar"));
            countryTypes.Add(new IdType(CountryTypeId.Reunion, "Reunion"));
            countryTypes.Add(new IdType(CountryTypeId.Romania, "Romania"));
            countryTypes.Add(new IdType(CountryTypeId.RussianFederation, "Russian Federation"));
            countryTypes.Add(new IdType(CountryTypeId.Rwanda, "Rwanda"));
            countryTypes.Add(new IdType(CountryTypeId.SaudiArabia, "Saudi Arabia"));
            countryTypes.Add(new IdType(CountryTypeId.SolomonIslands, "Solomon Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Seychelles, "Seychelles"));
            countryTypes.Add(new IdType(CountryTypeId.Sudan, "Sudan"));
            countryTypes.Add(new IdType(CountryTypeId.Sweden, "Sweden"));
            countryTypes.Add(new IdType(CountryTypeId.Singapore, "Singapore"));
            countryTypes.Add(new IdType(CountryTypeId.SaintHelena, "Saint Helena"));
            countryTypes.Add(new IdType(CountryTypeId.Slovenia, "Slovenia"));
            countryTypes.Add(new IdType(CountryTypeId.SvalbardJanMayen, "Svalbard and Jan Mayen"));
            countryTypes.Add(new IdType(CountryTypeId.Slovakia, "Slovakia"));
            countryTypes.Add(new IdType(CountryTypeId.SierraLeone, "Sierra Leone"));
            countryTypes.Add(new IdType(CountryTypeId.SanMarino, "San Marino"));
            countryTypes.Add(new IdType(CountryTypeId.Senegal, "Senegal"));
            countryTypes.Add(new IdType(CountryTypeId.Somalia, "Somalia"));
            countryTypes.Add(new IdType(CountryTypeId.Suriname, "Suriname"));
            countryTypes.Add(new IdType(CountryTypeId.SaoTomePrincipe, "Sao Tome and Principe"));
            countryTypes.Add(new IdType(CountryTypeId.USSR, "USSR (former)"));
            countryTypes.Add(new IdType(CountryTypeId.ElSalvador, "El Salvador"));
            countryTypes.Add(new IdType(CountryTypeId.Syria, "Syria"));
            countryTypes.Add(new IdType(CountryTypeId.Swaziland, "Swaziland"));
            countryTypes.Add(new IdType(CountryTypeId.TurksCaicosIslands, "Turks and Caicos Islands"));
            countryTypes.Add(new IdType(CountryTypeId.Chad, "Chad"));
            countryTypes.Add(new IdType(CountryTypeId.FrenchSouthernTerritories, "French Southern Territories"));
            countryTypes.Add(new IdType(CountryTypeId.Togo, "Togo"));
            countryTypes.Add(new IdType(CountryTypeId.Thailand, "Thailand"));
            countryTypes.Add(new IdType(CountryTypeId.Tajikistan, "Tajikistan"));
            countryTypes.Add(new IdType(CountryTypeId.Tokelau, "Tokelau"));
            countryTypes.Add(new IdType(CountryTypeId.TimorLeste, "Timor-Leste"));
            countryTypes.Add(new IdType(CountryTypeId.Turkmenistan, "Turkmenistan"));
            countryTypes.Add(new IdType(CountryTypeId.Tunisia, "Tunisia"));
            countryTypes.Add(new IdType(CountryTypeId.Tonga, "Tonga"));
            countryTypes.Add(new IdType(CountryTypeId.EastTimor, "East Timor"));
            countryTypes.Add(new IdType(CountryTypeId.Turkey, "Turkey"));
            countryTypes.Add(new IdType(CountryTypeId.TrinidadTobago, "Trinidad and Tobago"));
            countryTypes.Add(new IdType(CountryTypeId.Tuvalu, "Tuvalu"));
            countryTypes.Add(new IdType(CountryTypeId.Taiwan, "Taiwan"));
            countryTypes.Add(new IdType(CountryTypeId.Tanzania, "Tanzania"));
            countryTypes.Add(new IdType(CountryTypeId.Ukraine, "Ukraine"));
            countryTypes.Add(new IdType(CountryTypeId.Uganda, "Uganda"));
            countryTypes.Add(new IdType(CountryTypeId.UnitedKingdom, "United Kingdom"));
            countryTypes.Add(new IdType(CountryTypeId.UnitedStatesMinorOutlyingIslands, "United States Minor Outlying Islands"));
            countryTypes.Add(new IdType(CountryTypeId.UnitedStates, "United States"));
            countryTypes.Add(new IdType(CountryTypeId.Uruguay, "Uruguay"));
            countryTypes.Add(new IdType(CountryTypeId.Uzbekistan, "Uzbekistan"));
            countryTypes.Add(new IdType(CountryTypeId.VaticanCityState, "Vatican City State (Holy See)"));
            countryTypes.Add(new IdType(CountryTypeId.SaintVincentGrenadines, "Saint Vincent and the Grenadines"));
            countryTypes.Add(new IdType(CountryTypeId.Venezuela, "Venezuela"));
            countryTypes.Add(new IdType(CountryTypeId.VirginIslandsBritish, "Virgin Islands (British)"));
            countryTypes.Add(new IdType(CountryTypeId.VirginIslandsUS, "Virgin Islands (U.S.)"));
            countryTypes.Add(new IdType(CountryTypeId.Vietnam, "Viet Nam"));
            countryTypes.Add(new IdType(CountryTypeId.Vanuatu, "Vanuatu"));
            countryTypes.Add(new IdType(CountryTypeId.WallisFutuna, "Wallis and Futuna"));
            countryTypes.Add(new IdType(CountryTypeId.Samoa, "Samoa"));
            countryTypes.Add(new IdType(CountryTypeId.Yemen, "Yemen"));
            countryTypes.Add(new IdType(CountryTypeId.Mayotte, "Mayotte"));
            countryTypes.Add(new IdType(CountryTypeId.Yugoslavia, "Yugoslavia (former)"));
            countryTypes.Add(new IdType(CountryTypeId.SouthAfrica, "South Africa"));
            countryTypes.Add(new IdType(CountryTypeId.Zambia, "Zambia"));
            countryTypes.Add(new IdType(CountryTypeId.Zaire, "Zaire (former)"));
            countryTypes.Add(new IdType(CountryTypeId.Zimbabwe, "Zimbabwe"));

            countryTypesByCode["AD"] = CountryTypeId.Andorra;
            countryTypesByCode["AE"] = CountryTypeId.UnitedArabEmirates;
            countryTypesByCode["AF"] = CountryTypeId.Afghanistan;
            countryTypesByCode["AG"] = CountryTypeId.AntiguaBarbuda;
            countryTypesByCode["AI"] = CountryTypeId.Anguilla;
            countryTypesByCode["AL"] = CountryTypeId.Albania;
            countryTypesByCode["AM"] = CountryTypeId.Armenia;
            countryTypesByCode["AN"] = CountryTypeId.NetherlandsAntilles;
            countryTypesByCode["AO"] = CountryTypeId.Angola;
            countryTypesByCode["AQ"] = CountryTypeId.Antarctica;
            countryTypesByCode["AR"] = CountryTypeId.Argentina;
            countryTypesByCode["AS"] = CountryTypeId.AmericanSamoa;
            countryTypesByCode["AT"] = CountryTypeId.Austria;
            countryTypesByCode["AU"] = CountryTypeId.Australia;
            countryTypesByCode["AW"] = CountryTypeId.Aruba;
            countryTypesByCode["AX"] = CountryTypeId.AlandIslands;
            countryTypesByCode["AZ"] = CountryTypeId.Azerbaijan;
            countryTypesByCode["BA"] = CountryTypeId.BosniaHerzegovina;
            countryTypesByCode["BB"] = CountryTypeId.Barbados;
            countryTypesByCode["BD"] = CountryTypeId.Bangladesh;
            countryTypesByCode["BE"] = CountryTypeId.Belgium;
            countryTypesByCode["BF"] = CountryTypeId.BurkinaFaso;
            countryTypesByCode["BG"] = CountryTypeId.Bulgaria;
            countryTypesByCode["BH"] = CountryTypeId.Bahrain;
            countryTypesByCode["BI"] = CountryTypeId.Burundi;
            countryTypesByCode["BJ"] = CountryTypeId.Benin;
            countryTypesByCode["BM"] = CountryTypeId.Bermuda;
            countryTypesByCode["BN"] = CountryTypeId.BruneiDarussalam;
            countryTypesByCode["BO"] = CountryTypeId.Bolivia;
            countryTypesByCode["BR"] = CountryTypeId.Brazil;
            countryTypesByCode["BS"] = CountryTypeId.Bahamas;
            countryTypesByCode["BT"] = CountryTypeId.Bhutan;
            countryTypesByCode["BV"] = CountryTypeId.BouvetIsland;
            countryTypesByCode["BW"] = CountryTypeId.Botswana;
            countryTypesByCode["BY"] = CountryTypeId.Belarus;
            countryTypesByCode["BZ"] = CountryTypeId.Belize;
            countryTypesByCode["CA"] = CountryTypeId.Canada;
            countryTypesByCode["CC"] = CountryTypeId.CocosKeelingIslands;
            countryTypesByCode["CD"] = CountryTypeId.DemocraticRepublicCongo;
            countryTypesByCode["CF"] = CountryTypeId.CentralAfricanRepublic;
            countryTypesByCode["CG"] = CountryTypeId.Congo;
            countryTypesByCode["CH"] = CountryTypeId.Switzerland;
            countryTypesByCode["CI"] = CountryTypeId.CoteDIvoire;
            countryTypesByCode["CK"] = CountryTypeId.CookIslands;
            countryTypesByCode["CL"] = CountryTypeId.Chile;
            countryTypesByCode["CM"] = CountryTypeId.Cameroon;
            countryTypesByCode["CN"] = CountryTypeId.China;
            countryTypesByCode["CO"] = CountryTypeId.Colombia;
            countryTypesByCode["CR"] = CountryTypeId.CostaRica;
            countryTypesByCode["CS"] = CountryTypeId.SerbiaMontenegro;
            countryTypesByCode["CU"] = CountryTypeId.Cuba;
            countryTypesByCode["CV"] = CountryTypeId.CapeVerde;
            countryTypesByCode["CX"] = CountryTypeId.ChristmasIsland;
            countryTypesByCode["CY"] = CountryTypeId.Cyprus;
            countryTypesByCode["CZ"] = CountryTypeId.CzechRepublic;
            countryTypesByCode["DE"] = CountryTypeId.Germany;
            countryTypesByCode["DJ"] = CountryTypeId.Djibouti;
            countryTypesByCode["DK"] = CountryTypeId.Denmark;
            countryTypesByCode["DM"] = CountryTypeId.Dominica;
            countryTypesByCode["DO"] = CountryTypeId.DominicanRepublic;
            countryTypesByCode["DZ"] = CountryTypeId.Algeria;
            countryTypesByCode["EC"] = CountryTypeId.Ecuador;
            countryTypesByCode["EE"] = CountryTypeId.Estonia;
            countryTypesByCode["EG"] = CountryTypeId.Egypt;
            countryTypesByCode["EH"] = CountryTypeId.WesternSahara;
            countryTypesByCode["ER"] = CountryTypeId.Eritrea;
            countryTypesByCode["ES"] = CountryTypeId.Spain;
            countryTypesByCode["ET"] = CountryTypeId.Ethiopia;
            countryTypesByCode["FI"] = CountryTypeId.Finland;
            countryTypesByCode["FJ"] = CountryTypeId.Fiji;
            countryTypesByCode["FK"] = CountryTypeId.FalklandIslands;
            countryTypesByCode["FM"] = CountryTypeId.FederatedStatesMicronesia;
            countryTypesByCode["FO"] = CountryTypeId.FaroeIslands;
            countryTypesByCode["FR"] = CountryTypeId.France;
            countryTypesByCode["FX"] = CountryTypeId.FranceMetropolitan;
            countryTypesByCode["GA"] = CountryTypeId.Gabon;
            countryTypesByCode["GB"] = CountryTypeId.GreatBritain;
            countryTypesByCode["GD"] = CountryTypeId.Grenada;
            countryTypesByCode["GE"] = CountryTypeId.Georgia;
            countryTypesByCode["GF"] = CountryTypeId.FrenchGuiana;
            countryTypesByCode["GH"] = CountryTypeId.Ghana;
            countryTypesByCode["GI"] = CountryTypeId.Gibraltar;
            countryTypesByCode["GL"] = CountryTypeId.Greenland;
            countryTypesByCode["GM"] = CountryTypeId.Gambia;
            countryTypesByCode["GN"] = CountryTypeId.Guinea;
            countryTypesByCode["GP"] = CountryTypeId.Guadeloupe;
            countryTypesByCode["GQ"] = CountryTypeId.EquatorialGuinea;
            countryTypesByCode["GR"] = CountryTypeId.Greece;
            countryTypesByCode["GS"] = CountryTypeId.SGeorgiaSSandwichIslands;
            countryTypesByCode["GT"] = CountryTypeId.Guatemala;
            countryTypesByCode["GU"] = CountryTypeId.Guam;
            countryTypesByCode["GW"] = CountryTypeId.GuineaBissau;
            countryTypesByCode["GY"] = CountryTypeId.Guyana;
            countryTypesByCode["HK"] = CountryTypeId.HongKong;
            countryTypesByCode["HM"] = CountryTypeId.HeardIslandMcDonaldIslands;
            countryTypesByCode["HN"] = CountryTypeId.Honduras;
            countryTypesByCode["HR"] = CountryTypeId.Croatia;
            countryTypesByCode["HT"] = CountryTypeId.Haiti;
            countryTypesByCode["HU"] = CountryTypeId.Hungary;
            countryTypesByCode["ID"] = CountryTypeId.Indonesia;
            countryTypesByCode["IE"] = CountryTypeId.Ireland;
            countryTypesByCode["IL"] = CountryTypeId.Israel;
            countryTypesByCode["IN"] = CountryTypeId.India;
            countryTypesByCode["IO"] = CountryTypeId.BritishIndianOceanTerritory;
            countryTypesByCode["IQ"] = CountryTypeId.Iraq;
            countryTypesByCode["IR"] = CountryTypeId.Iran;
            countryTypesByCode["IS"] = CountryTypeId.Iceland;
            countryTypesByCode["IT"] = CountryTypeId.Italy;
            countryTypesByCode["JM"] = CountryTypeId.Jamaica;
            countryTypesByCode["JO"] = CountryTypeId.Jordan;
            countryTypesByCode["JP"] = CountryTypeId.Japan;
            countryTypesByCode["KE"] = CountryTypeId.Kenya;
            countryTypesByCode["KG"] = CountryTypeId.Kyrgyzstan;
            countryTypesByCode["KH"] = CountryTypeId.Cambodia;
            countryTypesByCode["KI"] = CountryTypeId.Kiribati;
            countryTypesByCode["KM"] = CountryTypeId.Comoros;
            countryTypesByCode["KN"] = CountryTypeId.SaintKittsNevis;
            countryTypesByCode["KP"] = CountryTypeId.KoreaNorth;
            countryTypesByCode["KR"] = CountryTypeId.KoreaSouth;
            countryTypesByCode["KW"] = CountryTypeId.Kuwait;
            countryTypesByCode["KY"] = CountryTypeId.CaymanIslands;
            countryTypesByCode["KZ"] = CountryTypeId.Kazakhstan;
            countryTypesByCode["LA"] = CountryTypeId.Laos;
            countryTypesByCode["LB"] = CountryTypeId.Lebanon;
            countryTypesByCode["LC"] = CountryTypeId.SaintLucia;
            countryTypesByCode["LI"] = CountryTypeId.Liechtenstein;
            countryTypesByCode["LK"] = CountryTypeId.SriLanka;
            countryTypesByCode["LR"] = CountryTypeId.Liberia;
            countryTypesByCode["LS"] = CountryTypeId.Lesotho;
            countryTypesByCode["LT"] = CountryTypeId.Lithuania;
            countryTypesByCode["LU"] = CountryTypeId.Luxembourg;
            countryTypesByCode["LV"] = CountryTypeId.Latvia;
            countryTypesByCode["LY"] = CountryTypeId.Libya;
            countryTypesByCode["MA"] = CountryTypeId.Morocco;
            countryTypesByCode["MC"] = CountryTypeId.Monaco;
            countryTypesByCode["MD"] = CountryTypeId.Moldova;
            countryTypesByCode["MG"] = CountryTypeId.Madagascar;
            countryTypesByCode["MH"] = CountryTypeId.MarshallIslands;
            countryTypesByCode["MK"] = CountryTypeId.Macedonia;
            countryTypesByCode["ML"] = CountryTypeId.Mali;
            countryTypesByCode["MM"] = CountryTypeId.Myanmar;
            countryTypesByCode["MN"] = CountryTypeId.Mongolia;
            countryTypesByCode["MO"] = CountryTypeId.Macao;
            countryTypesByCode["MP"] = CountryTypeId.NorthernMarianaIslands;
            countryTypesByCode["MQ"] = CountryTypeId.Martinique;
            countryTypesByCode["MR"] = CountryTypeId.Mauritania;
            countryTypesByCode["MS"] = CountryTypeId.Montserrat;
            countryTypesByCode["MT"] = CountryTypeId.Malta;
            countryTypesByCode["MU"] = CountryTypeId.Mauritius;
            countryTypesByCode["MV"] = CountryTypeId.Maldives;
            countryTypesByCode["MW"] = CountryTypeId.Malawi;
            countryTypesByCode["MX"] = CountryTypeId.Mexico;
            countryTypesByCode["MY"] = CountryTypeId.Malaysia;
            countryTypesByCode["MZ"] = CountryTypeId.Mozambique;
            countryTypesByCode["NA"] = CountryTypeId.Namibia;
            countryTypesByCode["NC"] = CountryTypeId.NewCaledonia;
            countryTypesByCode["NE"] = CountryTypeId.Niger;
            countryTypesByCode["NF"] = CountryTypeId.NorfolkIsland;
            countryTypesByCode["NG"] = CountryTypeId.Nigeria;
            countryTypesByCode["NI"] = CountryTypeId.Nicaragua;
            countryTypesByCode["NL"] = CountryTypeId.Netherlands;
            countryTypesByCode["NO"] = CountryTypeId.Norway;
            countryTypesByCode["NP"] = CountryTypeId.Nepal;
            countryTypesByCode["NR"] = CountryTypeId.Nauru;
            countryTypesByCode["NU"] = CountryTypeId.Niue;
            countryTypesByCode["NZ"] = CountryTypeId.NewZealand;
            countryTypesByCode["OM"] = CountryTypeId.Oman;
            countryTypesByCode["PA"] = CountryTypeId.Panama;
            countryTypesByCode["PE"] = CountryTypeId.Peru;
            countryTypesByCode["PF"] = CountryTypeId.FrenchPolynesia;
            countryTypesByCode["PG"] = CountryTypeId.PapuaNewGuinea;
            countryTypesByCode["PH"] = CountryTypeId.Philippines;
            countryTypesByCode["PK"] = CountryTypeId.Pakistan;
            countryTypesByCode["PL"] = CountryTypeId.Poland;
            countryTypesByCode["PM"] = CountryTypeId.SaintPierreMiquelon;
            countryTypesByCode["PN"] = CountryTypeId.Pitcairn;
            countryTypesByCode["PR"] = CountryTypeId.PuertoRico;
            countryTypesByCode["PS"] = CountryTypeId.PalestinianTerritory;
            countryTypesByCode["PT"] = CountryTypeId.Portugal;
            countryTypesByCode["PW"] = CountryTypeId.Palau;
            countryTypesByCode["PY"] = CountryTypeId.Paraguay;
            countryTypesByCode["QA"] = CountryTypeId.Qatar;
            countryTypesByCode["RE"] = CountryTypeId.Reunion;
            countryTypesByCode["RO"] = CountryTypeId.Romania;
            countryTypesByCode["RU"] = CountryTypeId.RussianFederation;
            countryTypesByCode["RW"] = CountryTypeId.Rwanda;
            countryTypesByCode["SA"] = CountryTypeId.SaudiArabia;
            countryTypesByCode["SB"] = CountryTypeId.SolomonIslands;
            countryTypesByCode["SC"] = CountryTypeId.Seychelles;
            countryTypesByCode["SD"] = CountryTypeId.Sudan;
            countryTypesByCode["SE"] = CountryTypeId.Sweden;
            countryTypesByCode["SG"] = CountryTypeId.Singapore;
            countryTypesByCode["SH"] = CountryTypeId.SaintHelena;
            countryTypesByCode["SI"] = CountryTypeId.Slovenia;
            countryTypesByCode["SJ"] = CountryTypeId.SvalbardJanMayen;
            countryTypesByCode["SK"] = CountryTypeId.Slovakia;
            countryTypesByCode["SL"] = CountryTypeId.SierraLeone;
            countryTypesByCode["SM"] = CountryTypeId.SanMarino;
            countryTypesByCode["SN"] = CountryTypeId.Senegal;
            countryTypesByCode["SO"] = CountryTypeId.Somalia;
            countryTypesByCode["SR"] = CountryTypeId.Suriname;
            countryTypesByCode["ST"] = CountryTypeId.SaoTomePrincipe;
            countryTypesByCode["SU"] = CountryTypeId.USSR;
            countryTypesByCode["SV"] = CountryTypeId.ElSalvador;
            countryTypesByCode["SY"] = CountryTypeId.Syria;
            countryTypesByCode["SZ"] = CountryTypeId.Swaziland;
            countryTypesByCode["TC"] = CountryTypeId.TurksCaicosIslands;
            countryTypesByCode["TD"] = CountryTypeId.Chad;
            countryTypesByCode["TF"] = CountryTypeId.FrenchSouthernTerritories;
            countryTypesByCode["TG"] = CountryTypeId.Togo;
            countryTypesByCode["TH"] = CountryTypeId.Thailand;
            countryTypesByCode["TJ"] = CountryTypeId.Tajikistan;
            countryTypesByCode["TK"] = CountryTypeId.Tokelau;
            countryTypesByCode["TL"] = CountryTypeId.TimorLeste;
            countryTypesByCode["TM"] = CountryTypeId.Turkmenistan;
            countryTypesByCode["TN"] = CountryTypeId.Tunisia;
            countryTypesByCode["TO"] = CountryTypeId.Tonga;
            countryTypesByCode["TP"] = CountryTypeId.EastTimor;
            countryTypesByCode["TR"] = CountryTypeId.Turkey;
            countryTypesByCode["TT"] = CountryTypeId.TrinidadTobago;
            countryTypesByCode["TV"] = CountryTypeId.Tuvalu;
            countryTypesByCode["TW"] = CountryTypeId.Taiwan;
            countryTypesByCode["TZ"] = CountryTypeId.Tanzania;
            countryTypesByCode["UA"] = CountryTypeId.Ukraine;
            countryTypesByCode["UG"] = CountryTypeId.Uganda;
            countryTypesByCode["UK"] = CountryTypeId.UnitedKingdom;
            countryTypesByCode["UM"] = CountryTypeId.UnitedStatesMinorOutlyingIslands;
            countryTypesByCode["US"] = CountryTypeId.UnitedStates;
            countryTypesByCode["UY"] = CountryTypeId.Uruguay;
            countryTypesByCode["UZ"] = CountryTypeId.Uzbekistan;
            countryTypesByCode["VA"] = CountryTypeId.VaticanCityState;
            countryTypesByCode["VC"] = CountryTypeId.SaintVincentGrenadines;
            countryTypesByCode["VE"] = CountryTypeId.Venezuela;
            countryTypesByCode["VG"] = CountryTypeId.VirginIslandsBritish;
            countryTypesByCode["VI"] = CountryTypeId.VirginIslandsUS;
            countryTypesByCode["VN"] = CountryTypeId.Vietnam;
            countryTypesByCode["VU"] = CountryTypeId.Vanuatu;
            countryTypesByCode["WF"] = CountryTypeId.WallisFutuna;
            countryTypesByCode["WS"] = CountryTypeId.Samoa;
            countryTypesByCode["XX"] = CountryTypeId.Unknown;
            countryTypesByCode["YE"] = CountryTypeId.Yemen;
            countryTypesByCode["YT"] = CountryTypeId.Mayotte;
            countryTypesByCode["YU"] = CountryTypeId.Yugoslavia;
            countryTypesByCode["ZA"] = CountryTypeId.SouthAfrica;
            countryTypesByCode["ZM"] = CountryTypeId.Zambia;
            countryTypesByCode["ZR"] = CountryTypeId.Zaire;
            countryTypesByCode["ZW"] = CountryTypeId.Zimbabwe;

            dafifToInternalCountries["AA"] = "AW";
            dafifToInternalCountries["AC"] = "AG";
            dafifToInternalCountries["AE"] = "AE";
            dafifToInternalCountries["AF"] = "AF";
            dafifToInternalCountries["AG"] = "DZ";
            dafifToInternalCountries["AJ"] = "AZ";
            dafifToInternalCountries["AL"] = "AL";
            dafifToInternalCountries["AM"] = "AM";
            dafifToInternalCountries["AN"] = "AD";
            dafifToInternalCountries["AO"] = "AO";
            dafifToInternalCountries["AQ"] = "AS";
            dafifToInternalCountries["AR"] = "AR";
            dafifToInternalCountries["AS"] = "AU";
            dafifToInternalCountries["AT"] = "XX";
            dafifToInternalCountries["AU"] = "AT";
            dafifToInternalCountries["AV"] = "AI";
            dafifToInternalCountries["AY"] = "AQ";
            dafifToInternalCountries["BA"] = "BH";
            dafifToInternalCountries["BB"] = "BB";
            dafifToInternalCountries["BC"] = "BW";
            dafifToInternalCountries["BD"] = "BM";
            dafifToInternalCountries["BE"] = "BE";
            dafifToInternalCountries["BF"] = "BS";
            dafifToInternalCountries["BG"] = "BD";
            dafifToInternalCountries["BH"] = "BZ";
            dafifToInternalCountries["BK"] = "BA";
            dafifToInternalCountries["BL"] = "BO";
            dafifToInternalCountries["BM"] = "XX";
            dafifToInternalCountries["BN"] = "BJ";
            dafifToInternalCountries["BO"] = "BY";
            dafifToInternalCountries["BP"] = "SB";
            dafifToInternalCountries["BQ"] = "XX";
            dafifToInternalCountries["BR"] = "BR";
            dafifToInternalCountries["BS"] = "XX";
            dafifToInternalCountries["BT"] = "BT";
            dafifToInternalCountries["BU"] = "BG";
            dafifToInternalCountries["BV"] = "BV";
            dafifToInternalCountries["BX"] = "BN";
            dafifToInternalCountries["BY"] = "BI";
            dafifToInternalCountries["CA"] = "CA";
            dafifToInternalCountries["CB"] = "KH";
            dafifToInternalCountries["CD"] = "TD";
            dafifToInternalCountries["CE"] = "LK";
            dafifToInternalCountries["CF"] = "CG";
            dafifToInternalCountries["CG"] = "CD";
            dafifToInternalCountries["CH"] = "CN";
            dafifToInternalCountries["CI"] = "CL";
            dafifToInternalCountries["CJ"] = "KY";
            dafifToInternalCountries["CK"] = "CC";
            dafifToInternalCountries["CM"] = "CM";
            dafifToInternalCountries["CN"] = "KM";
            dafifToInternalCountries["CO"] = "CO";
            dafifToInternalCountries["CQ"] = "MP";
            dafifToInternalCountries["CR"] = "XX";
            dafifToInternalCountries["CS"] = "CR";
            dafifToInternalCountries["CT"] = "CF";
            dafifToInternalCountries["CU"] = "CU";
            dafifToInternalCountries["CV"] = "CV";
            dafifToInternalCountries["CW"] = "CK";
            dafifToInternalCountries["CY"] = "CY";
            dafifToInternalCountries["DA"] = "DK";
            dafifToInternalCountries["DJ"] = "DJ";
            dafifToInternalCountries["DO"] = "DM";
            dafifToInternalCountries["DQ"] = "XX";
            dafifToInternalCountries["DR"] = "DO";
            dafifToInternalCountries["EC"] = "EC";
            dafifToInternalCountries["EG"] = "EG";
            dafifToInternalCountries["EI"] = "IE";
            dafifToInternalCountries["EK"] = "GQ";
            dafifToInternalCountries["EN"] = "EE";
            dafifToInternalCountries["ER"] = "ER";
            dafifToInternalCountries["ES"] = "SV";
            dafifToInternalCountries["ET"] = "ET";
            dafifToInternalCountries["EU"] = "XX";
            dafifToInternalCountries["EZ"] = "CZ";
            dafifToInternalCountries["FG"] = "GF";
            dafifToInternalCountries["FI"] = "FI";
            dafifToInternalCountries["FJ"] = "FJ";
            dafifToInternalCountries["FK"] = "FK";
            dafifToInternalCountries["FM"] = "FM";
            dafifToInternalCountries["FO"] = "FO";
            dafifToInternalCountries["FP"] = "PF";
            dafifToInternalCountries["FQ"] = "XX";
            dafifToInternalCountries["FR"] = "FR";
            dafifToInternalCountries["FS"] = "TF";
            dafifToInternalCountries["GA"] = "GM";
            dafifToInternalCountries["GB"] = "GA";
            dafifToInternalCountries["GG"] = "GE";
            dafifToInternalCountries["GH"] = "GH";
            dafifToInternalCountries["GI"] = "GI";
            dafifToInternalCountries["GJ"] = "GD";
            dafifToInternalCountries["GK"] = "XX";
            dafifToInternalCountries["GL"] = "GL";
            dafifToInternalCountries["GM"] = "DE";
            dafifToInternalCountries["GO"] = "XX";
            dafifToInternalCountries["GP"] = "GP";
            dafifToInternalCountries["GQ"] = "GU";
            dafifToInternalCountries["GR"] = "GR";
            dafifToInternalCountries["GT"] = "GT";
            dafifToInternalCountries["GV"] = "GN";
            dafifToInternalCountries["GY"] = "GY";
            dafifToInternalCountries["GZ"] = "XX";
            dafifToInternalCountries["HA"] = "HT";
            dafifToInternalCountries["HK"] = "HK";
            dafifToInternalCountries["HM"] = "HM";
            dafifToInternalCountries["HO"] = "HN";
            dafifToInternalCountries["HQ"] = "XX";
            dafifToInternalCountries["HR"] = "HR";
            dafifToInternalCountries["HU"] = "HU";
            dafifToInternalCountries["IC"] = "IS";
            dafifToInternalCountries["ID"] = "ID";
            dafifToInternalCountries["IM"] = "XX";
            dafifToInternalCountries["IN"] = "IN";
            dafifToInternalCountries["IO"] = "IO";
            dafifToInternalCountries["IP"] = "XX";
            dafifToInternalCountries["IR"] = "IR";
            dafifToInternalCountries["IS"] = "IL";
            dafifToInternalCountries["IT"] = "IT";
            dafifToInternalCountries["IV"] = "CI";
            dafifToInternalCountries["IZ"] = "IQ";
            dafifToInternalCountries["JA"] = "JP";
            dafifToInternalCountries["JE"] = "XX";
            dafifToInternalCountries["JM"] = "JM";
            dafifToInternalCountries["JN"] = "SJ";
            dafifToInternalCountries["JO"] = "JO";
            dafifToInternalCountries["JQ"] = "XX";
            dafifToInternalCountries["JU"] = "XX";
            dafifToInternalCountries["KE"] = "KE";
            dafifToInternalCountries["KG"] = "KG";
            dafifToInternalCountries["KN"] = "KP";
            dafifToInternalCountries["KQ"] = "XX";
            dafifToInternalCountries["KR"] = "KI";
            dafifToInternalCountries["KS"] = "KR";
            dafifToInternalCountries["KT"] = "CX";
            dafifToInternalCountries["KU"] = "KW";
            dafifToInternalCountries["KZ"] = "KZ";
            dafifToInternalCountries["LA"] = "LA";
            dafifToInternalCountries["LE"] = "LB";
            dafifToInternalCountries["LG"] = "LV";
            dafifToInternalCountries["LH"] = "LT";
            dafifToInternalCountries["LI"] = "LR";
            dafifToInternalCountries["LO"] = "SK";
            dafifToInternalCountries["LQ"] = "XX";
            dafifToInternalCountries["LS"] = "LI";
            dafifToInternalCountries["LT"] = "LS";
            dafifToInternalCountries["LU"] = "LU";
            dafifToInternalCountries["LY"] = "LY";
            dafifToInternalCountries["MA"] = "MG";
            dafifToInternalCountries["MB"] = "MQ";
            dafifToInternalCountries["MC"] = "MO";
            dafifToInternalCountries["MD"] = "MD";
            dafifToInternalCountries["MF"] = "YT";
            dafifToInternalCountries["MG"] = "MN";
            dafifToInternalCountries["MH"] = "MS";
            dafifToInternalCountries["MI"] = "MW";
            dafifToInternalCountries["MK"] = "MK";
            dafifToInternalCountries["ML"] = "ML";
            dafifToInternalCountries["MN"] = "MC";
            dafifToInternalCountries["MO"] = "MA";
            dafifToInternalCountries["MP"] = "MU";
            dafifToInternalCountries["MQ"] = "XX";
            dafifToInternalCountries["MR"] = "MR";
            dafifToInternalCountries["MT"] = "MT";
            dafifToInternalCountries["MU"] = "OM";
            dafifToInternalCountries["MV"] = "MV";
            dafifToInternalCountries["MX"] = "MX";
            dafifToInternalCountries["MY"] = "MY";
            dafifToInternalCountries["MZ"] = "MZ";
            dafifToInternalCountries["NC"] = "NC";
            dafifToInternalCountries["NE"] = "NU";
            dafifToInternalCountries["NF"] = "NF";
            dafifToInternalCountries["NG"] = "NE";
            dafifToInternalCountries["NH"] = "VU";
            dafifToInternalCountries["NI"] = "NG";
            dafifToInternalCountries["NL"] = "NL";
            dafifToInternalCountries["NO"] = "NO";
            dafifToInternalCountries["NP"] = "NP";
            dafifToInternalCountries["NR"] = "NR";
            dafifToInternalCountries["NS"] = "SR";
            dafifToInternalCountries["NT"] = "AN";
            dafifToInternalCountries["NU"] = "NI";
            dafifToInternalCountries["NZ"] = "NZ";
            dafifToInternalCountries["PA"] = "PY";
            dafifToInternalCountries["PC"] = "PN";
            dafifToInternalCountries["PE"] = "PE";
            dafifToInternalCountries["PF"] = "XX";
            dafifToInternalCountries["PG"] = "XX";
            dafifToInternalCountries["PK"] = "PK";
            dafifToInternalCountries["PL"] = "PL";
            dafifToInternalCountries["PM"] = "PA";
            dafifToInternalCountries["PO"] = "PT";
            dafifToInternalCountries["PP"] = "PG";
            dafifToInternalCountries["PS"] = "PW";
            dafifToInternalCountries["PU"] = "GW";
            dafifToInternalCountries["QA"] = "QA";
            dafifToInternalCountries["RE"] = "RE";
            dafifToInternalCountries["RM"] = "MH";
            dafifToInternalCountries["RO"] = "RO";
            dafifToInternalCountries["RP"] = "PH";
            dafifToInternalCountries["RQ"] = "PR";
            dafifToInternalCountries["RS"] = "RU";
            dafifToInternalCountries["RW"] = "RW";
            dafifToInternalCountries["SA"] = "SA";
            dafifToInternalCountries["SB"] = "PM";
            dafifToInternalCountries["SC"] = "KN";
            dafifToInternalCountries["SE"] = "SC";
            dafifToInternalCountries["SF"] = "SC";
            dafifToInternalCountries["SG"] = "SN";
            dafifToInternalCountries["SH"] = "SH";
            dafifToInternalCountries["SI"] = "SI";
            dafifToInternalCountries["SL"] = "SL";
            dafifToInternalCountries["SM"] = "SM";
            dafifToInternalCountries["SN"] = "SG";
            dafifToInternalCountries["SO"] = "SO";
            dafifToInternalCountries["SP"] = "ES";
            dafifToInternalCountries["ST"] = "LC";
            dafifToInternalCountries["SU"] = "SD";
            dafifToInternalCountries["SV"] = "SJ";
            dafifToInternalCountries["SW"] = "SE";
            dafifToInternalCountries["SY"] = "SY";
            dafifToInternalCountries["SZ"] = "CH";
            dafifToInternalCountries["TD"] = "TT";
            dafifToInternalCountries["TE"] = "XX";
            dafifToInternalCountries["TH"] = "TH";
            dafifToInternalCountries["TI"] = "TJ";
            dafifToInternalCountries["TK"] = "TC";
            dafifToInternalCountries["TL"] = "TK";
            dafifToInternalCountries["TN"] = "TO";
            dafifToInternalCountries["TO"] = "TG";
            dafifToInternalCountries["TP"] = "ST";
            dafifToInternalCountries["TS"] = "TN";
            dafifToInternalCountries["TT"] = "TP";
            dafifToInternalCountries["TU"] = "TR";
            dafifToInternalCountries["TV"] = "TV";
            dafifToInternalCountries["TW"] = "TW";
            dafifToInternalCountries["TX"] = "TM";
            dafifToInternalCountries["TZ"] = "TZ";
            dafifToInternalCountries["UG"] = "UG";
            dafifToInternalCountries["UK"] = "UK";
            dafifToInternalCountries["UP"] = "UA";
            dafifToInternalCountries["US"] = "US";
            dafifToInternalCountries["UV"] = "BF";
            dafifToInternalCountries["UY"] = "UY";
            dafifToInternalCountries["UZ"] = "UZ";
            dafifToInternalCountries["VC"] = "VC";
            dafifToInternalCountries["VE"] = "VE";
            dafifToInternalCountries["VI"] = "VG";
            dafifToInternalCountries["VM"] = "VN";
            dafifToInternalCountries["VQ"] = "VI";
            dafifToInternalCountries["VT"] = "VA";
            dafifToInternalCountries["WA"] = "NA";
            dafifToInternalCountries["WE"] = "XX";
            dafifToInternalCountries["WF"] = "WF";
            dafifToInternalCountries["WI"] = "EH";
            dafifToInternalCountries["WQ"] = "XX";
            dafifToInternalCountries["WS"] = "WS";
            dafifToInternalCountries["WZ"] = "SZ";
            dafifToInternalCountries["YI"] = "CS";
            dafifToInternalCountries["YM"] = "YE";
            dafifToInternalCountries["ZA"] = "ZM";
            dafifToInternalCountries["ZI"] = "ZW";
        }

        private void CheckCompatibilityVersion(string key, int val) {
            RegistryKey currUser = Registry.CurrentUser.CreateSubKey(key);
            int oldVal;
            try {
                oldVal = (int)currUser.GetValue("Version", -1);
                if (oldVal != val) {
                    Registry.CurrentUser.DeleteSubKey(key, false);
                    currUser = Registry.CurrentUser.CreateSubKey(key);
                    currUser.SetValue("Version", val);
                }
            }
            catch {
                Registry.CurrentUser.DeleteSubKey(key, false);
            }
        }
        #endregion

        #region Attributes
        #endregion
    }

    public class AirspaceSorter : IComparer {
    
        int IComparer.Compare(object x, object y) {
            // TODO:  Add AirspaceSorter.Compare implementation
            int result;
            AirspaceClasses p1 = ((AirspaceElement)x).AirspaceClass;
            AirspaceClasses p2 = ((AirspaceElement)y).AirspaceClass;
            AirspaceTypes t1 = ((AirspaceElement)x).AirspaceType;
            AirspaceTypes t2 = ((AirspaceElement)y).AirspaceType;

            if (t1 == t2) {
                if (p1 == p2) {
                    result = 0;
                }
                else if (p1 > p2) {
                    result = -1;
                }
                else {
                    result = 1;
                }
            }
            else {
                if (t1 > t2) {
                    result = -1;
                }
                else {
                    result = 1;
                }
            }
            return result;
        }
    }

    public class WayPointSorter : IComparer {
        public int Compare(object x, object y) {
            // TODO:  Add WayPointSorter.Compare implementation
            return ((WayPoint)x).longName.CompareTo(((WayPoint)y).longName);
        }
    }

    class ListViewItemSorter : IComparer {
        private int col;
        private bool ascending;

        public ListViewItemSorter() :
            this(0, true){}
        public ListViewItemSorter(int column) :
            this(column, true){}
        public ListViewItemSorter(int column, bool asc) {
            col = column;
            this.ascending = asc;
        }

        public int Compare(object x, object y) {
            string s1, s2;
            int cmp;
            s1 = ((ListViewItem)x).SubItems[col].Text;
            s2 = ((ListViewItem)y).SubItems[col].Text;
            cmp = s1.CompareTo(s2);
            if (!ascending) {
                cmp *= -1;
            }

            return cmp;
        }

        public int Col {
            get {
                return col;
            }
            set {
                if (value == col) {
                    ascending = !ascending;
                }
                else {
                    col = value;
                    ascending = true;
                }
            }
        }

        public bool Ascending {
            get {
                return ascending;
            }
            set {
                ascending = value;
            }
        }
    }

}
