// name   : Diagram.cs
// author : Harald Maier
// date   : 03.09.2003
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Collections;
using System.Windows.Forms;

using SoaringDotNET.GUI;
using SoaringDotNET.Data;

namespace SoaringDotNET.Controls
{
	/// <summary>
	/// 
	/// </summary>
	public class Diagram : System.Windows.Forms.UserControl
	{
        private enum GraphTypes {Barogram = 1, GPSHeight = 2, Vario = 4, Speed = 8, Engine = 16};

        private Flight currentFlight = null;
        private System.Windows.Forms.ContextMenu contextMenu;
        private System.Windows.Forms.MenuItem menuItemBaro;
        private System.Windows.Forms.MenuItem menuItemGPS;
        private System.Windows.Forms.MenuItem menuItemVario;
        private System.Windows.Forms.MenuItem menuItemSpeed;
        private System.Windows.Forms.MenuItem menuItemEngine;
        private SoaringDotNet app;
        private GraphTypes graphs;
        private Graphics bmpGraphics;
        private Bitmap graphicsBmp = null;
        private TextureBrush graphicsBrush = null;
        private Pen graphicsPen = null;
        private Pen markerPen;
        private Font font;
        private Pen baroPen = Pens.Blue;
        private Pen gpsPen = Pens.LightBlue;
        private Pen varioPen = Pens.Green;
        private Pen speedPen = Pens.Red;
        private Pen enginePen = Pens.Purple;
        private Pen turnPointPen = Pens.Goldenrod;
        private Pen dashPen;
        private Hashtable data;
        private int graphHeight, graphWidht, graphX, graphY;
        private System.Windows.Forms.MenuItem menuItem1;
        private System.Windows.Forms.MenuItem menuItemStartMarker;
        private System.Windows.Forms.MenuItem menuItemStopMarker;
        private int oldIdx;
        private int smoothFactor;

        public Diagram()
		{
			// 
			// TODO: Add constructor logic here
			//
            InitializeComponent();
            graphs = GraphTypes.Barogram;

            font = new Font("Arial", 8.0f);
            dashPen = new Pen(Color.Black);
            dashPen.DashStyle = DashStyle.Dash;

            markerPen = new Pen(Color.DarkMagenta);
            markerPen.DashStyle = DashStyle.Dash;

            oldIdx = -1;
            smoothFactor = 0;
		}
	
        #region Public functions
        public void RedrawDiagram(Graphics g) {
            float startLegend;
            float labelWidth;
            SizeF textSize;
            int i, n;
            int x, y, x2, y2;
            double roundLevel = 0;
            double res, range;
            double [] dataSet;
            Hashtable stats;
            double minimum, maximum;
            int minTime, maxTime, diffTime;
            string t, yLabel1, yLabel2;
            string []labels = {"Barogram", "GPS Height", "Variometer", "Speed", "Engine noise"};
            string []keys = {"Baro", "GPS", "Vario", "Speed", "Engine"};
            GraphTypes []displayTypes = {GraphTypes.Barogram, GraphTypes.GPSHeight, GraphTypes.Vario, GraphTypes.Speed, GraphTypes.Engine};

            Pen []pens = {baroPen, gpsPen, varioPen, speedPen, enginePen};
            Pen pen;

            graphicsBmp = new Bitmap(Width, Height, CreateGraphics());
            bmpGraphics = Graphics.FromImage(graphicsBmp);

            bmpGraphics.Clear(this.BackColor);
            
            // draw legend
            labelWidth = 0;
            foreach (string s in labels) {
                labelWidth += 60; // line + space
                textSize = bmpGraphics.MeasureString(s, font);
                labelWidth += textSize.Width + 10; // text + space
            }
            startLegend = (Width - labelWidth) / 2;

            for (i = 0; i < labels.Length; i++) {
                bmpGraphics.DrawLine(pens[i], startLegend, Height - 14, startLegend + 50, Height - 14);
                startLegend += 60;
                bmpGraphics.DrawString(labels[i], font, Brushes.Black, startLegend, Height - 20);
                textSize = bmpGraphics.MeasureString(labels[i], font);
                startLegend += textSize.Width + 10;
            }

            if (currentFlight != null) {
                i = 0;
                if ((graphs & GraphTypes.Barogram) != 0) {
                    i++;
                }
                if ((graphs & GraphTypes.Engine) != 0) {
                    i++;
                }
                if ((graphs & GraphTypes.GPSHeight) != 0) {
                    i++;
                }
                if ((graphs & GraphTypes.Speed) != 0) {
                    i++;
                }
                if ((graphs & GraphTypes.Vario) != 0) {
                    i++;
                }

                graphHeight = Height - 90; // room for top and label on bottom
                graphWidht = Width  - 20; // room for left and right space
                graphY = Height - 70;
                // if only one curve, show y labels
                if (i == 1) {
                    switch (graphs) {
                    case GraphTypes.Barogram:
                        stats = GetMinMax((double[])data["Baro"]);
                        minimum = (double) stats["Min"];
                        maximum = (double) stats["Max"];
                        yLabel1 = "Height";
                        yLabel2 = "[m]";
                        if (maximum - minimum < 2000) {
                            roundLevel = 100;
                        }
                        else if (maximum - minimum < 3000) {
                            roundLevel = 200;
                        }
                        else {
                            roundLevel = 500;
                        }
                        break;
                    case GraphTypes.Engine:
                        stats = GetMinMax((double[])data["Engine"]);
                        minimum = (double) stats["Min"];
                        maximum = (double) stats["Max"];
                        if (maximum - minimum < 200) {
                            roundLevel = 10;
                        }
                        else {
                            roundLevel = 20;
                        }
                        yLabel1 = "Noise";
                        yLabel2 = "[]";
                        break;
                    case GraphTypes.GPSHeight:
                        stats = GetMinMax((double[])data["GPS"]);
                        minimum = (double) stats["Min"];
                        maximum = (double) stats["Max"];
                        if (maximum - minimum < 2000) {
                            roundLevel = 100;
                        }
                        else if (maximum - minimum < 3000) {
                            roundLevel = 200;
                        }
                        else {
                            roundLevel = 500;
                        }
                        yLabel1 = "Height";
                        yLabel2 = "[m]";
                        break;
                    case GraphTypes.Speed:
                        stats = GetMinMax((double[])data["Speed"]);
                        if ((double) stats["Max"] < 200) {
                            roundLevel = 5;
                        }
                        else {
                            roundLevel = 10;
                        }
                        stats["Min"] = 0.0;
                        yLabel1 = "Speed";
                        yLabel2 = "[km/h]";
                        break;
                    case GraphTypes.Vario:
                        stats = GetMinMax((double[])data["Vario"]);
                        minimum = (double) stats["Min"];
                        maximum = (double) stats["Max"];
                        if (maximum - minimum < 3) {
                            roundLevel = 0.1;
                        }
                        else if (maximum - minimum < 5) {
                            roundLevel = 0.2;
                        }
                        else if (maximum - minimum < 7) {
                            roundLevel = 0.5;
                        }
                        else {
                            roundLevel = 1.0;
                        }
                        yLabel1 = "Vario";
                        yLabel2 = "[m/s]";
                        break;
                    default:
                        stats = new Hashtable();
                        stats["Min"] = 0.0;
                        stats["Max"] = 0.0;
                        yLabel1 = "";
                        yLabel2 = "";
                        break;
                    }

                    minimum = (double) stats["Min"];
                    maximum = (double) stats["Max"];
                    minimum = minimum >= 0 ? minimum - (minimum % roundLevel) : minimum - ((minimum % roundLevel) + roundLevel);
                    maximum += (roundLevel - (maximum % roundLevel));
                    
                    t = string.Format("{0}", minimum);
                    textSize = bmpGraphics.MeasureString(t, font);
                    t = string.Format("{0}", maximum);
                    textSize.Width = Math.Max(textSize.Width, bmpGraphics.MeasureString(t, font).Width);
                    graphX = (int)Math.Ceiling(textSize.Width) + 10;
                    graphWidht -= (graphX);
                }
                else if (i == 2 && (graphs & GraphTypes.Barogram) != 0 && (graphs & GraphTypes.GPSHeight) != 0) {
                    stats = GetMinMax((double[])data["Baro"]);
                    minimum = (double) stats["Min"];
                    maximum = (double) stats["Max"];
                    stats = GetMinMax((double[])data["GPS"]);
                    minimum = Math.Min((double) stats["Min"], minimum);
                    maximum = Math.Max((double) stats["Max"], maximum);
                    if (maximum - minimum < 2000) {
                        roundLevel = 100;
                    }
                    else if (maximum - minimum < 3000) {
                        roundLevel = 200;
                    }
                    else {
                        roundLevel = 500;
                    }
                    yLabel1 = "Height";
                    yLabel2 = "[m]";
                    minimum = minimum >= 0 ? minimum - (minimum % roundLevel) : minimum - ((minimum % roundLevel) + roundLevel);
                    maximum += (roundLevel - (maximum % roundLevel));
                    
                    t = string.Format("{0}", minimum);
                    textSize = bmpGraphics.MeasureString(t, font);
                    t = string.Format("{0}", maximum);
                    textSize.Width = Math.Max(textSize.Width, bmpGraphics.MeasureString(t, font).Width);
                    graphX = (int)Math.Ceiling(textSize.Width) + 10;
                    graphWidht -= (graphX);
                }
                else {
                    graphX = 20;
                    maximum = minimum = 0;
                    yLabel1 = "";
                    yLabel2 = "";
                }

                // draw axis
                bmpGraphics.DrawLine(Pens.Black, graphX, graphY, graphX + graphWidht, graphY);
                bmpGraphics.DrawLine(Pens.Black, graphX, graphY, graphX, 20);

                range = maximum - minimum;
                // draw y ticks
                if (range > 0) {
                    res = minimum;
                    while (res <= maximum) {
                        t = string.Format("{0}", res);
                        y = (int)(graphY - ((res - minimum) * graphHeight / range));
                        bmpGraphics.DrawLine(Pens.Black, graphX - 5, y, graphX, y);
                        bmpGraphics.DrawLine(dashPen, graphX, y, graphX + graphWidht, y);
                        textSize = bmpGraphics.MeasureString(t, font);
                        bmpGraphics.DrawString(t, font, Brushes.Black, graphX - 7 - textSize.Width,
                            (y - (textSize.Height / 2)));
                        res += roundLevel;
                    }

                    textSize = bmpGraphics.MeasureString(yLabel1 + " " + yLabel2, font);
                    bmpGraphics.DrawString(yLabel1 + " " + yLabel2, font, Brushes.Black, 5, 2);
                }

                // draw x ticks
                minTime = currentFlight[0].fixTime;
                maxTime = currentFlight[currentFlight.Count - 1].fixTime;
                diffTime = currentFlight.RecordingTime;
                // 1h
                if (diffTime < 3600) {
                    n = 300;
                }
                else if (diffTime < 7200) {
                    // 2h
                    n = 600;
                }
                else if (diffTime < 18000) {
                    // 5h
                    n = 1200;
                }
                else if (diffTime < 28800) {
                    // 8h
                    n = 1800;
                }
                else {
                    n = 3600;
                }

                i = minTime + (n - (minTime % n));
                while (i <= maxTime) {
                    t = SoaringDotNET.Data.Time.ToTimeString(i);
                    textSize = bmpGraphics.MeasureString(t, font);
                    x = graphX + ((i - minTime) * graphWidht / diffTime);
                    bmpGraphics.DrawLine(Pens.Black, x, graphY, x, graphY + 5);
                    bmpGraphics.DrawLine(dashPen, x, graphY, x, 20);
                    bmpGraphics.DrawString(t, font, Brushes.Black, x - (textSize.Width / 2), graphY + 10);
                    i += n;
                }

                t = "Time [UTC]";
                textSize = bmpGraphics.MeasureString(t, font);
                bmpGraphics.DrawString(t, font, Brushes.Black, Width - textSize.Width, graphY + 40);

                // draw all curves
                for (i = 0; i < displayTypes.Length; i++) {
                    if ((graphs & displayTypes[i]) != 0) {
                        dataSet = (double [])data[keys[i]];
                        stats = GetMinMax(dataSet);
                        minimum = (double) stats["Min"];
                        maximum = (double) stats["Max"];
                        if (smoothFactor > 0) {
                            dataSet = MovingAverage((double [])data[keys[i]], smoothFactor);
                        }
                        // get range for both elevations
                        if ((graphs & (GraphTypes.Barogram | GraphTypes.GPSHeight)) == graphs) {
                            stats = GetMinMax(displayTypes[i] == GraphTypes.Barogram ? (double [])data["GPS"] : (double [])data["Baro"]);
                            maximum = Math.Max((double) stats["Max"], maximum);
                            minimum = Math.Min((double) stats["Min"], minimum);
                        }

                        switch (displayTypes[i]) {
                        case GraphTypes.Barogram:
                        case GraphTypes.GPSHeight:
                            if (maximum - minimum < 2000) {
                                roundLevel = 100;
                            }
                            else if (maximum - minimum < 3000) {
                                roundLevel = 200;
                            }
                            else {
                                roundLevel = 500;
                            }
                            break;
                        case GraphTypes.Engine:
                            if (maximum - minimum < 200) {
                                roundLevel = 10;
                            }
                            else {
                                roundLevel = 20;
                            }
                            break;
                        case GraphTypes.Speed:
                            if (maximum < 200) {
                                roundLevel = 5;
                            }
                            else {
                                roundLevel = 10;
                            }
                            break;
                        case GraphTypes.Vario:
                            if (maximum - minimum < 3) {
                                roundLevel = 0.1;
                            }
                            else if (maximum - minimum < 5) {
                                roundLevel = 0.2;
                            }
                            else if (maximum - minimum < 7) {
                                roundLevel = 0.5;
                            }
                            else {
                                roundLevel = 1.0;
                            }
                            break;
                        }

                        minimum = minimum >= 0 ? minimum - (minimum % roundLevel) : minimum - ((minimum % roundLevel) + roundLevel);
                        maximum += (roundLevel - (maximum % roundLevel));
                        range = maximum - minimum;
                        pen = pens[i];
                        if (dataSet.Length > 0) {
                            x = graphX + ((currentFlight[0].fixTime - minTime) * graphWidht / diffTime);
                            y = (int)(graphY - ((dataSet[0] - minimum) * graphHeight / range));
                            for (n = 1; n < dataSet.Length - 1; n++) {
                                x2 = graphX + ((currentFlight[n + 1].fixTime - minTime) * graphWidht / diffTime);
                                y2 = (int)(graphY - ((dataSet[n + 1] - minimum) * graphHeight / range));
                                bmpGraphics.DrawLine(pen, x, y, x2, y2);
                                x = x2;
                                y = y2;
                            }
                        }
                    }
                }

                // draw task points
                foreach (WayPoint wp in currentFlight.currentTask) {
                    if (wp.reachPoint.fixTime >= 0) {
                        x = graphX + ((wp.reachPoint.fixTime - minTime) * graphWidht / diffTime);
                        t = wp.longName;
                        textSize = bmpGraphics.MeasureString(t, font);
                        bmpGraphics.DrawLine(turnPointPen, x, graphY, x, 20);
                        bmpGraphics.DrawString(t, font, Brushes.Black, x - (textSize.Width / 2), graphY + 25);
                    }
                }
            }
            else {
                oldIdx = -1;
                textSize = bmpGraphics.MeasureString("Sorry, no current flight to display!", Font);
                bmpGraphics.DrawString("Sorry, no current flight to display!", Font, Brushes.Black, (Width - textSize.Width) / 2,
                    (Height - textSize.Height) / 2);
            }
            // show diagram
            g.DrawImage(graphicsBmp, 0, 0);

            // create new brush an pen objects for layered drawing (for floating objects)
            graphicsBrush = new TextureBrush(graphicsBmp);
            graphicsPen = new Pen(graphicsBrush, 1);

            // draw marker
            DrawMarker(false);
            DrawFlightMarkers(true);
            DrawFlightMarkers(false);
            System.GC.Collect();
        }

        public void Switch(Flight f) {
            if (currentFlight != f) {
                currentFlight = f;
                if (currentFlight != null) {
                    data = GetFlightData();
                }
                oldIdx = 0;
                RedrawDiagram(CreateGraphics());
            }
        }

        public void OnZoomIn() {
            int newWidth = Width / 2;
            Width = Width + newWidth;
            RedrawDiagram(CreateGraphics());
        }

        public void OnZoomOut() {
            Width = Math.Max(Parent.Width, Width - (Width / 4));
            RedrawDiagram(CreateGraphics());
        }

        #endregion

        #region Protected functions
        protected override void OnPaint(System.Windows.Forms.PaintEventArgs e) {
            // TODO:  Add Diagram.OnPaint implementation
            RedrawDiagram(e.Graphics);
        }

        protected override void OnKeyDown(System.Windows.Forms.KeyEventArgs e) {
            // TODO:  Add Diagram.OnKeyDown implementation
            base.OnKeyDown (e);
            int idx = oldIdx;

            if (currentFlight != null) {
                switch (e.KeyCode) {
                case Keys.Right:
                    if (oldIdx < currentFlight.Count - 1) {
                        idx = oldIdx + 1;
                    }
                    break;
                case Keys.Left:
                    if (oldIdx > 0) {
                        idx = oldIdx - 1;
                    }
                    break;
                case Keys.PageDown:
                    idx = Math.Min(oldIdx + 10, currentFlight.Count - 1);
                    break;
                case Keys.PageUp:
                    idx = Math.Max(oldIdx - 10, 0);
                    break;
                case Keys.Home:
                    idx = 0;
                    break;
                case Keys.End:
                    idx = currentFlight.Count -1;
                    break;
                }

                DrawMarker(true);
                oldIdx = idx;
                DrawMarker(false);
                FlightPoint p1 = currentFlight[idx];
                FlightPoint p2 = currentFlight[idx < currentFlight.Count - 1 ? idx + 1 : idx];
                app.UpdatePosInfo(p1.Latitude, p1.Longitude, p1.fixTime, p1.Elevation, p1.speedKMH(p2), p1.varioMPS(p2));
            }
        }

        protected override bool IsInputKey(Keys keyData) {
            // TODO:  Add Diagram.IsInputKey implementation
            bool ok = false;
            switch (keyData) {
            case Keys.Right:
            case Keys.Left:
            case Keys.PageDown:
            case Keys.PageUp:
            case Keys.Home:
            case Keys.End:
                ok = true;
                break;
            }
            return ok;
        }

        protected override void OnMouseEnter(EventArgs e) {
            // TODO:  Add Explorer.OnMouseEnter implementation
            base.OnMouseEnter (e);
            Focus();
        }
        #endregion

        #region Private Functions
        private void InitializeComponent() {
            this.contextMenu = new System.Windows.Forms.ContextMenu();
            this.menuItemBaro = new System.Windows.Forms.MenuItem();
            this.menuItemGPS = new System.Windows.Forms.MenuItem();
            this.menuItemVario = new System.Windows.Forms.MenuItem();
            this.menuItemSpeed = new System.Windows.Forms.MenuItem();
            this.menuItemEngine = new System.Windows.Forms.MenuItem();
            this.menuItem1 = new System.Windows.Forms.MenuItem();
            this.menuItemStartMarker = new System.Windows.Forms.MenuItem();
            this.menuItemStopMarker = new System.Windows.Forms.MenuItem();
            // 
            // contextMenu
            // 
            this.contextMenu.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
                                                                                        this.menuItemBaro,
                                                                                        this.menuItemGPS,
                                                                                        this.menuItemVario,
                                                                                        this.menuItemSpeed,
                                                                                        this.menuItemEngine,
                                                                                        this.menuItem1,
                                                                                        this.menuItemStartMarker,
                                                                                        this.menuItemStopMarker});
            this.contextMenu.Popup += new System.EventHandler(this.OnPopupContextMenu);
            // 
            // menuItemBaro
            // 
            this.menuItemBaro.Index = 0;
            this.menuItemBaro.Text = "Barogram";
            this.menuItemBaro.Click += new System.EventHandler(this.OnBaroClicked);
            // 
            // menuItemGPS
            // 
            this.menuItemGPS.Index = 1;
            this.menuItemGPS.Text = "GPS Height";
            this.menuItemGPS.Click += new System.EventHandler(this.OnGpsClicked);
            // 
            // menuItemVario
            // 
            this.menuItemVario.Index = 2;
            this.menuItemVario.Text = "Variometer";
            this.menuItemVario.Click += new System.EventHandler(this.OnVarioClicked);
            // 
            // menuItemSpeed
            // 
            this.menuItemSpeed.Index = 3;
            this.menuItemSpeed.Text = "Speed";
            this.menuItemSpeed.Click += new System.EventHandler(this.OnSpeedClicked);
            // 
            // menuItemEngine
            // 
            this.menuItemEngine.Index = 4;
            this.menuItemEngine.Text = "Engine noise level";
            this.menuItemEngine.Click += new System.EventHandler(this.OnEngineClicked);
            // 
            // menuItem1
            // 
            this.menuItem1.Index = 5;
            this.menuItem1.Text = "-";
            // 
            // menuItemStartMarker
            // 
            this.menuItemStartMarker.Index = 6;
            this.menuItemStartMarker.Text = "Set start marker";
            this.menuItemStartMarker.Click += new System.EventHandler(this.OnStartMarkerClicked);
            // 
            // menuItemStopMarker
            // 
            this.menuItemStopMarker.Index = 7;
            this.menuItemStopMarker.Text = "Set stop marker";
            this.menuItemStopMarker.Click += new System.EventHandler(this.OnStopMarkerClicked);
            // 
            // Diagram
            // 
            this.BackColor = System.Drawing.Color.White;
            this.ClientSize = new System.Drawing.Size(352, 232);
            this.ContextMenu = this.contextMenu;
            this.Name = "Diagram";

        }

        private void OnPopupContextMenu(object sender, System.EventArgs e) {
            menuItemBaro.Checked = (graphs & GraphTypes.Barogram) != 0;
            menuItemGPS.Checked = (graphs & GraphTypes.GPSHeight) != 0;
            menuItemVario.Checked = (graphs & GraphTypes.Vario) != 0;
            menuItemSpeed.Checked = (graphs & GraphTypes.Speed) != 0;
            menuItemEngine.Checked = (graphs & GraphTypes.Engine) != 0;
            menuItemStartMarker.Enabled = currentFlight != null;
            menuItemStopMarker.Enabled = currentFlight != null;
        }

        private void OnBaroClicked(object sender, System.EventArgs e) {
            if ((graphs & GraphTypes.Barogram) != 0) {
                graphs &= ~GraphTypes.Barogram;
            }
            else {
                graphs |= GraphTypes.Barogram;
            }
            RedrawDiagram(CreateGraphics());
        }

        private void OnGpsClicked(object sender, System.EventArgs e) {
            if ((graphs & GraphTypes.GPSHeight) != 0) {
                graphs &= ~GraphTypes.GPSHeight;
            }
            else {
                graphs |= GraphTypes.GPSHeight;
            }
            RedrawDiagram(CreateGraphics());
        }

        private void OnVarioClicked(object sender, System.EventArgs e) {
            if ((graphs & GraphTypes.Vario) != 0) {
                graphs &= ~GraphTypes.Vario;
            }
            else {
                graphs |= GraphTypes.Vario;
            }
            RedrawDiagram(CreateGraphics());
        }

        private void OnSpeedClicked(object sender, System.EventArgs e) {
            if ((graphs & GraphTypes.Speed) != 0) {
                graphs &= ~GraphTypes.Speed;
            }
            else {
                graphs |= GraphTypes.Speed;
            }
            RedrawDiagram(CreateGraphics());
        }

        private void OnEngineClicked(object sender, System.EventArgs e) {
            if ((graphs & GraphTypes.Engine) != 0) {
                graphs &= ~GraphTypes.Engine;
            }
            else {
                graphs |= GraphTypes.Engine;
            }
            RedrawDiagram(CreateGraphics());
        }

        private void OnStartMarkerClicked(object sender, System.EventArgs e) {
            DrawFlightMarkers(true);
            currentFlight.Start = oldIdx;
            DrawFlightMarkers(false);
        }

        private void OnStopMarkerClicked(object sender, System.EventArgs e) {
            DrawFlightMarkers(true);
            currentFlight.Stop = oldIdx;
            DrawFlightMarkers(false);
        }


        private Hashtable GetFlightData() {
            Hashtable data = new Hashtable();
            double [] baroData = null, gpsData = null, varioData = null, speedData = null, engineData = null;
            int n = currentFlight.Count;
            int nMinus1 = n - 1;
            int i;
            FlightPoint p1, p2;

            baroData = new double[n];
            data.Add("Baro", baroData);
            gpsData = new double[n];
            data.Add("GPS", gpsData);
            varioData = new double[n];
            data.Add("Vario", varioData);
            speedData = new double[n];
            data.Add("Speed", speedData);
            engineData = new double[n];
            data.Add("Engine", engineData);

            for (i = 0; i < n; i++) {
                p1 = currentFlight[i];
                p2 = (i < nMinus1 ?  currentFlight[i + 1] : currentFlight[i]);

                baroData[i] = p1.pAlt;
                gpsData[i] = p1.gnssAlt;
                varioData[i] = p1.varioMPS(p2);
                speedData[i] = p1.speedKMH(p2);
                engineData[i] = p1.engineNoiseLevel;
            }

            return data;
        }

        private Hashtable GetMinMax(double [] values) {
            Hashtable results = new Hashtable();
            double min, max;
            int i;

            if (values.Length > 0) {
                min = values[0];
                max = values[0];
            }
            else {
                min = max = 0;
            }

            for (i = 1; i < values.Length; i++) {
                min = Math.Min(min, values[i]);
                max = Math.Max(max, values[i]);
            }

            results.Add("Min", min);
            results.Add("Max", max);

            return results;
        }

        private double [] MovingAverage(double [] values, int range) {
            int N = values.Length;
            int i, j, idx, div = range + range + 1;
            double [] results = new double[N];
            double res;

            for (i = 0; i < N; i++) {
                res = 0;
                for (j = i + range; j >= i - range; j--) {
                    idx = Math.Abs(j);
                    if (idx >= N) {
                        idx = Math.Abs(N - idx - 1);
                    }
                    res += values[idx];
                }
                results[i] = res / div;
            }
            return results;
        }

        private void DrawMarker(bool erase) {
            if (oldIdx != -1) {
                int x, x1, x2;
                // check not overwrite existing markers
                x = graphX + ((currentFlight[oldIdx].fixTime - currentFlight[0].fixTime) * graphWidht / currentFlight.RecordingTime);
                x1 = currentFlight.Start != -1 ? graphX + ((currentFlight[currentFlight.Start].fixTime - currentFlight[0].fixTime) * graphWidht / currentFlight.RecordingTime) : -1;
                x2 = currentFlight.Stop != -1 ? graphX + ((currentFlight[currentFlight.Stop].fixTime - currentFlight[0].fixTime) * graphWidht / currentFlight.RecordingTime) : -1;
                if (x != x1 && x != x2) {
                    Graphics g = CreateGraphics();
                    g.DrawLine(erase ? graphicsPen : markerPen, x, 20, x, graphY);
                }
            }
        }

        private void DrawFlightMarkers(bool erase) {
            if (currentFlight != null) {
                Graphics g = CreateGraphics();
                int x1 = 0, x2 = 0;
                Point [] p = new Point[4];

                if (currentFlight.Start != -1) {
                    x1 = graphX + ((currentFlight[currentFlight.Start].fixTime - currentFlight[0].fixTime) * graphWidht / currentFlight.RecordingTime);
                    g.DrawLine(erase ? graphicsPen : markerPen, x1, 20, x1, graphY);
                    
                    p[0].X= x1;
                    p[0].Y = 17;
                    p[1].X = x1 - 4;
                    p[1].Y = 11;
                    p[2].X = x1 + 5;
                    p[2].Y = 11;
                    p[3].X = x1;
                    p[3].Y = 17;

                    g.FillPolygon(erase ? graphicsBrush : Brushes.LightGreen, p);
                }

                if (currentFlight.Stop != -1) {
                    x2 = graphX + ((currentFlight[currentFlight.Stop].fixTime - currentFlight[0].fixTime) * graphWidht / currentFlight.RecordingTime);
                    g.DrawLine(erase ? graphicsPen : markerPen, x2, 20, x2, graphY);

                    p[0].X= x2;
                    p[0].Y = 17;
                    p[1].X = x2 - 4;
                    p[1].Y = 11;
                    p[2].X = x2 + 5;
                    p[2].Y = 11;
                    p[3].X = x2;
                    p[3].Y = 17;

                    g.FillPolygon(erase ? graphicsBrush : Brushes.LightSalmon, p);
                }

                if (currentFlight.Start != -1 && currentFlight.Stop != -1) {
                    SolidBrush br = new SolidBrush(Color.FromArgb(30, Color.DarkMagenta));
                    g.FillRectangle(erase ? (Brush)graphicsBrush : (Brush)br, x1, graphY + 2, x2 - x1 + 1, 10);
                }
            }
        }

        #endregion

        #region Attributes
        public SoaringDotNet App {
            get {
                return app;
            }
            set {
                app = value;
            }
        }

        public int SmoothFactor {
            get {
                return smoothFactor;
            }
            set {
                if (value < 0) {
                    value = 0;
                }
                smoothFactor = value;
                RedrawDiagram(CreateGraphics());
            }
        }
        #endregion
    
        protected override void OnMouseWheel(MouseEventArgs e) {
            // TODO:  Add Diagram.OnMouseWheel implementation
            base.OnMouseWheel (e);
            if (e.Delta < 0) {
                OnZoomIn();
            }
            else {
                OnZoomOut();
            }
        }    
    }
}
