// name   : OptimizationDialog.cs
// author : Harald Maier
// date   : 01.04.2004
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.Windows.Forms;
using System.Collections;
using System.Text.RegularExpressions;

using SoaringDotNET.GUI;
using SoaringDotNET.Optimization;
using SoaringDotNET.Data;
using SoaringDotNET.Controls;

namespace SoaringDotNET.Dialogs
{
	/// <summary>
	/// 
	/// </summary>
	public class OptimizationDialog : System.Windows.Forms.Form
	{
        private System.Windows.Forms.ListView listViewResults;
        private System.Windows.Forms.ColumnHeader columnLat;
        private System.Windows.Forms.ColumnHeader columnLon;
        private System.Windows.Forms.ColumnHeader columnDist;
        private System.Windows.Forms.ColumnHeader columnPoints;
        private System.Windows.Forms.ColumnHeader columnTurnpoint;
        private System.Windows.Forms.ColumnHeader columnTime;
        private System.Windows.Forms.ColumnHeader columnTaskDist;
        private System.Windows.Forms.ColumnHeader columnHeaderResult;
        private System.Windows.Forms.ColumnHeader columnElevation;
        private System.Windows.Forms.ProgressBar progressBar;
        private Flight flight = null;
        private System.Windows.Forms.Button buttonOK;
        private System.Windows.Forms.Button buttonCancel;
        private System.Windows.Forms.Label label2;
        private System.Windows.Forms.ComboBox comboBoxGliders;
        private System.Windows.Forms.TextBox gliderIndex;
        private System.Windows.Forms.Label label3;
        private System.Windows.Forms.Button buttonStartOptim;
        private System.Windows.Forms.Label label1;
        private System.Windows.Forms.ListView listViewOptimizations;
        private SoaringDotNET.Controls.DiagramControl diagramControl;
        bool isOptimized;

		public OptimizationDialog(Flight flight)
		{
			// 
			// TODO: Add constructor logic here
			//
            Regex r = new Regex(@"[- ]+");

            InitializeComponent();
            
            this.flight = flight;
            diagramControl.Diagram.Switch(flight);

            foreach (DictionaryEntry entry in AppContents.gliderIndexes) {
                comboBoxGliders.Items.Add(entry.Value);
            }

            string t = r.Replace(flight.Glider.ToUpper(), "");
            if (AppContents.gliderIndexes.ContainsKey(t)) {
                comboBoxGliders.SelectedItem = AppContents.gliderIndexes[t];
                flight.GliderIndex = ((Glider)AppContents.gliderIndexes[t]).Index;
            }
            else {
                MessageBox.Show(string.Format("Glider name {0} not found in list of gliders!\nPlease select glider name from list.", 
                    flight.Glider), "Warning", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
            }

            gliderIndex.Text = flight.GliderIndex.ToString();
            isOptimized = flight.currentTask.OptimizedFor != OptimizeTypes.Unknown;

            foreach (OptimizationBase opt in flight.optimizedTasks) {
                ListViewItem it = new ListViewItem(string.Format("{0}: {1:F1} km", opt.ToString(), opt.OptimizedTask.TaskDistanceKM));
                it.Tag = opt;
                listViewOptimizations.Items.Add(it);
                if (opt.OptimizedTask == flight.currentTask) {
                    it.Selected = true;
                }
            }
		}

        private void InitializeComponent() {
            System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(OptimizationDialog));
            this.listViewResults = new System.Windows.Forms.ListView();
            this.columnTurnpoint = new System.Windows.Forms.ColumnHeader();
            this.columnTime = new System.Windows.Forms.ColumnHeader();
            this.columnLat = new System.Windows.Forms.ColumnHeader();
            this.columnLon = new System.Windows.Forms.ColumnHeader();
            this.columnElevation = new System.Windows.Forms.ColumnHeader();
            this.columnDist = new System.Windows.Forms.ColumnHeader();
            this.columnTaskDist = new System.Windows.Forms.ColumnHeader();
            this.columnPoints = new System.Windows.Forms.ColumnHeader();
            this.progressBar = new System.Windows.Forms.ProgressBar();
            this.buttonOK = new System.Windows.Forms.Button();
            this.buttonCancel = new System.Windows.Forms.Button();
            this.label2 = new System.Windows.Forms.Label();
            this.comboBoxGliders = new System.Windows.Forms.ComboBox();
            this.gliderIndex = new System.Windows.Forms.TextBox();
            this.label3 = new System.Windows.Forms.Label();
            this.buttonStartOptim = new System.Windows.Forms.Button();
            this.label1 = new System.Windows.Forms.Label();
            this.listViewOptimizations = new System.Windows.Forms.ListView();
            this.columnHeaderResult = new System.Windows.Forms.ColumnHeader();
            this.diagramControl = new SoaringDotNET.Controls.DiagramControl();
            this.SuspendLayout();
            // 
            // listViewResults
            // 
            this.listViewResults.Columns.AddRange(new System.Windows.Forms.ColumnHeader[] {
                                                                                              this.columnTurnpoint,
                                                                                              this.columnTime,
                                                                                              this.columnLat,
                                                                                              this.columnLon,
                                                                                              this.columnElevation,
                                                                                              this.columnDist,
                                                                                              this.columnTaskDist,
                                                                                              this.columnPoints});
            this.listViewResults.FullRowSelect = true;
            this.listViewResults.HideSelection = false;
            this.listViewResults.Location = new System.Drawing.Point(8, 88);
            this.listViewResults.Name = "listViewResults";
            this.listViewResults.Size = new System.Drawing.Size(568, 168);
            this.listViewResults.TabIndex = 2;
            this.listViewResults.View = System.Windows.Forms.View.Details;
            // 
            // columnTurnpoint
            // 
            this.columnTurnpoint.Text = "Turnpoint";
            this.columnTurnpoint.Width = 80;
            // 
            // columnTime
            // 
            this.columnTime.Text = "Time";
            // 
            // columnLat
            // 
            this.columnLat.Text = "Latitude";
            this.columnLat.Width = 80;
            // 
            // columnLon
            // 
            this.columnLon.Text = "Longitude";
            this.columnLon.Width = 80;
            // 
            // columnElevation
            // 
            this.columnElevation.Text = "Elevation";
            this.columnElevation.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            // 
            // columnDist
            // 
            this.columnDist.Text = "Distance";
            this.columnDist.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            this.columnDist.Width = 66;
            // 
            // columnTaskDist
            // 
            this.columnTaskDist.Text = "Task dist.";
            this.columnTaskDist.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            this.columnTaskDist.Width = 66;
            // 
            // columnPoints
            // 
            this.columnPoints.Text = "Points";
            this.columnPoints.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            // 
            // progressBar
            // 
            this.progressBar.Location = new System.Drawing.Point(8, 56);
            this.progressBar.Name = "progressBar";
            this.progressBar.Size = new System.Drawing.Size(568, 23);
            this.progressBar.Step = 1;
            this.progressBar.TabIndex = 4;
            // 
            // buttonOK
            // 
            this.buttonOK.DialogResult = System.Windows.Forms.DialogResult.OK;
            this.buttonOK.Location = new System.Drawing.Point(208, 528);
            this.buttonOK.Name = "buttonOK";
            this.buttonOK.TabIndex = 5;
            this.buttonOK.Text = "OK";
            // 
            // buttonCancel
            // 
            this.buttonCancel.DialogResult = System.Windows.Forms.DialogResult.Cancel;
            this.buttonCancel.Location = new System.Drawing.Point(307, 528);
            this.buttonCancel.Name = "buttonCancel";
            this.buttonCancel.TabIndex = 6;
            this.buttonCancel.Text = "Cancel";
            // 
            // label2
            // 
            this.label2.AutoSize = true;
            this.label2.Location = new System.Drawing.Point(8, 8);
            this.label2.Name = "label2";
            this.label2.Size = new System.Drawing.Size(35, 16);
            this.label2.TabIndex = 7;
            this.label2.Text = "Glider";
            // 
            // comboBoxGliders
            // 
            this.comboBoxGliders.Location = new System.Drawing.Point(8, 24);
            this.comboBoxGliders.Name = "comboBoxGliders";
            this.comboBoxGliders.Size = new System.Drawing.Size(184, 21);
            this.comboBoxGliders.Sorted = true;
            this.comboBoxGliders.TabIndex = 8;
            this.comboBoxGliders.SelectedIndexChanged += new System.EventHandler(this.OnGliderChanged);
            // 
            // gliderIndex
            // 
            this.gliderIndex.Location = new System.Drawing.Point(200, 24);
            this.gliderIndex.Name = "gliderIndex";
            this.gliderIndex.Size = new System.Drawing.Size(40, 20);
            this.gliderIndex.TabIndex = 9;
            this.gliderIndex.Text = "";
            this.gliderIndex.TextChanged += new System.EventHandler(this.OnIndexChanged);
            // 
            // label3
            // 
            this.label3.AutoSize = true;
            this.label3.Location = new System.Drawing.Point(200, 8);
            this.label3.Name = "label3";
            this.label3.Size = new System.Drawing.Size(32, 16);
            this.label3.TabIndex = 10;
            this.label3.Text = "Index";
            // 
            // buttonStartOptim
            // 
            this.buttonStartOptim.Location = new System.Drawing.Point(463, 528);
            this.buttonStartOptim.Name = "buttonStartOptim";
            this.buttonStartOptim.TabIndex = 11;
            this.buttonStartOptim.Text = "Start";
            this.buttonStartOptim.Click += new System.EventHandler(this.OnStartOptimization);
            // 
            // label1
            // 
            this.label1.AutoSize = true;
            this.label1.Location = new System.Drawing.Point(584, 32);
            this.label1.Name = "label1";
            this.label1.Size = new System.Drawing.Size(42, 16);
            this.label1.TabIndex = 12;
            this.label1.Text = "Results";
            // 
            // listViewOptimizations
            // 
            this.listViewOptimizations.Columns.AddRange(new System.Windows.Forms.ColumnHeader[] {
                                                                                                    this.columnHeaderResult});
            this.listViewOptimizations.FullRowSelect = true;
            this.listViewOptimizations.HideSelection = false;
            this.listViewOptimizations.Location = new System.Drawing.Point(584, 56);
            this.listViewOptimizations.Name = "listViewOptimizations";
            this.listViewOptimizations.Size = new System.Drawing.Size(200, 200);
            this.listViewOptimizations.TabIndex = 13;
            this.listViewOptimizations.View = System.Windows.Forms.View.List;
            this.listViewOptimizations.SelectedIndexChanged += new System.EventHandler(this.OnOptimizationsChanged);
            // 
            // columnHeaderResult
            // 
            this.columnHeaderResult.Width = 200;
            // 
            // diagramControl
            // 
            this.diagramControl.AutoScroll = true;
            this.diagramControl.Location = new System.Drawing.Point(8, 264);
            this.diagramControl.Name = "diagramControl";
            this.diagramControl.Size = new System.Drawing.Size(776, 256);
            this.diagramControl.TabIndex = 0;
            // 
            // OptimizationDialog
            // 
            this.AcceptButton = this.buttonOK;
            this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
            this.CancelButton = this.buttonCancel;
            this.ClientSize = new System.Drawing.Size(794, 568);
            this.Controls.Add(this.diagramControl);
            this.Controls.Add(this.listViewOptimizations);
            this.Controls.Add(this.label1);
            this.Controls.Add(this.buttonStartOptim);
            this.Controls.Add(this.label3);
            this.Controls.Add(this.gliderIndex);
            this.Controls.Add(this.label2);
            this.Controls.Add(this.comboBoxGliders);
            this.Controls.Add(this.buttonCancel);
            this.Controls.Add(this.buttonOK);
            this.Controls.Add(this.progressBar);
            this.Controls.Add(this.listViewResults);
            this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.FixedDialog;
            this.Icon = ((System.Drawing.Icon)(resources.GetObject("$this.Icon")));
            this.MaximizeBox = false;
            this.MinimizeBox = false;
            this.Name = "OptimizationDialog";
            this.ShowInTaskbar = false;
            this.StartPosition = System.Windows.Forms.FormStartPosition.CenterParent;
            this.Text = "Optimization";
            this.ResumeLayout(false);

        }

        private void OnIndexChanged(object sender, System.EventArgs e) {
            try {
                if (gliderIndex.Text.Length > 0) {
                    flight.GliderIndex = int.Parse(gliderIndex.Text);
                    if (isOptimized) {
                        ShowPoints();
                    }
                }
            }
            catch (FormatException ex) {
                MessageBox.Show(this, "Invalid number " + gliderIndex.Text + "\n\n" + ex.ToString(), "Error", MessageBoxButtons.OK, 
                    MessageBoxIcon.Error);
            }
        }

        private void OnGliderChanged(object sender, System.EventArgs e) {
            gliderIndex.Text = ((Glider)comboBoxGliders.SelectedItem).Index.ToString();
            flight.Glider = ((Glider)comboBoxGliders.SelectedItem).Abbreviation;
        }

        private void OnStartOptimization(object sender, System.EventArgs e) {
            int i, start, stop;
            isOptimized = false;
            buttonCancel.Enabled = false;
            buttonOK.Enabled = false;
            ListViewItem item;

            listViewResults.Items.Clear();
            listViewResults.Refresh();
            listViewOptimizations.Items.Clear();
            listViewOptimizations.Refresh();

            progressBar.Maximum = flight.optimizedTasks.Count;
            Cursor = Cursors.WaitCursor;
            start = flight.Start != -1 ? flight.Start : 0;
            stop = flight.Stop != -1 ? flight.Stop : flight.Count - 1;

            flight.minElevationSinceStartOfTask = new int [flight.Count];
            flight.minElevationSinceStartOfTask[start] = flight[start].Elevation;
            for (i = start + 1; i <= stop; i++) {
                flight.minElevationSinceStartOfTask[i] = Math.Min(flight.minElevationSinceStartOfTask[i - 1], flight[i].Elevation);
            }

            while (flight.calcDistanceThread.IsAlive) {
                System.Threading.Thread.Sleep(10);
            }

            foreach (OptimizationBase optimizer in flight.optimizedTasks) {
                optimizer.Optimize();
                item = new ListViewItem(string.Format("{0}: {1:F1} km", optimizer.ToString(), optimizer.OptimizedTask.TaskDistanceKM));
                item.Tag = optimizer;
                listViewOptimizations.Items.Add(item);
                if (listViewOptimizations.Items.Count == 1) {
                    listViewOptimizations.Items[0].Selected = true;
                }
                progressBar.PerformStep();
                Application.DoEvents();
            }

            isOptimized = true;

            Cursor = Cursors.Default;
            buttonCancel.Enabled = true;
            buttonOK.Enabled = true;
            progressBar.Value = 0;        
        }
    
        private void ShowTask() {
            int i;
            double length;
            string lStringAll, lStringTask;
            WayPoint p1;
            ListViewItem l;
            Task optimizedTask;

            OptimizationBase optimizer = (OptimizationBase)listViewOptimizations.SelectedItems[0].Tag;
            optimizedTask = optimizer.OptimizedTask;
            
            listViewResults.Items.Clear();

            for (i = 0; i < optimizedTask.Count; i++) {
                p1 = optimizedTask[i];

                l = new ListViewItem(p1.longName);
                l.ImageIndex = 0;
                // ignore takeoff and takeoff -> start
                length = i > 0 ? p1.distanceKM(optimizedTask[i - 1]) : 0;
                lStringAll = i > 0 ? string.Format(AppContents.ni, "{0:F1} km", length) : "";
                if (i > 1 && i < optimizedTask.Count - 1) {
                    lStringTask = lStringAll;
                }
                else {
                    lStringTask = "";
                }

                if (optimizedTask.TaskType == TaskTypes.TriangleStartOnLeg) {
                    // no points from takoff -> start, start -> TP 1
                    // no points from TP 3 -> finish, finish -> landing
                    if (i > 1 && i < optimizedTask.Count - 2) {
                        if (i == 2) {
                            // first TP, get length to last TP
                            length = p1.distanceKM(optimizedTask[4]);
                            lStringTask = string.Format(AppContents.ni, "{0:F1} km", length);
                        }
                    }
                    else {
                        lStringTask = "";
                    }
                }

                l.SubItems.AddRange(new string [] {p1.ToStringTime(), p1.ToStringLat(), p1.ToStringLong(), string.Format("{0} m", p1.reachPoint.Elevation), lStringAll, lStringTask, ""});
                listViewResults.Items.Add(l);
            }

            l = new ListViewItem("Total");
            l.ImageIndex = 0;
            l.SubItems.AddRange(new string [] {"", "", "", "", string.Format(AppContents.ni, "{0:F1} km", optimizedTask.TotalDistanceKM), string.Format(AppContents.ni, "{0:F1} km", optimizedTask.TaskDistanceKM), ""});
            listViewResults.Items.Add(l);

            ShowPoints();

            Task oldTask = flight.currentTask;
            flight.currentTask = optimizedTask;
            diagramControl.Refresh();
            flight.currentTask = oldTask;
        }

        private void ShowPoints() {
            int i;
            double sumPoints, length, points;
            string pString;
            WayPoint p1;
            ListViewItem l;

            sumPoints = 0;
            Task optimizedTask;

            OptimizationBase optimizer = (OptimizationBase)listViewOptimizations.SelectedItems[0].Tag;
            optimizedTask = optimizer.OptimizedTask;
            if (optimizedTask.Count > 0) {
                // ignore takeoff
                for (i = 1; i < optimizedTask.Count; i++) {
                    p1 = optimizedTask[i];

                    l = listViewResults.Items[i];
                    
                    length = p1.distanceKM(optimizedTask[i - 1]);

                    if (optimizedTask.TaskType != TaskTypes.TriangleStartOnLeg) {
                        // no points from takeoff -> start
                        // no points from finish -> landing
                        if (i > 1 && i < optimizedTask.Count - 1) {
                            points = length * 100.0 / flight.GliderIndex * optimizer.GetLegWeight(i - 2);
                            pString = string.Format(AppContents.ni, "{0:F1}", points);
                        }
                        else {
                            points = 0.0;
                            pString = "";
                        }
                    }
                    else {
                        // no points from takoff -> start, start -> TP 1
                        // no points from TP 3 -> finish, finish -> landing
                        if (i > 1 && i < optimizedTask.Count - 2) {
                            if (i == 2) {
                                // first TP, get length to last TP
                                length = p1.distanceKM(optimizedTask[4]);
                            }
                            points = length * 100.0 / flight.GliderIndex * optimizer.GetLegWeight(i - 2);
                            pString = string.Format(AppContents.ni, "{0:F1}", points);
                        }
                        else {
                            points = 0.0;
                            pString = "";
                        }
                    }

                    l.SubItems[7].Text = pString;
                    sumPoints += points;
                }

                l = listViewResults.Items[i];
                l.SubItems[7].Text = string.Format(AppContents.ni, "{0:F1}", sumPoints);
            }
        }

        private void OnOptimizationsChanged(object sender, System.EventArgs e) {
            if (listViewOptimizations.SelectedIndices.Count > 0) {
                ShowTask();
            }
        }

        public Task OptimizedTask {
            get {
                Task optimizedTask = null;
                if (listViewOptimizations.SelectedItems.Count > 0) {
                    OptimizationBase optimizer = (OptimizationBase)listViewOptimizations.SelectedItems[0].Tag;
                    optimizedTask = optimizer.OptimizedTask;
                }
                return optimizedTask;
            }
        }

        public Diagram Diagram {
            get {
                return diagramControl.Diagram;
            }
        }
    }
}
