using System;
using System.IO;

/*

                           PDB file format

        This should be used only for reference because it
        assumes no structure padding and big-endian
        byte alignment.  All I/O to the actual PDB file must
        be done with padding and byte order independent code.

*/

namespace SoaringDotNET.FileFormats
{
	/// <summary>
	/// 
	/// </summary>
	public class PDBFile : BinaryWriter
	{
        private const int MaxPDBNameSize = 32;
        public  const int PDBHeaderSize = 78;
        public const int  PDBRecordEntrySize = 8;

        public enum PDBFlags :ushort {
            pdbResourceFlag = 0x0001,          /* Is this a resource file ? */
            pdbReadOnlyFlag = 0x0002,          /* Is database read only ? */
            pdbAppInfoDirtyFlag = 0x0004,      /* Is application info block dirty ? */
            pdbBackupFlag = 0x0008,            /* Back up to PC if no conduit defined */
            pdbOKToInstallNewer = 0x0010,      /* OK to install a newer version if current database open */
            pdbResetAfterInstall = 0x0020,     /* Must reset machine after installation */
            pdbStream = 0x0080,                /* Used for file streaming */
            pdbOpenFlag = 0x8000               /* Not closed properly */
        };

        public enum PDBVersion :ushort {
            pdbVerReadOnly = 0x0001,
            pdbVerWrite = 0x0002,
            pdbVerReadWrite = 0x0003,
            pdbVerLeaveOpen = 0x0004,
            pdbVerExclusive = 0x0008,
            pdbVerShowSecret = 0x0010
        };

        public struct Header {
            public string fileName; // Database name on Palm //
            public ushort flags;
            public ushort version;
            public uint   creationTime;
            public uint   modificationTime;
            public uint   backupTime;
            public uint   modificationNumber;
            public uint   appInfoOffset;
            public uint   sortInfoOffset;
            public string type; // Application database type //
            public string creator; // Creator ID //
            public uint   uniqueID;
            public uint   nextRecordID;
            public ushort numRecords;
        }
        public Header header;

		public PDBFile(System.IO.Stream output) :
            base(output)
		{
			// 
			// TODO: Add constructor logic here
			//
		}

        public void WriteHeader() {
            // get time in secs past 1900
            DateTime s = new DateTime(1900, 1, 1);
            long now = (DateTime.Now.Subtract(s)).Ticks;
            now /= 10000000;

            Write(header.fileName, MaxPDBNameSize); // name //
            Write(header.flags); // flags //
            Write(header.version);
            Write((uint)now); // creationTime //
            Write((uint)now); // modificationTime //
            Write((uint)(0)); // backupTime //
            Write((uint)(0)); // modificationNumber //
            Write((uint)(0)); // appInfoOffset //
            Write((uint)(0)); // sortInfoOffset //
            Write(header.type, 4); // type //
            Write(header.creator, 4); // Creator //
            Write((uint)(0)); // uniqueID //
            Write((uint)(0)); // nextRecordID //
            Write(header.numRecords); // number of records //
        }

        public override void Write(string value) {
            // TODO:  Add PDBFile.Write implementation
            Write (value.ToCharArray());
        }
    
        public void Write(string value, int width) {
            // TODO:  Add PDBFile.Write implementation
            Write(value.PadRight(width, '\0').Substring(0, width).ToCharArray());
        }
    
        public override void Write(float value) {
            // TODO:  Add PDBFile.Write implementation
            base.Write (Swap(value));
        }
    
        public override void Write(ulong value) {
            // TODO:  Add PDBFile.Write implementation
            throw new Exception("Write unsigned long not supported");
        }
    
        public override void Write(long value) {
            // TODO:  Add PDBFile.Write implementation
            throw new Exception("Write long not supported");
        }
    
        public override void Write(uint value) {
            // TODO:  Add PDBFile.Write implementation
            base.Write (Swap(value));
        }
    
        public override void Write(int value) {
            // TODO:  Add PDBFile.Write implementation
            base.Write (Swap(value));
        }
    
        public override void Write(ushort value) {
            // TODO:  Add PDBFile.Write implementation
            base.Write (Swap(value));
        }
    
        public override void Write(short value) {
            // TODO:  Add PDBFile.Write implementation
            base.Write (Swap(value));
        }
    
        public void Write(short [] value) {
            // TODO:  Add PDBFile.Write implementation
            foreach (short s in value) {
                base.Write(Swap(s));
            }
        }
    
        public void Write(short [] value, int count) {
            // TODO:  Add PDBFile.Write implementation
            int i;
            for (i = 0; i < count; i++) {
                base.Write(Swap(value[i]));
            }
        }
    
        public override void Write(decimal value) {
            // TODO:  Add PDBFile.Write implementation
            throw new Exception("Write decimal not supported");
        }
    
        public override void Write(double value) {
            // TODO:  Add PDBFile.Write implementation
            //base.Write (Swap(value));
        }
    
        public override void Write(char[] chars) {
            // TODO:  Add PDBFile.Write implementation
            base.Write (chars);
        }
   
        public void Write(char[] chars, int width) {
            // TODO:  Add PDBFile.Write implementation
            int i;
            for (i = 0; i < chars.Length && i < width; i++) {
                base.Write (chars[i]);
            }
            while (i < width) {
                base.Write((byte)'\0');
            }
        }
   
        public override void Write(char[] chars, int index, int count) {
            // TODO:  Add PDBFile.Write implementation
            base.Write (chars, index, count);
        }
    
        public override void Write(bool value) {
            // TODO:  Add PDBFile.Write implementation
            base.Write (value == true ? (char)1 : (char)0);
        }

        #region Swap to Big endian
        private short Swap(short v) {
            return (short)Swap((ushort) v);
        }

        private ushort Swap(ushort v) {
            return (ushort)(((v >> 8) & 0x00FF) | ((v << 8) & 0xFF00));
        }

        private int Swap(int v) {
            return (int)Swap((uint) v);
        }

        private uint Swap(uint v) {
            return ((v >> 24) & 0x000000FF) | ((v >> 8)  & 0x0000FF00) | ((v << 8) & 0x00FF0000) | ((v << 24) & 0xFF000000);
        }

        //private ulong Swap(double v) {
        //    return Swap((ulong)v);
        //}

        private byte [] Swap(float v) {
            byte [] a = {0, 0, 0, 0};
            byte [] b = {0, 0, 0, 0};
            BinaryWriter w = new BinaryWriter(new MemoryStream(a));
            w.Write(v);
            w.Close();
            b[0] = a[3];
            b[1] = a[2];
            b[2] = a[1];
            b[3] = a[0];
            return b;
        }
        #endregion
    }
}
