// name   : SUAFileHandlerTimNewport.cs
// author : Harald Maier
// date   : 10.10.2005
//
//
// This program is free software; you can redistribute it and/or modify  
// it under the terms of the GNU General Public License as published by  
// the Free Software Foundation; either version 2 of the License, or     
// (at your option) any later version.                                   

using System;
using System.IO;
using System.Collections;
using System.Text.RegularExpressions;
using System.Windows.Forms;

using SoaringDotNET.GUI;
using SoaringDotNET.Data;

namespace SoaringDotNET.FileFormats
{
	/// <summary>
	/// 
	/// </summary>
	public class SUAFileHandlerTimNewport : SoaringDotNET.FileFormats.SUAFileHandlerBase
	{
		public SUAFileHandlerTimNewport()
		{
			// 
			// TODO: Add constructor logic here
			//
		}

        public override ArrayList WriteFile(ArrayList suas) {
            ArrayList file = new ArrayList(suas.Count * 6);
            bool firstSeg;
            string tmp;
            Hashtable includeId = new Hashtable();

            Airspace space = null;
            file.Add(string.Format("# Generated by {0} Version {1}", Application.ProductName, Application.ProductVersion));
            file.Add("# NOTE: This file, like any other generated by this program, comes with absolutly no warranty.");
            file.Add("# It's in the users responsibility to check for correctness and accuracy. This data is unofficial");
            file.Add(string.Format("# and is used by your own risk!{0}", Environment.NewLine));

            includeId.Add(AirspaceTypes.Alert, null);
            includeId.Add(AirspaceTypes.Danger, null);
            includeId.Add(AirspaceTypes.GliderProhibited, null);
            includeId.Add(AirspaceTypes.Military, null);
            includeId.Add(AirspaceTypes.MilitaryAerodromeTraffic, null);
            includeId.Add(AirspaceTypes.Prohibited, null);
            includeId.Add(AirspaceTypes.Restricted, null);
            includeId.Add(AirspaceTypes.Warning, null);

            foreach (AirspaceElement sua in suas) {
                space = sua.airspace;
                file.Add("INCLUDE=YES");
                if (space.id != "" && includeId.ContainsKey(space.airspaceType)) {
                    tmp = string.Format("{0}:{1}",  space.id, space.name);
                }
                else {
                    tmp = space.name;
                }
                file.Add(string.Format("TITLE={0}", tmp));

                switch (space.airspaceType) {
                case AirspaceTypes.TerminalControlArea:
                    tmp = "TMA";
                    break;
                case AirspaceTypes.ControlZone:
                case AirspaceTypes.ControlArea:
                    tmp = "CTA/CTR";
                    break;
                case AirspaceTypes.Alert:
                    tmp = "ALERT";
                    break;
                case AirspaceTypes.Warning:
                    tmp = "WARN";
                    break;
                case AirspaceTypes.Other:
                case AirspaceTypes.Unknown:
                    tmp = "OTHER";
                    break;
                case AirspaceTypes.Danger:
                    tmp = "DANGER";
                    break;
                case AirspaceTypes.Military:
                    tmp = "MOA";
                    break;
                case AirspaceTypes.MilitaryAerodromeTraffic:
                    tmp = "MATZ";
                    break;
                case AirspaceTypes.Prohibited:
                case AirspaceTypes.GliderProhibited:
                    tmp = "PROHIBITED";
                    break;
                case AirspaceTypes.Restricted:
                    tmp = "RESTRICTED";
                    break;
                case AirspaceTypes.Soaring:
                    tmp = "GSEC";
                    break;
                case AirspaceTypes.Temporary:
                    tmp = "TEMP";
                    break;
                case AirspaceTypes.TMZ:
                    tmp = "TMZ";
                    break;
                case AirspaceTypes.Airway:
                    tmp = "AIRWAYS";
                    break;
                }

                file.Add(string.Format("TYPE={0}", tmp));
                file.Add(string.Format("CLASS={0}", sua.AirspaceClass != AirspaceClasses.Unknown ? sua.AirspaceClass.ToString() : "X"));

                file.Add(string.Format("RADIO={0}", space.radioFrequency));
                file.Add(string.Format("ACTIVE={0}", space.remark));

                file.Add(string.Format("BASE={0}", space.lowerBound));
                file.Add(string.Format("TOPS={0}", space.upperBound));
                firstSeg = true;
                foreach (AirspaceSegment seg in sua) {
                    switch (seg.type) {
                    case AirspaceSegmentTypes.Line:
                        if (firstSeg) {
                            file.Add(string.Format("POINT={0} {1}", WgsPoint.ToCompactString(seg.latFrom, true), WgsPoint.ToCompactString(seg.lonFrom, false)));
                            firstSeg = false;
                        }
                        file.Add(string.Format("POINT={0} {1}", WgsPoint.ToCompactString(seg.latTo, true), WgsPoint.ToCompactString(seg.lonTo, false)));
                        break;
                    case AirspaceSegmentTypes.Circle:
                        file.Add(string.Format(AppContents.ni, "CIRCLE RADIUS={0} CENTRE={1} {2}", seg.radius, WgsPoint.ToCompactString(seg.latCenter, true), WgsPoint.ToCompactString(seg.lonCenter, false)));
                        break;
                    case AirspaceSegmentTypes.ArcClock:
                    case AirspaceSegmentTypes.ArcCounterClock:
                        if (firstSeg) {
                            file.Add(string.Format("POINT={0} {1}", WgsPoint.ToCompactString(seg.latFrom, true), WgsPoint.ToCompactString(seg.lonFrom, false)));
                        }
                        file.Add(string.Format(AppContents.ni, "{0}CLOCKWISE RADIUS={1} CENTRE={2} {3} TO={4} {5}", seg.type == AirspaceSegmentTypes.ArcCounterClock ? "ANTI-" : "",
                            seg.radius, WgsPoint.ToCompactString(seg.latCenter, true), WgsPoint.ToCompactString(seg.lonCenter, false), 
                            WgsPoint.ToCompactString(seg.latTo, true), WgsPoint.ToCompactString(seg.lonTo, false)));
                        firstSeg = false;
                        break;
                    }
                }
                file.Add("#");
            }
            file.Add("END");

            return file;
        }

        public override ArrayList ReadFile(ArrayList file, bool useMichaelMeier, AirspaceClasses defaultClass, bool overwriteClass, SUATypes defaultType, bool overwriteType) {
            ArrayList suas = new ArrayList();
            bool include = true;
            char [] splitt = {'\t', ' '};
            Match m = null;
            GroupCollection coll;
            string t;
            string []tt;
            string airspaceActive = "", airspaceRadio = "";
            AirspaceClasses airspaceClass = AirspaceClasses.Unknown;
            AirspaceTypes airspaceType = AirspaceTypes.Unknown;
            int lat, lon;
            bool haveDetails = false;
            bool haveClassStmt = false;

            Regex tokens = new Regex("(?:(?<first>^INCLUDE)[ \t]*=[ \t]*(?<second>YES|NO))|" +
                "(?:(?<first>^TITLE)[ \t]*=[ \t]*(?<second>.+))|" +
                "(?:(?<first>^BASE)[ \t]*=[ \t]*(?<second>.+))|" +
                "(?:(?<first>^POINT)[ \t]*=[ \t]*(?<second>[N|S][0-9]{6}[ \t]*[E|W][0-9]{7}))|" +
                "(?:(?<first>^CIRCLE)[ \t]+RADIUS[ \t]*=[ \t]*(?<radius>[0-9]+\\.?[0-9]*)[ \t]+CENTRE[ \t]*=[ \t]*(?<centre>[N|S][0-9]{6}[ \t]*[E|W][0-9]{7}))|" +
                "(?:(?<first>^CLOCKWISE|ANTI-CLOCKWISE)[ \t]+RADIUS[ \t]*=[ \t]*(?<radius>[0-9]+\\.?[0-9]*)[ \t]+CENTRE[ \t]*=[ \t]*(?<centre>[N|S][0-9]{6}[ \t]*[E|W][0-9]{7})[ \t]+TO[ \t]*=[ \t]*(?<to>[N|S][0-9]{6}[ \t]*[E|W][0-9]{7}))|" +
                "(?:(?<first>^TOPS)[ \t]*=[ \t]*(?<second>.+))|" +
                "(?:(?<first>^CLASS)[ \t]*=[ \t]*(?<second>.*))|" +
                "(?:(?<first>^RADIO)[ \t]*=[ \t]*(?<second>.*))|" +
                "(?:(?<first>^ACTIVE)[ \t]*=[ \t]*(?<second>.*))|" +
                "(?:(?<first>^TYPE)[ \t]*=[ \t]*(?<second>.+))", RegexOptions.IgnoreCase);
            
            Airspace space = new Airspace();
            AirspaceSegment segment = null;

            try {
                foreach (string line in file) {
                    m = tokens.Match(line.Trim());
                    if (m.Success) {
                        coll = m.Groups;
                        t = coll["first"].Value.ToUpper();
                        if (t == "INCLUDE") {
                            include = (coll["second"].Value == "YES");
                        }
                        else if (include) {
                            switch (t) {
                            case "TITLE":
                                if (space.name.Length > 0 && haveDetails) {
                                    CheckImportOptions(space, useMichaelMeier, defaultClass, overwriteClass, defaultType, overwriteType);
                                    suas.Add(space);
                                    haveDetails = false;
                                    space = new Airspace();
                                    segment = null;
                                }

                                space.name = coll["second"].Value;
                                space.airspaceType = airspaceType;
                                space.airspaceClass = airspaceClass;
                                space.remark = airspaceActive;
                                space.radioFrequency = airspaceRadio;
                                break;
                            case "TYPE":
                                if (space.name.Length > 0 && haveDetails) {
                                    CheckImportOptions(space, useMichaelMeier, defaultClass, overwriteClass, defaultType, overwriteType);
                                    suas.Add(space);
                                    haveDetails = false;
                                    space = new Airspace();
                                    segment = null;
                                }

                                t = coll["second"].Value.Replace(" ", "").ToUpper();
                                switch (t) {
                                case "CTLZ":
                                    airspaceType = AirspaceTypes.ControlZone;
                                    if (!haveClassStmt) {
                                        airspaceClass = AirspaceClasses.Unknown;
                                    }
                                    break;
                                case "CTA":
                                case "CTR":
                                case "CTA/CTR":
                                case "C":
                                    airspaceType = AirspaceTypes.ControlArea;
                                    if (!haveClassStmt) {
                                        airspaceClass = AirspaceClasses.Unknown;
                                    }
                                    break;
                                case "TCA":
                                case "TMA":
                                case "MTCA":
                                    airspaceType = AirspaceTypes.TerminalControlArea;
                                    if (!haveClassStmt) {
                                        airspaceClass = AirspaceClasses.Unknown;
                                    }
                                    break;
                                case "AIRWAYS":
                                case "A":
                                    airspaceType = AirspaceTypes.Airway;
                                    if (!haveClassStmt) {
                                        airspaceClass = AirspaceClasses.Unknown;
                                    }
                                    break;
                                case "RESTRICTED":
                                case "R":
                                    airspaceType = AirspaceTypes.Restricted;
                                    if (!haveClassStmt) {
                                        airspaceClass = AirspaceClasses.Unknown;
                                    }
                                    break;
                                case "PROHIBITED":
                                case "P":
                                    airspaceType = AirspaceTypes.Prohibited;
                                    if (!haveClassStmt) {
                                        airspaceClass = AirspaceClasses.Unknown;
                                    }
                                    break;
                                case "DANGER":
                                case "D":
                                    airspaceType = AirspaceTypes.Danger;
                                    if (!haveClassStmt) {
                                        airspaceClass = AirspaceClasses.Unknown;
                                    }
                                    break;
                                case "OTHER":
                                case "O":
                                    airspaceType = AirspaceTypes.Other;
                                    if (!haveClassStmt) {
                                        airspaceClass = AirspaceClasses.Unknown;
                                    }
                                    break;
                                case "TRAININGZONE":
                                case "Z":
                                    airspaceType = AirspaceTypes.Temporary;
                                    if (!haveClassStmt) {
                                        airspaceClass = AirspaceClasses.Unknown;
                                    }
                                    break;
                                case "MATZ":
                                    airspaceType = AirspaceTypes.MilitaryAerodromeTraffic;
                                    if (!haveClassStmt) {
                                        airspaceClass = AirspaceClasses.Unknown;
                                    }
                                    break;
                                case "MOA":
                                case "M":
                                    airspaceType = AirspaceTypes.Military;
                                    if (!haveClassStmt) {
                                        airspaceClass = AirspaceClasses.Unknown;
                                    }
                                    break;
                                case "TRAFFICINFO":
                                case "I":
                                    airspaceType = AirspaceTypes.Unknown;
                                    if (!haveClassStmt) {
                                        airspaceClass = AirspaceClasses.Unknown;
                                    }
                                    break;
                                case "CLASSA":
                                    airspaceClass = AirspaceClasses.A;
                                    airspaceType = AirspaceTypes.ControlArea;
                                    break;
                                case "CLASSB":
                                    airspaceClass = AirspaceClasses.B;
                                    airspaceType = AirspaceTypes.ControlArea;
                                    break;
                                case "CLASSC":
                                    airspaceClass = AirspaceClasses.C;
                                    airspaceType = AirspaceTypes.ControlArea;
                                    break;
                                case "CLASSD":
                                    airspaceClass = AirspaceClasses.D;
                                    airspaceType = AirspaceTypes.ControlArea;
                                    break;
                                case "CLASSE":
                                    airspaceClass = AirspaceClasses.E;
                                    airspaceType = AirspaceTypes.ControlArea;
                                    break;
                                case "CLASSF":
                                    airspaceClass = AirspaceClasses.F;
                                    airspaceType = AirspaceTypes.ControlArea;
                                    break;
                                case "CLASSG":
                                    airspaceClass = AirspaceClasses.G;
                                    airspaceType = AirspaceTypes.ControlArea;
                                    break;
                                case "SOARING":
                                case "GSEC":
                                    airspaceType = AirspaceTypes.Soaring;
                                    if (!haveClassStmt) {
                                        airspaceClass = AirspaceClasses.Unknown;
                                    }
                                    break;
                                case "TMZ":
                                    airspaceType = AirspaceTypes.TMZ;
                                    if (!haveClassStmt) {
                                        airspaceClass = AirspaceClasses.Unknown;
                                    }
                                    break;
                                case "ALERT":
                                    airspaceType = AirspaceTypes.Alert;
                                    if (!haveClassStmt) {
                                        airspaceClass = AirspaceClasses.Unknown;
                                    }
                                    break;
                                default:
                                    airspaceType = AirspaceTypes.Unknown;
                                    if (!haveClassStmt) {
                                        airspaceClass = AirspaceClasses.Unknown;
                                    }
                                    break;
                                }

                                if (space != null) {
                                    space.airspaceType = airspaceType;
                                    space.airspaceClass = airspaceClass;
                                }
                                break;
                            case "CLASS":
                                if (space.name.Length > 0 && haveDetails) {
                                    CheckImportOptions(space, useMichaelMeier, defaultClass, overwriteClass, defaultType, overwriteType);
                                    suas.Add(space);
                                    haveDetails = false;
                                    space = new Airspace();
                                    segment = null;
                                }
                                
                                haveClassStmt = true;

                                t = coll["second"].Value.ToUpper();
                                switch (t) {
                                case "A":
                                    airspaceClass = AirspaceClasses.A;
                                    break;
                                case "B":
                                    airspaceClass = AirspaceClasses.B;
                                    break;
                                case "C":
                                    airspaceClass = AirspaceClasses.C;
                                    break;
                                case "D":
                                    airspaceClass = AirspaceClasses.D;
                                    break;
                                case "E":
                                    airspaceClass = AirspaceClasses.E;
                                    break;
                                case "F":
                                    airspaceClass = AirspaceClasses.F;
                                    break;
                                case "G":
                                    airspaceClass = AirspaceClasses.G;
                                    break;
                                default:
                                    airspaceClass = AirspaceClasses.Unknown;
                                    break;
                                }

                                if (space != null) {
                                    space.airspaceClass = airspaceClass;
                                }

                                break;
                            case "POINT":
                                haveDetails = true;
                                tt = coll["second"].Value.Split(splitt);
                                lat = WgsPoint.FromCompactString(tt[0], true);
                                lon = WgsPoint.FromCompactString(tt[1], false);
                                if (segment == null) {
                                    segment = new AirspaceSegment();
                                    // from
                                    segment.latFrom = lat;
                                    segment.lonFrom = lon;
                                }
                                else {
                                    segment.type = AirspaceSegmentTypes.Line;
                                    // to
                                    segment.latTo = lat;
                                    segment.lonTo = lon;

                                    space.points.Add(segment);
                                    segment = new AirspaceSegment();
                                    segment.type = AirspaceSegmentTypes.Unknown;
                                    // from = to
                                    segment.latFrom = lat;
                                    segment.lonFrom = lon;
                                }
                                break;
                            case "CLOCKWISE":
                            case "ANTI-CLOCKWISE":
                                haveDetails = true;
                                if (segment == null) {
                                    throw new Exception("No prev. POINT definition for clockwise or anti-clockwise arc");
                                }
                                if (t == "CLOCKWISE") {
                                    segment.type = AirspaceSegmentTypes.ArcClock;
                                }
                                else {
                                    segment.type = AirspaceSegmentTypes.ArcCounterClock;
                                }
                                // center, radius, to
                                tt = coll["centre"].Value.Split(splitt);
                                segment.latCenter = WgsPoint.FromCompactString(tt[0], true);
                                segment.lonCenter = WgsPoint.FromCompactString(tt[1], false);
                                segment.radius = float.Parse(coll["radius"].Value, AppContents.ni);
                                tt = coll["to"].Value.Split(splitt);
                                lat = WgsPoint.FromCompactString(tt[0], true);
                                lon = WgsPoint.FromCompactString(tt[1], false);
                                segment.latTo = lat;
                                segment.lonTo = lon;

                                space.points.Add(segment);
                                segment = new AirspaceSegment();
                                segment.type = AirspaceSegmentTypes.Unknown;
                                // from = to
                                segment.latFrom = lat;
                                segment.lonFrom = lon;
                                break;
                            case "CIRCLE":
                                haveDetails = true;
                                if (space.points.Count > 0) {
                                    throw new Exception("Found other segments with circle entry");
                                }
                                else {
                                    segment = new AirspaceSegment();
                                    segment.type = AirspaceSegmentTypes.Circle;
                                }
                                // center, radius
                                tt = coll["centre"].Value.Split(splitt);
                                segment.latCenter = WgsPoint.FromCompactString(tt[0], true);
                                segment.lonCenter = WgsPoint.FromCompactString(tt[1], false);
                                segment.radius = float.Parse(coll["radius"].Value, AppContents.ni);
                                space.points.Add(segment);
                                segment = new AirspaceSegment();
                                break;
                            case "TOPS":
                                space.upperBound = coll["second"].Value;
                                break;
                            case "BASE":
                                space.lowerBound = coll["second"].Value;
                                break;
                            case "RADIO":
                                airspaceRadio = coll["second"].Value;
                                if (airspaceRadio == null) {
                                    airspaceRadio = "";
                                }
                                if (space != null) {
                                    space.radioFrequency = airspaceRadio;
                                }
                                break;
                            case "ACTIVE":
                                airspaceActive = coll["second"].Value;
                                if (airspaceActive == null) {
                                    airspaceActive = "";
                                }
                                if (space != null) {
                                    space.remark = airspaceActive;
                                }
                                break;
                            }
                        }
                    }
                }

                if (space.name.Length > 0 && haveDetails) {
                    CheckImportOptions(space, useMichaelMeier, defaultClass, overwriteClass, defaultType, overwriteType);
                    suas.Add(space);
                }
            }
            catch (Exception e) {
                MessageBox.Show("An exception occured while reading file (Tim Newport Peace)\n\nThe fault line is:\n\n" +
                    m.ToString() + "\n\n" + e.StackTrace + "\n\n" + e.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Stop);
                suas.Clear();
            }

            return suas;
        }
    }
}
